package validators

import (
	"testing"

	. "github.com/smartystreets/goconvey/convey"
)

func TestDescriptionValidation(t *testing.T) {
	Convey("Given a description", t, func() {
		Convey("thats super long (> 300) characters", func() {
			description := "I AM A GARBAGE LONG DESCRIPTION OKAY THANKS YES TOO LONG TOO BELONG ON TWITCH.COM. Thats not a real website. Sure. Prove it. I can't, I'm just a string of text. So am I. We are the same string. Great. Nice to meet us. Thanks. Almost there. Just want to be above 300 characters or so by a good measure. Okay I think we did it and can probably stop now. Great thanks, we'll stop now."
			err := IsDescriptionValid(&description)
			Convey("We should return an error indicating description is too long", func() {
				So(err, ShouldEqual, errDescriptionTooLong)
			})
		})
		Convey("that contains some html garbage and newlines", func() {
			description := "I am really a strange<FSDFsd< description \n And there should be no newline here"
			Convey("the html characters should be escaped, newlines removed, and no error returned", func() {
				err := IsDescriptionValid(&description)
				So(err, ShouldBeNil)
				escapedDescription := "I am really a strange&lt;FSDFsd&lt; description  And there should be no newline here"
				So(description, ShouldEqual, escapedDescription)
			})
		})
		Convey("thats p chill and playing by the rules", func() {
			description := "I am an accurate description of a user with no fussy html characters or strange attempts to add newlines"
			err := IsDescriptionValid(&description)
			So(err, ShouldBeNil)
		})
		Convey("A javascript should be escaped", func() {
			description := "<script> I'm a bad javascript\n </script>"
			err := IsDescriptionValid(&description)
			So(err, ShouldBeNil)
			escapedDescription := "&lt;script&gt; I'm a bad javascript &lt;/script&gt;"
			So(description, ShouldEqual, escapedDescription)
		})
		Convey("A description contains special characters but should not be escaped", func() {
			description := "I'm a good description&bio. 还有中文哦！"
			err := IsDescriptionValid(&description)
			So(err, ShouldBeNil)
			escapedDescription := "I'm a good description&bio. 还有中文哦！"
			So(description, ShouldEqual, escapedDescription)
		})
	})
}
