package validators

import (
	"regexp"
	"strings"

	"code.justin.tv/web/users-service/models"
	"code.justin.tv/web/users-service/models/apperror"
)

const (
	constDisplaynameMinLen           = 3
	constDisplaynameMaxLen           = 25
	constDisplaynameNonAsciiCharSize = 2
	constDisplaynameAsciiCharSize    = 1
)

var (
	// Valid international character sets in displayname are - Chinese, Japanese and Korean.
	regexValidInternationalCharsets = regexp.MustCompile(`\p{Han}|\p{Katakana}|\p{Hiragana}|\p{Hangul}`)
	regexChineseCharsets            = regexp.MustCompile(`\p{Han}`)
	regexJapaneseCharsets           = regexp.MustCompile(`\p{Katakana}|\p{Hiragana}`)
	regexKoreanCharsets             = regexp.MustCompile(`\p{Hangul}`)

	// Only numbers and underscore are allowed in international displayname apart from international charsets.
	regexValidAsciiCharsets = regexp.MustCompile(`[0-9_]`)

	//  Only users with language set to these locales can get International displaynames.
	validLocales          = []string{"zh-cn", "zh-tw", "ja", "ko"}
	blackListedCharacters = []string{"卍", "卐"}
	whiteListedCharacters = []string{"ー"}

	ErrUnexpectedDisplaynameOverrideErr = apperror.New("Unexpected error during displayname override", true, "displayname_unexpcted_err")
	ErrDisplaynameTooShort              = apperror.New("The name you selected is too short.", true, "displayname_too_short")
	ErrDisplaynameTooLong               = apperror.New("The name you selected is too long.", true, "displayname_too_long")
	ErrDisplaynameInvalidCharacters     = apperror.New("The name you selected contains invalid characters.", true, "invalid_chars_in_displayname")
	ErrDisplaynameChangeAgain           = apperror.New("You may not change your display name again.", true, "displayname_change_again")
	ErrDisplaynameOnlyCapitalization    = apperror.New("You may not change your display name, only the capitalization of it.", true, "displayname_only_cap")
)

func hasInternationalCharacter(dn *string) bool {
	if dn == nil {
		return false
	}

	for _, char := range strings.Split(*dn, "") {
		if regexValidInternationalCharsets.MatchString(char) {
			return true
		}
	}
	return false
}

func hasBlackListedCharacter(dn *string) bool {
	for _, char := range strings.Split(*dn, "") {
		if containsChar(char, blackListedCharacters) {
			return true
		}
	}

	return false
}

func hasInvalidCharactersInCJK(dn *string) bool {
	for _, char := range strings.Split(*dn, "") {
		if (!regexValidInternationalCharsets.MatchString(char) && !regexValidAsciiCharsets.MatchString(char)) || containsChar(char, blackListedCharacters) {
			if !containsChar(char, whiteListedCharacters) {
				return true
			}
		}
	}

	return false
}

func userHasValidLocale(user *models.Properties) bool {

	if user.Language == nil {
		return false
	}

	for _, locale := range validLocales {
		if *user.Language == locale {
			return true
		}
	}

	return false
}

func checkLengthValidityForCJK(dn *string) error {
	length := 0
	for _, char := range strings.Split(*dn, "") {
		if regexValidInternationalCharsets.MatchString(char) {
			length += constDisplaynameNonAsciiCharSize
		} else if regexValidAsciiCharsets.MatchString(char) {
			length += constDisplaynameAsciiCharSize
		}
	}
	if length < constDisplaynameMinLen {
		return ErrDisplaynameTooShort
	}
	if length > constDisplaynameMaxLen {
		return ErrDisplaynameTooLong
	}
	return nil
}

func IsDisplaynameCJK(dn *string) bool {
	return hasInternationalCharacter(dn)
}

func GetDisplaynameLang(dn *string) string {
	if dn == nil {
		return "en"
	}

	for _, char := range strings.Split(*dn, "") {
		if regexKoreanCharsets.MatchString(char) {
			return "ko"
		}
	}

	for _, char := range strings.Split(*dn, "") {
		if regexJapaneseCharsets.MatchString(char) {
			return "ja"
		}
	}

	for _, char := range strings.Split(*dn, "") {
		if regexChineseCharsets.MatchString(char) {
			return "zh"
		}
	}

	return "en"
}

func IsDisplaynameValid(dn *string, user *models.Properties) error {

	// Check if displayname is nil, which we will allow since it will not be written as such
	if dn == nil {
		return nil
	}

	// Check if new displayname is a simple capitalization change.
	if strings.EqualFold(*dn, *user.Displayname) {
		return nil
	}

	// Check if displayname already set with international characters.
	if *dn != "" && hasInternationalCharacter(user.Displayname) {
		return ErrDisplaynameChangeAgain
	}

	// Check if user's locale is CJK, else return false
	if !userHasValidLocale(user) {
		return ErrDisplaynameOnlyCapitalization
	}

	// Check if displayname has at least one international character.
	if !hasInternationalCharacter(dn) {
		return ErrDisplaynameOnlyCapitalization
	}

	// Check if invalid characters are present
	if hasInvalidCharactersInCJK(dn) {
		return ErrDisplaynameInvalidCharacters
	}

	if err := checkLengthValidityForCJK(dn); err != nil {
		return err
	}

	return nil
}

func IsDisplaynameValidForOverride(user *models.Properties, new *string) error {
	if user == nil || user.Login == nil || user.Displayname == nil {
		return ErrUnexpectedDisplaynameOverrideErr
	}
	// Check if displayname is nil, which we will allow since it will not be written as such
	if new == nil {
		return nil
	}

	// Check if invalid characters are present if the new displayname is CJK.
	if hasInternationalCharacter(new) {
		if hasInvalidCharactersInCJK(new) {
			return ErrDisplaynameInvalidCharacters
		}
		if err := checkLengthValidityForCJK(new); err != nil {
			return err
		}
	} else {
		// Check if new displayname is a simple capitalization change.
		if strings.EqualFold(*user.Displayname, *new) {
			return nil
		}

		// If it is a displayname revert, the displayname should always be same as the login.
		if strings.EqualFold(*user.Login, *new) {
			return nil
		}

		return ErrDisplaynameOnlyCapitalization
	}

	return nil
}

func containsChar(char string, characters []string) bool {
	for _, c := range characters {
		if c == char {
			return true
		}
	}
	return false
}
