package validators

import (
	"testing"

	"code.justin.tv/web/users-service/models"
	. "github.com/smartystreets/goconvey/convey"
)

func TestDisplaynameValidation(t *testing.T) {
	Convey("Given a user named cado", t, func() {
		login := "cado"
		olddn := "cado"
		Convey("And user is CJK language", func() {
			lang := "ja"
			user := models.Properties{Login: &login, Displayname: &olddn, Language: &lang}
			Convey("and new displayname only changes capitalization of login", func() {
				dn := "caDO"
				err := IsDisplaynameValid(&dn, &user)
				Convey("we should return valid", func() {
					So(err, ShouldBeNil)
				})
			})
			Convey("and new displayname is not just a capitalization change - but does not contain international characters", func() {
				dn := "smallgarbagecan"
				err := IsDisplaynameValid(&dn, &user)
				Convey("we should return ", func() {
					So(err, ShouldEqual, ErrDisplaynameOnlyCapitalization)
				})
			})
			Convey("and new displayname contains invalid characters", func() {
				dn := "漢漢h"
				err := IsDisplaynameValid(&dn, &user)
				Convey("we should return ", func() {
					So(err, ShouldEqual, ErrDisplaynameInvalidCharacters)
				})
			})
			Convey("and user already set with international characters", func() {
				dn := "漢漢h"
				olddn = "漢字"
				err := IsDisplaynameValid(&dn, &user)
				Convey("we should return ", func() {
					So(err, ShouldEqual, ErrDisplaynameChangeAgain)
				})
			})
			Convey("and new displayname is too short - < 3", func() {
				dn := "漢"
				err := IsDisplaynameValid(&dn, &user)
				Convey("we should return ", func() {
					So(err, ShouldEqual, ErrDisplaynameTooShort)
				})
			})
			Convey("and new displayname is too long - > 25", func() {
				dn := "漢漢漢漢漢漢漢漢漢漢漢漢漢"
				err := IsDisplaynameValid(&dn, &user)
				Convey("we should return ", func() {
					So(err, ShouldEqual, ErrDisplaynameTooLong)
				})
			})
			Convey("and new displayname contains Korean characters", func() {
				dn := "ᄑ9_ᄁ78"
				err := IsDisplaynameValid(&dn, &user)
				Convey("we should return ", func() {
					So(err, ShouldBeNil)
				})
			})
			Convey("and new displayname contains Japanese characters", func() {
				dn := "る6ロ1字2_"
				err := IsDisplaynameValid(&dn, &user)
				Convey("we should return ", func() {
					So(err, ShouldBeNil)
				})
			})
			Convey("and new displayname contains Chinese characters", func() {
				dn := "_34漢字05"
				err := IsDisplaynameValid(&dn, &user)
				Convey("we should return ", func() {
					So(err, ShouldBeNil)
				})
			})
			Convey("and new displayname contains CJK characters and current displayname has a space", func() {
				dn := "_34漢字05"
				olddn = " cado"
				err := IsDisplaynameValid(&dn, &user)
				Convey("we should return ", func() {
					So(err, ShouldBeNil)
				})
			})
			Convey("and the character is blacklisted", func() {
				dn := "invalid卍卐"
				err := IsDisplaynameValid(&dn, &user)
				Convey("we should return invalid", func() {
					So(err, ShouldNotBeNil)
				})
			})
			Convey("and the character is Katakana dash", func() {
				dn := "哈哈哈哈哈哈哈ー"
				err := IsDisplaynameValid(&dn, &user)
				Convey("we should return valid", func() {
					So(err, ShouldBeNil)
				})
			})
			Convey("and new displayname is Korean", func() {
				dn := "향찰"
				lang := GetDisplaynameLang(&dn)
				Convey("we should return ko", func() {
					So(lang, ShouldEqual, "ko")
				})
			})
			Convey("and new displayname is Japanese", func() {
				dn := "のコン"
				lang := GetDisplaynameLang(&dn)
				Convey("we should return ja", func() {
					So(lang, ShouldEqual, "ja")
				})
			})
			Convey("and new displayname is Chinese", func() {
				dn := "汉字"
				lang := GetDisplaynameLang(&dn)
				Convey("we should return zh", func() {
					So(lang, ShouldEqual, "zh")
				})
			})
		})
		Convey("and user is not CJK language and has English displayname", func() {
			lang := "en"
			user := models.Properties{Login: &login, Displayname: &olddn, Language: &lang}
			Convey("and new displayname contains CJK characters", func() {
				dn := "ᄑ9_ᄁ78"
				err := IsDisplaynameValid(&dn, &user)
				Convey("we should return ", func() {
					So(err, ShouldEqual, ErrDisplaynameOnlyCapitalization)
				})
			})
			Convey("and new displayname is a capitalization change with override", func() {
				dn := "Cado"
				err := IsDisplaynameValidForOverride(&user, &dn)
				Convey("we should return ", func() {
					So(err, ShouldEqual, nil)
				})
			})
			Convey("and new displayname is not just a capitalization change with override", func() {
				dn := "CadoNew"
				err := IsDisplaynameValidForOverride(&user, &dn)
				Convey("we should return ", func() {
					So(err, ShouldEqual, ErrDisplaynameOnlyCapitalization)
				})
			})
			Convey("and new displayname contains valid CJK with override", func() {
				dn := "哈哈123"
				err := IsDisplaynameValidForOverride(&user, &dn)
				Convey("we should return ", func() {
					So(err, ShouldEqual, nil)
				})
			})
			Convey("and new displayname contains invalid CJK with override", func() {
				dn := "哈哈test"
				err := IsDisplaynameValidForOverride(&user, &dn)
				Convey("we should return ", func() {
					So(err, ShouldEqual, ErrDisplaynameInvalidCharacters)
				})
			})
		})
		Convey("and user has CJK user name and with override", func() {
			cjkdn := "哈哈"
			user := models.Properties{Login: &login, Displayname: &cjkdn}
			Convey("and new displayname contains CJK characters", func() {
				dn := "哈哈123"
				err := IsDisplaynameValidForOverride(&user, &dn)
				Convey("we should return ", func() {
					So(err, ShouldEqual, nil)
				})
			})
			Convey("and new displayname contains CJK characters and try to change back to displaynamne same as login", func() {
				dn := login
				err := IsDisplaynameValidForOverride(&user, &dn)
				Convey("we should return ", func() {
					So(err, ShouldEqual, nil)
				})
			})
			Convey("and new displayname contains CJK characters and try to change back to displaynamne different as login", func() {
				dn := "different"
				err := IsDisplaynameValidForOverride(&user, &dn)
				Convey("we should return ", func() {
					So(err, ShouldEqual, ErrDisplaynameOnlyCapitalization)
				})
			})
		})
	})
}
