package validators

import (
	"net/mail"
	"regexp"

	"code.justin.tv/web/users-service/models"
	"code.justin.tv/web/users-service/models/apperror"
)

const (
	MaxEmailLength = 254
)

var (
	errInvalidEmail      = apperror.New("Email address is not valid", true, "invalid_email")
	errWorkEmailRequired = apperror.New("Admins must use their work email", true, "invalid_email")
	regexSafeEmailDomain = regexp.MustCompile(`(^.*@|\.)((?:justin|twitch)\.tv|www-cdn\.jtvnw\.net|s\.jtvnw\.net)$`)
	whitelistedEmails    = []string{"qa.automation@gmail.com", "qa.twitch.auto@gmail.com"}
)

func IsEmailValid(email *string, user *models.Properties) error {
	if email == nil {
		return nil
	}

	if len(*email) > MaxEmailLength {
		return errInvalidEmail
	}

	e, err := mail.ParseAddress(*email)
	if err != nil {
		return errInvalidEmail
	}

	// If this isn't just an email address (e.g. "Bob Johnson <bjo@gmail.com>") fail validation
	if *email != e.Address {
		return errInvalidEmail
	}

	// If user is admin
	if user.Admin != nil && *user.Admin {
		// They need an email address with a safe domain
		if !regexSafeEmailDomain.MatchString(*email) {
			return errWorkEmailRequired
		}
	}

	return nil
}

func IsWhitelistedEmail(email *string) bool {
	if email == nil {
		return false
	}

	b := regexSafeEmailDomain.MatchString(*email)
	if b {
		return true
	}

	for _, whiltelistedEmail := range whitelistedEmails {
		if *email == whiltelistedEmail {
			return true
		}
	}

	return false
}
