package validators

import (
	"testing"

	"code.justin.tv/web/users-service/models"

	. "github.com/smartystreets/goconvey/convey"
)

func TestEmailValidation(t *testing.T) {
	Convey("When validating an email address", t, func() {

		Convey("Given a non admin user", func() {
			IsAdmin := false
			user := models.Properties{Admin: &IsAdmin}
			Convey("and a valid email", func() {
				email := "cado@couldbereal.tv"
				err := IsEmailValid(&email, &user)
				Convey("err should be nil", func() {
					So(err, ShouldBeNil)
				})
			})
			Convey("and an invalid email", func() {
				email := "garbageisanemailmaybe"
				err := IsEmailValid(&email, &user)
				Convey("we should return an error", func() {
					So(err, ShouldEqual, errInvalidEmail)
				})
			})
			Convey("Given a long email", func() {
				email := "IAmAverylongEmailifyouknowwhatImean@wellIneedalotmorecharacterssotimetocopypastewellIneedalotmorecharacterssotimetocopypastewellIneedalotmorecharacterssotimetocopypastewellIneedalotmorecharacterssotimetocopypastewellIneedalotmorecharacterssotimetocopypa.com"
				err := IsEmailValid(&email, &user)
				Convey("We should return an error", func() {
					So(err, ShouldEqual, errInvalidEmail)
				})
			})
		})
		Convey("Given an admin user", func() {
			IsAdmin := true
			user := models.Properties{Admin: &IsAdmin}
			Convey("and a valid email that isn't whitelisted", func() {
				email := "cado@nottwitch.tv"
				err := IsEmailValid(&email, &user)
				Convey("err should indicate a work email is required", func() {
					So(err, ShouldEqual, errWorkEmailRequired)
				})
			})
			Convey("and a valid email that is whitelisted", func() {
				email := "cado@twitch.tv"
				err := IsEmailValid(&email, &user)
				Convey("err should be nil", func() {
					So(err, ShouldBeNil)
				})
			})
			Convey("and an invalid email", func() {
				email := "garbageisanemailmaybe"
				err := IsEmailValid(&email, &user)
				Convey("we should return an error", func() {
					So(err, ShouldEqual, errInvalidEmail)
				})
			})
		})
	})
}
