package validators

import (
	"regexp"
	"strings"

	"code.justin.tv/web/users-service/models/apperror"
)

const (
	constLoginMinLenOverride = 3
	constLoginMinLen         = 4
	constLoginMaxLen         = 25
)

var (
	// OnCreateAbuseRegex checks for banned words
	OnCreateAbuseRegex = regexp.MustCompile(`retard|hijoeputa|pajera|putito|wetback|serefsiz|culeado|estupido|cojudo|jackass|idiotas|conchetu|prostituta|chuchon|pastrula|pastrulas|encule|pastrulo|pastrulos|cagado|gilipoya|culona|cerote|babosa|gilipolla|chinguen|bitches|sudaca|indios|malparida|mames|dickhead|gilipoyas|zorras|bicha|cagar|marika|mamon|culiado|pendeja|vadia|culeros|gonorrea|mamar|maricas|estupida|huevon|atdhe|pajero|sikeyim|skank|cagada|amcik|perro|reputo|reputa|merde|siktir|carajo|cazzo|trola|baboso|connard|pinga|gilipollas|imbecil|salope|polla|yarrak|cagones|indio|culiao|cabro|wanker|niggers|pelotudo|sikerim|indigena|boludo|chucha|zorra|pinche|hijueputa|orospu|putas|malparido|porn|culero|putos|concha|buceta|bastard|chinga|viado|marica|boob|merda|caralho|suck|maricon|vagina|nigga|penis|pendejo|fuck|verga|asshole|mierda|pussy|nigger|whore|slut|fag|sex|cunt|shit|bitch|pinto|rape|twat|twitchsupport|twitchtvsupport|twitch_support|twitchtv_support|twitchstaff|twitchadmin|twitch_rep|twitch_tech|twitch_report|twitch_mod|twitch_help|twitch_administrator`)

	// OnUpdateAbuseRegex checks for variations on banned words
	OnUpdateAbuseRegex = regexp.MustCompile(`pu+ss+y|ni+gg+a|\Wass\W|cu+nt|fa+g|fu+ck|ni+gg+e+r|\Wpo+rn\W|bi+tch|pe+ni+s|\Wtits\W|vagina|who+re|slu+t|\Wsex\W|\Wporno\W|\Wanal\W|moderator|^justinfan\d+$`)

	NameRegex = regexp.MustCompile(`\A[a-z0-9][a-z0-9_]*\z`)

	ErrLoginTooShort          = apperror.New("The login you selected is too short.", true, "login_too_short")
	ErrLoginTooLong           = apperror.New("The login you selected is too long.", true, "login_too_long")
	ErrLoginInvalidCharacters = apperror.New("The login you selected contains invalid characters.", true, "invalid_chars_in_login")
	// ErrLoginUsingVariationOnBannedWords is for users trying to get around the banned words
	ErrLoginUsingVariationOnBannedWords = apperror.New("Login contains variations on banned words.", true, "login_use_variation_on_banned_words")
	// ErrLoginUsingBannedWords is for users who used banned words
	ErrLoginUsingBannedWords = apperror.New("Login contains banned words.", true, "login_use_banned_words")
)

func hasNonValidCharacters(l string) bool {
	return !NameRegex.MatchString(l)
}

func checkLength(l string, overrideLength bool) error {
	length := len(l)
	var validMinLength int
	if overrideLength {
		validMinLength = constLoginMinLenOverride
	} else {
		validMinLength = constLoginMinLen
	}
	if length < validMinLength {
		return ErrLoginTooShort
	}
	if length > constLoginMaxLen {
		return ErrLoginTooLong
	}
	return nil
}

func hasBannedWords(l string) error {
	if OnCreateAbuseRegex.MatchString(l) {
		return ErrLoginUsingBannedWords
	}
	if OnUpdateAbuseRegex.MatchString(l) {
		return ErrLoginUsingVariationOnBannedWords
	}
	return nil
}

// IsLoginValid is a port of the valid_login? function in web/web in Moderation.rb
func IsLoginValid(l *string, overrideLength bool) error {

	// Check if login is nil, which we will allow since it will not be written as such
	if l == nil {
		return nil
	}

	login := strings.ToLower(*l)

	if err := checkLength(login, overrideLength); err != nil {
		return err
	}

	// Check if invalid characters are present
	if hasNonValidCharacters(login) {
		return ErrLoginInvalidCharacters
	}

	if err := hasBannedWords(login); err != nil {
		return err
	}

	return nil
}
