package validators

import (
	"testing"

	. "github.com/smartystreets/goconvey/convey"
)

func TestLoginValidation(t *testing.T) {
	Convey("Testing login changes", t, func() {
		Convey("trying to change login to something invalid", func() {
			Convey("to a login that is too short < 4 characters", func() {
				login := "hi"
				err := IsLoginValid(&login, false)
				Convey("we should return", func() {
					So(err, ShouldEqual, ErrLoginTooShort)
				})
			})
			Convey("to a login that is too long > 25 characters", func() {
				login := "ICanWriteAReallyLongSentenceMaybeWithTheRequiredCharacters"
				err := IsLoginValid(&login, false)
				Convey("we should return", func() {
					So(err, ShouldEqual, ErrLoginTooLong)
				})
			})
			Convey("to a login that has invalid characters", func() {
				login := "hey!!!!!!!!!!!漢!!!"
				err := IsLoginValid(&login, false)
				Convey("we should return", func() {
					So(err, ShouldEqual, ErrLoginInvalidCharacters)
				})
			})
			Convey("to a login that has a -", func() {
				login := "hey-hi"
				err := IsLoginValid(&login, false)
				Convey("we should return", func() {
					So(err, ShouldEqual, ErrLoginInvalidCharacters)
				})
			})
			Convey("to a login with banned words", func() {
				login := "hey_estupido"
				err := IsLoginValid(&login, false)
				Convey("we should return", func() {
					So(err, ShouldEqual, ErrLoginUsingBannedWords)
				})
			})
			Convey("to a login with variations on banned words", func() {
				login := "hey_sluuuuuuuuuuuuuuut"
				err := IsLoginValid(&login, false)
				Convey("we should return", func() {
					So(err, ShouldEqual, ErrLoginUsingVariationOnBannedWords)
				})
			})
			Convey("to a login with justinfan as prefix", func() {
				login := "justinfan123"
				err := IsLoginValid(&login, false)
				Convey("we should return", func() {
					So(err, ShouldEqual, ErrLoginUsingVariationOnBannedWords)
				})
			})
			Convey("to a login with justinfan inside", func() {
				login := "123justinfan123"
				err := IsLoginValid(&login, false)
				Convey("we should return", func() {
					So(err, ShouldBeNil)
				})
			})
		})

		Convey("trying to change to a valid login", func() {
			Convey("to a login that is > 4 chars", func() {
				login := "egflores"
				err := IsLoginValid(&login, false)
				Convey("we should return valid", func() {
					So(err, ShouldBeNil)
				})
			})
			Convey("to a login that is 3 chars", func() {
				login := "abc"
				err := IsLoginValid(&login, true)
				Convey("we should return valid", func() {
					So(err, ShouldBeNil)
				})
			})
		})
	})
}
