package validators

import (
	"testing"
	"time"

	"code.justin.tv/web/users-service/models"
)

type validPropertiesScenario struct {
	User   models.Properties
	Params *models.FilterParams

	Valid bool
}

func (s validPropertiesScenario) Test(t *testing.T) {
	actual := ValidateProperties(s.User, s.Params)

	if actual != s.Valid {
		t.Errorf("expected valid: %t, got valid: %t", s.Valid, actual)
	}
}

func TestValidUser(t *testing.T) {
	now := time.Now()
	tru := true

	for name, scenario := range map[string]validPropertiesScenario{
		"nil params": {
			Valid: true,
		},
		"filtering nothing": {
			Params: &models.FilterParams{},
			Valid:  true,
		},
		"user not deleted and filtering not deleted": {
			Params: &models.FilterParams{
				NotDeleted: true,
			},
			Valid: true,
		},
		"user deleted and filtering not deleted": {
			User: models.Properties{
				DeletedOn: &now,
			},
			Params: &models.FilterParams{
				NotDeleted: true,
			},
		},
		"user no tos violation and filtering no tos violation": {
			Params: &models.FilterParams{
				NoTOSViolation: true,
			},
			Valid: true,
		},
		"user tos violation and filtering no tos violation": {
			User: models.Properties{
				TermsOfServiceViolation: &tru,
			},
			Params: &models.FilterParams{
				NoTOSViolation: true,
			},
		},
		"user no dmca violation and filtering no dmca violation": {
			Params: &models.FilterParams{
				NoDMCAViolation: true,
			},
			Valid: true,
		},
		"user dmca violation and filtering no dmca violation": {
			User: models.Properties{
				DmcaViolation: &tru,
			},
			Params: &models.FilterParams{
				NoDMCAViolation: true,
			},
		},
	} {
		t.Run(name, scenario.Test)
	}
}
