package main

import (
	"context"
	"fmt"
	"os"

	"code.justin.tv/feeds/errors"
	"github.com/urfave/cli"
)

const (
	meepoAddressProduction     = "http://meepo-production.internal.justin.tv"
	usersServiceAddressStaging = "http://users-service.dev.us-west2.twitch.tv"
)

func main() {
	err := runMain()
	if err != nil {
		fmt.Println(fmt.Sprintf("%+v", err))
	}
}

func runMain() error {
	ctx := context.Background()
	utility, err := newUtility(&utilityConfig{
		meepoAddress:        meepoAddressProduction,
		usersServiceAddress: usersServiceAddressStaging,
	})
	if err != nil {
		return err
	}

	app := cli.NewApp()
	app.Name = "meepo-util"
	app.Usage = "manipulate squad streams"
	app.Version = "0.5.0"
	app.Commands = []cli.Command{
		{
			Name:      "get-squad",
			Usage:     "gets a squad stream by ID or by a member's login",
			UsageText: "meepo-util get-squad --squad-id SQUAD_ID | meepo-util get-squad --login MEMBER_LOGIN",
			Flags: []cli.Flag{
				&cli.StringFlag{
					Name: "squad-id",
				},
				&cli.StringFlag{
					Name: "login",
				},
			},
			Action: func(c *cli.Context) error {
				squadID := c.String("squad-id")
				if squadID != "" {
					squad, err := utility.getSquadBySquadID(ctx, squadID)
					if err != nil {
						return err
					}

					return utility.printSquad(ctx, squad)
				}

				login := c.String("login")
				if login != "" {
					squad, err := utility.getSquadByMemberLogin(ctx, login)
					if err != nil {
						return err
					}

					return utility.printSquad(ctx, squad)
				}

				return errors.New("must specify a squad ID or member login")
			},
		},
		{
			Name:      "create-squad",
			Usage:     "creates a new squad stream",
			UsageText: "meepo-util create-squad [--pending] USER_LOGIN_1 USER_LOGIN_2 USER_LOGIN_3 USER_LOGIN_4",
			Action: func(c *cli.Context) error {
				squad, err := utility.createSquad(ctx, c.Args(), c.Bool("pending"))
				if err != nil {
					return err
				}

				return utility.printSquad(ctx, squad)
			},
			Flags: []cli.Flag{
				&cli.BoolFlag{
					Name: "pending",
				},
			},
		},
		{
			Name:      "start-squad",
			Usage:     "starts a pending squad",
			UsageText: "meepo-util start-squad SQUAD_ID",
			Action: func(c *cli.Context) error {
				squadID := c.Args().Get(0)
				if squadID == "" {
					return errors.New("squad-id is required")
				}

				squad, err := utility.startSquad(ctx, squadID)
				if err != nil {
					return err
				}

				return utility.printSquad(ctx, squad)
			},
		},
		{
			Name:      "end-squad",
			Usage:     "ends a squad stream",
			UsageText: "meepo-util end-squad SQUAD_ID",
			Action: func(c *cli.Context) error {
				squadID := c.Args().Get(0)
				if squadID == "" {
					return errors.New("must specify a squad ID")
				}

				squad, err := utility.endSquad(ctx, squadID)
				if err != nil {
					return err
				}

				return utility.printSquad(ctx, squad)
			},
		},
		{
			Name:      "add-member",
			Usage:     "adds a member to a squad stream",
			UsageText: "meepo-util add-member SQUAD_ID USER_LOGIN",
			Action: func(c *cli.Context) error {
				squadID := c.Args().Get(0)
				if squadID == "" {
					return errors.New("squad-id is required")
				}

				login := c.Args().Get(1)
				if login == "" {
					return errors.New("login is required")
				}

				squad, err := utility.addMemberToSquad(ctx, login, squadID)
				if err != nil {
					return err
				}

				return utility.printSquad(ctx, squad)
			},
		},
		{
			Name:      "remove-member",
			Usage:     "removes a member from a squad stream",
			UsageText: "meepo-util remove-member USER_LOGIN",
			Action: func(c *cli.Context) error {
				login := c.Args().Get(0)
				if login == "" {
					return errors.New("login is required")
				}

				squad, err := utility.removeMember(ctx, login)
				if err != nil {
					return err
				}

				return utility.printSquad(ctx, squad)
			},
		},
		{
			Name:      "invite-member",
			Usage:     "invite a member to a squad stream",
			UsageText: "meepo-util invite-member SQUAD_ID RECIPIENT_LOGIN_OR_ID",
			Action: func(c *cli.Context) error {
				squadID := c.Args().Get(0)
				recipientLoginOrID := c.Args().Get(1)
				if squadID == "" {
					return errors.New("squad ID is required")
				}
				if recipientLoginOrID == "" {
					return errors.New("recipient login or ID is required")
				}

				squad, err := utility.inviteMember(ctx, squadID, recipientLoginOrID)
				if err != nil {
					return err
				}

				return utility.printSquad(ctx, squad)
			},
		},
		{
			Name:      "reject-invitation",
			Usage:     "reject a pending invitation to a squad stream",
			UsageText: "meepo-util reject-invitation SQUAD_ID INVITATION_ID",
			Action: func(c *cli.Context) error {
				squadID := c.Args().Get(0)
				invitationID := c.Args().Get(1)
				if squadID == "" {
					return errors.New("squad ID is required")
				}
				if invitationID == "" {
					return errors.New("invitation ID is required")
				}

				squad, err := utility.rejectInvitation(ctx, squadID, invitationID)
				if err != nil {
					return err
				}

				return utility.printSquad(ctx, squad)
			},
		},
		{
			Name:      "delete-invitation",
			Usage:     "delete a rejected invitation to a squad stream",
			UsageText: "meepo-util delete-invitation SQUAD_ID INVITATION_ID",
			Action: func(c *cli.Context) error {
				squadID := c.Args().Get(0)
				invitationID := c.Args().Get(1)
				if squadID == "" {
					return errors.New("squad ID is required")
				}
				if invitationID == "" {
					return errors.New("invitation ID is required")
				}

				squad, err := utility.deleteInvitation(ctx, squadID, invitationID)
				if err != nil {
					return err
				}

				return utility.printSquad(ctx, squad)
			},
		},
	}

	return app.Run(os.Args)
}
