// +build integration

package main

import (
	"context"
	"testing"

	. "github.com/smartystreets/goconvey/convey"
)

const meepoAddressStaging = "http://meepo-staging.internal.justin.tv"

const (
	// Arbitrary, but real users
	userLogin1 = "turtleeatspigeon1"
	userLogin2 = "turtleeatspigeon2"
	userLogin3 = "turtleeatspigeon3"
)

var userLogins = []string{
	userLogin1,
	userLogin2,
	userLogin3,
}

func Setup(meepoAddress string, usersServiceAddress string, f func(context context.Context, utilInstance *utility)) func() {
	return func() {
		// Set up
		ctx := context.Background()
		util, err := newUtility(&utilityConfig{
			meepoAddress:        meepoAddress,
			usersServiceAddress: usersServiceAddress,
		})
		So(err, ShouldBeNil)

		// Execute test
		f(ctx, util)

		// Tear down
		Reset(func() {
			// End all squads that were created by the test.
			for _, userLogin := range userLogins {
				sq, err := util.getSquadByMemberLogin(ctx, userLogin)
				So(err, ShouldBeNil)
				if sq == nil {
					continue
				}

				So(sq.ID, ShouldNotBeBlank)
				_, err = util.endSquad(ctx, sq.ID)
				So(err, ShouldBeNil)
			}
		})
	}
}

func WithSquad(ctx context.Context, util *utility, isPending bool, userLogins []string, f func(squad *squad)) func() {
	return func() {
		// Before running tests, verify that the user logins are not in a squad.
		for _, userLogin := range userLogins {
			sq, err := util.getSquadByMemberLogin(ctx, userLogin)
			So(err, ShouldBeNil)
			So(sq, ShouldBeNil)
		}

		createdSquad, err := util.createSquad(ctx, userLogins, isPending)
		So(err, ShouldBeNil)
		So(createdSquad, ShouldNotBeNil)

		f(createdSquad)
	}
}

func TestUtility(t *testing.T) {
	Convey("Utility", t, Setup(meepoAddressStaging, usersServiceAddressStaging, func(ctx context.Context, util *utility) {
		Convey("createSquad", func() {
			createdSquad, err := util.createSquad(ctx, []string{userLogin1, userLogin2}, true)
			So(err, ShouldBeNil)
			So(createdSquad, ShouldNotBeNil)
			So(createdSquad.ID, ShouldNotBeEmpty)
			So(createdSquad.Status, ShouldEqual, "PENDING")
			So(createdSquad.MemberLogins, ShouldHaveLength, 2)
			So(createdSquad.MemberLogins, ShouldContain, userLogin1)
			So(createdSquad.MemberLogins, ShouldContain, userLogin2)
		})

		Convey("startSquad", WithSquad(ctx, util, true, []string{userLogin1, userLogin2}, func(createdSquad *squad) {
			startedSquad, err := util.startSquad(ctx, createdSquad.ID)
			So(err, ShouldBeNil)
			So(startedSquad, ShouldNotBeNil)
			So(startedSquad.Status, ShouldEqual, "LIVE")
		}))
	}))
}
