package main

import (
	"context"
	"log"
	"strconv"

	"code.justin.tv/feeds/errors"
	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/web/users-service/client/usersclient_internal"
	usersmodels "code.justin.tv/web/users-service/models"
)

type usersClient struct {
	users usersclient_internal.InternalClient
}

func newUsersClient(usersServiceEndpoint string) (*usersClient, error) {
	internalClient, err := usersclient_internal.NewClient(twitchclient.ClientConf{
		Host: usersServiceEndpoint,
	})
	if err != nil {
		return nil, err
	}

	return &usersClient{
		users: internalClient,
	}, nil
}

func (u *usersClient) convertLoginsToIDs(ctx context.Context, logins []string) ([]string, error) {
	if len(logins) == 0 {
		return []string{}, nil
	}

	results, err := u.users.GetUsers(ctx, &usersmodels.FilterParams{
		Logins: logins,
	}, nil)
	if err != nil {
		return nil, err
	}

	loginsToIDs := make(map[string]string, len(logins))
	for _, userProperties := range results.Results {
		if userProperties.Login == nil {
			continue
		}

		loginsToIDs[*userProperties.Login] = userProperties.ID
	}

	userIDs := make([]string, 0, len(logins))
	for _, login := range logins {
		id := loginsToIDs[login]
		if id == "" {
			return nil, errors.Errorf("could not load ID for login, \"%s\"", login)
		}

		userIDs = append(userIDs, id)
	}

	return userIDs, nil
}

func (u *usersClient) convertLoginToID(ctx context.Context, login string) (string, error) {
	if login == "" {
		return "", nil
	}
	ids, err := u.convertLoginsToIDs(ctx, []string{login})
	if err != nil {
		return "", err
	}
	return ids[0], nil
}

func (u *usersClient) convertIDsToLogins(ctx context.Context, userIDs []string) ([]string, error) {
	if len(userIDs) == 0 {
		return []string{}, nil
	}
	results, err := u.users.GetUsers(ctx, &usersmodels.FilterParams{
		IDs: userIDs,
	}, nil)
	if err != nil {
		return nil, err
	}

	idsToLogins := make(map[string]string, len(userIDs))
	for _, userProperties := range results.Results {
		if userProperties.Login == nil {
			continue
		}

		idsToLogins[userProperties.ID] = *userProperties.Login
	}

	logins := make([]string, 0, len(userIDs))
	for _, userID := range userIDs {
		login := idsToLogins[userID]
		if login == "" {
			login = "id:" + userID
			log.Printf("could not load login for user ID, \"%s\"", userID)
		}

		logins = append(logins, login)
	}

	return logins, nil
}

func (u *usersClient) convertIDToLogin(ctx context.Context, userID string) (string, error) {
	if userID == "" {
		return "", nil
	}
	logins, err := u.convertIDsToLogins(ctx, []string{userID})
	if err != nil {
		return "", err
	}
	return logins[0], nil
}

func (u *usersClient) isID(ctx context.Context, loginOrUserID string) bool {
	_, err := strconv.ParseFloat(loginOrUserID, 64)
	return err == nil
}

func (u *usersClient) convertToID(ctx context.Context, loginOrUserID string) (string, error) {
	if loginOrUserID == "" {
		return "", nil
	}

	if u.isID(ctx, loginOrUserID) {
		return loginOrUserID, nil
	}

	// Not ID so get ID for login
	ids, err := u.convertToIDs(ctx, []string{loginOrUserID})
	if err != nil {
		return "", err
	}

	return ids[0], nil
}

func (u *usersClient) convertToIDs(ctx context.Context, loginsOrUserIDs []string) ([]string, error) {
	if len(loginsOrUserIDs) == 0 {
		return []string{}, nil
	}

	// Bucket input into logins slices
	logins := make([]string, 0, len(loginsOrUserIDs))
	for _, loginOrUserID := range loginsOrUserIDs {
		if loginOrUserID == "" {
			return nil, errors.Errorf("could not convert an empty string into user ID")
		}

		if !u.isID(ctx, loginOrUserID) {
			logins = append(logins, loginOrUserID)
		}
	}

	ids, err := u.convertLoginsToIDs(ctx, logins)
	if err != nil {
		return nil, err
	}

	loginsToIDs := make(map[string]string, len(ids))
	for i, id := range ids {
		loginsToIDs[logins[i]] = id
	}

	// Create returned result, maintaining order of function parameter
	results := make([]string, 0, len(loginsOrUserIDs))
	for _, loginOrUserID := range loginsOrUserIDs {
		if u.isID(ctx, loginOrUserID) {
			results = append(results, loginOrUserID)
		} else {
			results = append(results, loginsToIDs[loginOrUserID])
		}
	}

	return results, nil
}
