package main

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"strconv"

	"code.justin.tv/feeds/errors"
	"code.justin.tv/twitch-events/meepo/rpc/meepo"
)

type utilityConfig struct {
	usersServiceAddress string
	meepoAddress        string
}

func newUtility(config *utilityConfig) (*utility, error) {
	usersClient, err := newUsersClient(config.usersServiceAddress)
	if err != nil {
		return nil, err
	}

	return &utility{
		users: usersClient,
		meepo: meepo.NewMeepoProtobufClient(config.meepoAddress, &http.Client{}),
	}, nil
}

type utility struct {
	users *usersClient
	meepo meepo.Meepo
}

func (u *utility) getSquadByMemberLogin(ctx context.Context, login string) (*squad, error) {
	userID, err := u.users.convertLoginToID(ctx, login)
	if err != nil {
		return nil, err
	}

	return u.getSquadByMemberID(ctx, userID)
}

func (u *utility) processSquadResponse(ctx context.Context, response *meepo.Squad) (*squad, error) {
	if response == nil {
		return nil, nil
	}
	if response.Status == meepo.Squad_ENDED {
		return u.convertSquad(ctx, response, []*meepo.Invitation{}, []*meepo.Invitation{})
	}

	statuses := []meepo.Invitation_Status{meepo.Invitation_PENDING, meepo.Invitation_REJECTED}
	invitations, err := u.getInvitationsBySquadID(ctx, response.Id, response.OwnerId, statuses)
	if err != nil {
		return nil, err
	}
	return u.convertSquad(ctx, response, invitations[0], invitations[1])
}

func (u *utility) getSquadByMemberID(ctx context.Context, userID string) (*squad, error) {
	getSquadResponse, err := u.meepo.GetSquadByChannelID(ctx, &meepo.GetSquadByChannelIDRequest{
		ChannelId: userID,
	})
	if err != nil {
		return nil, err
	}

	return u.processSquadResponse(ctx, getSquadResponse.GetSquad())
}

func (u *utility) getSquadBySquadID(ctx context.Context, squadID string) (*squad, error) {
	getSquadResponse, err := u.meepo.GetSquadByID(ctx, &meepo.GetSquadByIDRequest{
		Id: squadID,
	})
	if err != nil {
		return nil, err
	}

	return u.processSquadResponse(ctx, getSquadResponse.GetSquad())
}

func (u *utility) createSquad(ctx context.Context, logins []string, leavePending bool) (*squad, error) {
	if !leavePending && len(logins) < 2 {
		return nil, errors.Errorf("cannot create and start squad with less than 2 members")
	}

	allUserIDs := make([]string, 0, len(logins))
	remainingLogins := make([]string, 0, len(logins))
	for _, login := range logins {
		_, err := strconv.Atoi(login)
		if err != nil {
			remainingLogins = append(remainingLogins, login)
		} else {
			allUserIDs = append(allUserIDs, login)
		}
	}

	if len(remainingLogins) > 0 {
		userIDs, err := u.users.convertLoginsToIDs(ctx, logins)
		if err != nil {
			return nil, err
		}
		allUserIDs = append(allUserIDs, userIDs...)
	}

	owner := allUserIDs[0]
	squadResp, err := u.meepo.CreateSquad(ctx, &meepo.CreateSquadRequest{
		OwnerId:  owner,
		CallerId: owner,
	})
	if err != nil {
		return nil, errors.Wrapf(err, "creating squad failed. owner: %s", owner)
	}
	squadID := squadResp.Squad.Id

	for _, member := range allUserIDs[1:] {
		_, err := u.meepo.CreateMembership(ctx, &meepo.CreateMembershipRequest{
			CallerId:     owner,
			SquadId:      squadID,
			TargetUserId: member,
		})
		if err != nil {
			return nil, errors.Wrapf(err, "adding user to join squad failed.  sender: %s, recipient: %s", owner, member)
		}
	}

	if leavePending {
		return u.getSquadBySquadID(ctx, squadID)
	}
	return u.startSquad(ctx, squadID)
}

func (u *utility) startSquad(ctx context.Context, squadID string) (*squad, error) {
	if squadID == "" {
		return nil, nil
	}

	squad, err := u.getSquadBySquadID(ctx, squadID)
	if err != nil {
		return nil, err
	}
	if squad == nil {
		return nil, errors.Errorf("squad with ID \"%s\" does not exist", squadID)
	}

	resp, err := u.meepo.UpdateSquad(ctx, &meepo.UpdateSquadRequest{
		Id:       squadID,
		Status:   meepo.Squad_LIVE,
		CallerId: squad.OwnerID,
	})
	if err != nil {
		return nil, err
	}

	updatedSquad := resp.GetSquad()
	if squad == nil {
		return nil, errors.Errorf("updating squad with id %s returned nil", squadID)
	}

	return u.processSquadResponse(ctx, updatedSquad)
}

func (u *utility) addMemberToSquad(ctx context.Context, login, squadID string) (*squad, error) {
	recipientID, err := u.users.convertLoginToID(ctx, login)
	if err != nil {
		return nil, err
	}

	squad, err := u.getSquadBySquadID(ctx, squadID)
	if err != nil {
		return nil, err
	}
	if squad == nil {
		return nil, errors.Errorf("squad with ID \"%s\" does not exist", squadID)
	}

	_, err = u.meepo.CreateMembership(ctx, &meepo.CreateMembershipRequest{
		CallerId:     squad.OwnerID,
		SquadId:      squadID,
		TargetUserId: recipientID,
	})
	if err != nil {
		return nil, err
	}

	return u.getSquadBySquadID(ctx, squadID)
}

func (u *utility) removeMember(ctx context.Context, login string) (*squad, error) {
	userID, err := u.users.convertLoginToID(ctx, login)
	if err != nil {
		return nil, err
	}

	getSquadResponse, err := u.meepo.GetSquadByChannelID(ctx, &meepo.GetSquadByChannelIDRequest{
		ChannelId: userID,
	})
	if err != nil {
		return nil, err
	}
	if getSquadResponse.Squad == nil {
		return nil, nil
	}

	squadID := getSquadResponse.Squad.GetId()
	leaveSquadResponse, err := u.meepo.LeaveSquad(ctx, &meepo.LeaveSquadRequest{
		SquadId:  squadID,
		MemberId: userID,
		CallerId: userID,
	})
	if err != nil {
		return nil, err
	}

	return u.processSquadResponse(ctx, leaveSquadResponse.GetSquad())
}

func (u *utility) endSquad(ctx context.Context, squadID string) (*squad, error) {
	getSquadResp, err := u.meepo.GetSquadByID(ctx, &meepo.GetSquadByIDRequest{
		Id: squadID,
	})
	if err != nil {
		return nil, err
	}

	squad := getSquadResp.GetSquad()
	if squad == nil {
		return nil, nil
	}

	for _, member := range squad.MemberIds {
		_, err := u.meepo.LeaveSquad(ctx, &meepo.LeaveSquadRequest{
			MemberId: member,
			SquadId:  squadID,
			CallerId: member,
		})

		if err != nil {
			return nil, err
		}
	}

	return u.getSquadBySquadID(ctx, squadID)
}

func (u *utility) convertSquad(ctx context.Context, sq *meepo.Squad, pendingInvs []*meepo.Invitation, rejectedInvs []*meepo.Invitation) (*squad, error) {
	if sq == nil {
		return nil, nil
	}

	memberLogins, err := u.users.convertIDsToLogins(ctx, sq.MemberIds)
	if err != nil {
		return nil, err
	}

	ownerLogin, err := u.users.convertIDToLogin(ctx, sq.OwnerId)
	if err != nil {
		return nil, err
	}

	var pendingInvitations []*invitation
	var rejectedInvitations []*invitation
	for _, inv := range pendingInvs {
		condensedInvitation, err := u.convertInvitation(ctx, inv)
		if err != nil {
			return nil, err
		}
		pendingInvitations = append(pendingInvitations, condensedInvitation)
	}

	for _, inv := range rejectedInvs {
		condensedInvitation, err := u.convertInvitation(ctx, inv)
		if err != nil {
			return nil, err
		}
		rejectedInvitations = append(rejectedInvitations, condensedInvitation)
	}

	statusStr := meepo.Squad_Status_name[int32(sq.Status)]
	return &squad{
		ID:                  sq.Id,
		MemberIDs:           sq.MemberIds,
		MemberLogins:        memberLogins,
		OwnerLogin:          ownerLogin,
		OwnerID:             sq.OwnerId,
		Status:              statusStr,
		PendingInvitations:  pendingInvitations,
		RejectedInvitations: rejectedInvitations,
	}, nil
}

func (u *utility) printSquad(ctx context.Context, squad *squad) error {
	b, err := json.MarshalIndent(squad, "", "  ")
	if err != nil {
		return err
	}

	fmt.Println(string(b))
	return nil
}

func (u *utility) convertInvitation(ctx context.Context, inv *meepo.Invitation) (*invitation, error) {
	if inv == nil {
		return nil, nil
	}

	recipientLogin, err := u.users.convertIDToLogin(ctx, inv.RecipientId)
	if err != nil {
		return nil, err
	}

	return &invitation{
		ID:             inv.Id,
		RecipientID:    inv.RecipientId,
		RecipientLogin: recipientLogin,
	}, nil
}

func (u *utility) getInvitationsBySquadID(ctx context.Context, squadID string, ownerID string, statuses []meepo.Invitation_Status) ([][]*meepo.Invitation, error) {
	invitations := make([][]*meepo.Invitation, len(statuses))
	for i, status := range statuses {
		getInvitationsResponse, err := u.meepo.GetInvitationsBySquadID(ctx, &meepo.GetInvitationsBySquadIDRequest{
			CallerId: ownerID,
			SquadId:  squadID,
			Status:   status,
		})

		if err != nil {
			return nil, errors.Wrapf(err, "could not get invitations for squad with ID \"%s\" and status %s", squadID, status.String())
		}

		invitations[i] = getInvitationsResponse.GetInvitations()
	}

	return invitations, nil
}

func (u *utility) inviteMember(ctx context.Context, squadID string, recipientIDOrLogin string) (*squad, error) {
	recipientID, err := u.users.convertToID(ctx, recipientIDOrLogin)
	if err != nil {
		return nil, err
	}

	squad, err := u.getSquadBySquadID(ctx, squadID)
	if err != nil {
		return nil, err
	}

	_, err = u.meepo.CreateInvitation(ctx, &meepo.CreateInvitationRequest{
		SenderId:    squad.OwnerID,
		RecipientId: recipientID,
		CallerId:    squad.OwnerID,
	})
	if err != nil {
		return nil, err
	}

	updatedSquad, err := u.getSquadBySquadID(ctx, squadID)
	if err != nil {
		return nil, err
	}

	return updatedSquad, nil
}

func (u *utility) rejectInvitation(ctx context.Context, squadID, invitationID string) (*squad, error) {
	squad, err := u.getSquadBySquadID(ctx, squadID)
	if err != nil {
		return nil, err
	}

	var recipientID string
	for _, inv := range squad.PendingInvitations {
		if inv.ID == invitationID {
			recipientID = inv.RecipientID
		}
	}

	if recipientID == "" {
		return nil, errors.Errorf("could not find invitation %s for squad %s", invitationID, squadID)
	}

	_, err = u.meepo.RejectInvitation(ctx, &meepo.RejectInvitationRequest{
		InvitationId: invitationID,
		CallerId:     recipientID,
	})
	if err != nil {
		return nil, err
	}

	updatedSquad, err := u.getSquadBySquadID(ctx, squadID)
	if err != nil {
		return nil, err
	}

	return updatedSquad, nil
}

func (u *utility) deleteInvitation(ctx context.Context, squadID, invitationID string) (*squad, error) {
	squad, err := u.getSquadBySquadID(ctx, squadID)
	if err != nil {
		return nil, err
	}

	if squad.OwnerID == "" {
		return nil, errors.Errorf("could not delete invitation %s for ended squad %s", invitationID, squadID)
	}

	_, err = u.meepo.DeleteInvitation(ctx, &meepo.DeleteInvitationRequest{
		InvitationId: invitationID,
		CallerId:     squad.OwnerID,
	})
	if err != nil {
		return nil, err
	}

	updatedSquad, err := u.getSquadBySquadID(ctx, squadID)
	if err != nil {
		return nil, err
	}

	return updatedSquad, nil
}
