package clients

import (
	"context"
	"encoding/json"
	"sync"
	"time"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/log"
	service_common "code.justin.tv/feeds/service-common"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/sns"
)

// ChannelStatePublisher facilitates publishing messages on channels that become part of a live squad,
// and channels that are no longer part of a live squad.
//
// This is meant to let Meepo tell the Tags service when it should add a "Squad Stream" tag to a channel
// and when it should remove the "Squad Stream" tag.
type ChannelStatePublisher interface {
	PublishChannelsInLiveSquad(ctx context.Context, squadID string, memberUserIDs []string)
	PublishChannelIsInLiveSquad(ctx context.Context, squadID, memberUserID string)
	PublishChannelOutOfLiveSquad(ctx context.Context, memberUserID string)
}

// ChannelStatePublisherConfig fetches config info for meepo's ChannelStatePublisher
type ChannelStatePublisherConfig struct {
	SNSTopicARN *distconf.Str
}

// Load loads the config for the backend.
func (c *ChannelStatePublisherConfig) Load(d *distconf.Distconf) error {
	c.SNSTopicARN = d.Str("meepo.channel_state_sns_topic_arn", "")
	if c.SNSTopicARN.Get() == "" {
		return errors.New("could not get SNS topic for channel state updates")
	}

	return nil
}

type channelStateMessage struct {
	ChannelID string  `json:"channel_id"`
	SquadID   *string `json:"squad_id"`
	Timestamp string  `json:"timestamp"`
	Version   int     `json:"version"`
}

type channelStatePublisher struct {
	SNSClient *sns.SNS
	Config    *ChannelStatePublisherConfig
	Log       *log.ElevatedLog
}

func (c *channelStatePublisher) PublishChannelsInLiveSquad(ctx context.Context, squadID string, memberUserIDs []string) {
	wg := sync.WaitGroup{}
	for _, memberUserID := range memberUserIDs {
		wg.Add(1)
		go func(m string) {
			defer wg.Done()
			c.PublishChannelIsInLiveSquad(ctx, squadID, m)
		}(memberUserID)
	}
	wg.Wait()
}

func (c *channelStatePublisher) PublishChannelIsInLiveSquad(ctx context.Context, squadID, memberUserID string) {
	c.publishChannelStateMessage(ctx, memberUserID, &squadID)
}

func (c *channelStatePublisher) PublishChannelOutOfLiveSquad(ctx context.Context, memberUserID string) {
	c.publishChannelStateMessage(ctx, memberUserID, nil)
}

func (c *channelStatePublisher) publishChannelStateMessage(ctx context.Context, channelID string, squadID *string) {
	snsTopic := c.Config.SNSTopicARN.Get()

	msg := &channelStateMessage{
		ChannelID: channelID,
		SquadID:   squadID,
		Timestamp: time.Now().UTC().Format(time.RFC3339),
		Version:   1,
	}
	msgJSON, err := json.Marshal(msg)
	if err != nil {
		c.logError(ctx, err, msg, snsTopic)
		return
	}

	messageAttributes := map[string]*sns.MessageAttributeValue{
		"type": {
			DataType:    aws.String("String"),
			StringValue: aws.String("squad-channel-update"),
		},
	}
	publishInput := &sns.PublishInput{
		TopicArn:          aws.String(snsTopic),
		Message:           aws.String(string(msgJSON)),
		MessageAttributes: messageAttributes,
	}

	req, _ := c.SNSClient.PublishRequest(publishInput)
	if err := service_common.ContextSend(ctx, req, c.Log); err != nil {
		c.logError(ctx, err, msg, snsTopic)
		return
	}
}

func (c *channelStatePublisher) logError(ctx context.Context, err error, msg *channelStateMessage, snsTopic string) {
	wrapperErr := errors.Wrap(err, "could not publish channel status to sns")

	squadID := "null"
	if msg.SquadID != nil {
		squadID = *msg.SquadID
	}

	c.Log.LogCtx(ctx,
		"err", wrapperErr,
		"channel_id", msg.ChannelID,
		"squad_id", squadID,
		"sns_topic", snsTopic)
}

var _ ChannelStatePublisher = &channelStatePublisher{}

// NewChannelStatePublisher creates a ChannelStatePublisher that publishes to SNS
func NewChannelStatePublisher(snsClient *sns.SNS, config *ChannelStatePublisherConfig, log *log.ElevatedLog) ChannelStatePublisher {
	return &channelStatePublisher{
		SNSClient: snsClient,
		Config:    config,
		Log:       log,
	}
}
