package clients

import (
	"context"
	"time"

	"code.justin.tv/amzn/TwitchVXFollowingServiceECSTwirp"
	"code.justin.tv/common/twirp"
	"code.justin.tv/foundation/twitchclient"
)

const (
	followsGetFollowTimeout = 200 * time.Millisecond
)

// Follows is an interface based on feeds' follows client
type Follows interface {
	GetFollow(context.Context, *TwitchVXFollowingServiceECSTwirp.FollowReq) (*TwitchVXFollowingServiceECSTwirp.FollowResp, error)
}

// FollowsClient used within meepo
type FollowsClient interface {
	IsFollowing(ctx context.Context, sourceID, targetID string) (bool, error)
}

type followsImpl struct {
	baseClient TwitchVXFollowingServiceECSTwirp.TwitchVXFollowingServiceECS
}

// NewFollowsClient creates a new client for use within meepo
func NewFollowsClient(host string, stats twitchclient.Statter) FollowsClient {
	httpClient := twitchclient.NewHTTPClient(twitchclient.ClientConf{
		Transport: twitchclient.TransportConf{
			MaxIdleConnsPerHost: 100,
		},
		Host:  host,
		Stats: stats,
	})

	followsClient := TwitchVXFollowingServiceECSTwirp.NewTwitchVXFollowingServiceECSProtobufClient(host, httpClient)

	return &followsImpl{baseClient: followsClient}
}

func (u *followsImpl) IsFollowing(ctx context.Context, sourceID, targetID string) (bool, error) {
	ctx, cancel := context.WithTimeout(ctx, followsGetFollowTimeout)
	defer cancel()

	_, err := u.baseClient.GetFollow(ctx, &TwitchVXFollowingServiceECSTwirp.FollowReq{
		FromUserId:   sourceID,
		TargetUserId: targetID,
	})

	if err != nil {
		e, ok := err.(twirp.Error)
		if ok && e.Code() == twirp.NotFound {
			return false, nil
		}

		return false, err
	}

	return true, nil
}
