package clients

import (
	"context"
	"time"

	"code.justin.tv/cb/hallpass/view"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/foundation/twitchclient"
	"github.com/afex/hystrix-go/hystrix"

	hallpassclient "code.justin.tv/cb/hallpass/client/hallpass"
)

const (
	hallpassGetV1IsEditorTimeout = 250 // milliseconds
	getIsEditor                  = "hallpass.get_is_editor"
)

func init() {
	hystrix.Configure(map[string]hystrix.CommandConfig{
		getIsEditor: {
			Timeout:                hallpassGetV1IsEditorTimeout,
			MaxConcurrentRequests:  25,
			RequestVolumeThreshold: 50,
			SleepWindow:            5000, // milliseconds
			ErrorPercentThreshold:  25,
		},
	})
}

// Hallpass is an interface based on CB's hallpass client
type Hallpass interface {
	GetV1IsEditor(ctx context.Context, channelID string, editorID string, reqOpts *twitchclient.ReqOpts) (*view.GetIsEditorResponse, error)
}

// HallpassClient used within meepo
type HallpassClient interface {
	IsEditor(ctx context.Context, channelID, editorID string) (bool, error)
}

type hallpassImpl struct {
	baseClient Hallpass
}

// NewHallpassClient creates a new hallpass client for use within meepo
func NewHallpassClient(host string, stats twitchclient.Statter) (HallpassClient, error) {
	clientConf := twitchclient.ClientConf{
		Transport: twitchclient.TransportConf{
			MaxIdleConnsPerHost: 100,
		},
		Host:  host,
		Stats: stats,
	}

	hallpass, err := hallpassclient.NewClient(clientConf)
	if err != nil {
		return nil, errors.New("failed to start new hallpass client")
	}

	return &hallpassImpl{baseClient: hallpass}, nil
}

func (h *hallpassImpl) IsEditor(ctx context.Context, channelID, editorID string) (bool, error) {
	ctx, cancel := context.WithTimeout(ctx, hallpassGetV1IsEditorTimeout*time.Millisecond)
	defer cancel()

	var resp *view.GetIsEditorResponse
	err := hystrix.DoC(ctx, getIsEditor, func(ctx context.Context) error {
		result, err := h.baseClient.GetV1IsEditor(ctx, channelID, editorID, nil)
		if err != nil || result == nil {
			resp = &view.GetIsEditorResponse{}
			return errors.Wrap(err, "error getting editor status")
		}
		resp = result
		return nil
	}, func(ctx context.Context, e error) error {
		resp = &view.GetIsEditorResponse{}
		return errors.Wrap(e, "hystrix circuit fallback: error performing IsEditor check in Hallpass")
	})

	if err != nil {
		return false, errors.Wrap(err, "error getting editor status")
	}

	return resp.IsEditor, nil
}
