package clients

import (
	"context"
	"time"

	dart "code.justin.tv/amzn/TwitchDartReceiverTwirp"
	"code.justin.tv/feeds/log"
	"code.justin.tv/foundation/twitchclient"
)

const SquadStreamInvite = "squad_stream_invite"

// Notifier is the interface used to send Meepo notifications
type Notifier interface {
	PublishNotification(ctx context.Context, req *dart.PublishNotificationRequest) (*dart.PublishNotificationResponse, error)
}

// NotifierClient is the interface used to send Meepo notifications
type NotifierClient interface {
	SendInviteNotification(ctx context.Context, senderID, recipient, inviteID string) (*dart.PublishNotificationResponse, error)
}

type notifierImpl struct {
	dartReceiver Notifier
	stats        twitchclient.Statter
	log          *log.ElevatedLog
}

// NewNotifier returns a new Notifier
func NewNotifier(host string, logger *log.ElevatedLog, stats twitchclient.Statter) NotifierClient {

	httpClient := twitchclient.NewHTTPClient(twitchclient.ClientConf{
		Transport: twitchclient.TransportConf{
			MaxIdleConnsPerHost: 100,
		},
		Host:  host,
		Stats: stats,
	})

	receiverClient := dart.NewReceiverProtobufClient(host, httpClient)

	return &notifierImpl{
		dartReceiver: receiverClient,
		stats:        stats,
		log:          logger,
	}
}

func (n *notifierImpl) SendInviteNotification(ctx context.Context, sender, recipient, invite string) (*dart.PublishNotificationResponse, error) {
	now := time.Now()

	req := &dart.PublishNotificationRequest{
		NotificationType: SquadStreamInvite, // CustomID from Dart UI
		SenderId:         sender,
		Recipient: &dart.PublishNotificationRequest_RecipientId{
			RecipientId: recipient,
		},
		Metadata: map[string]string{
			"invite_id": invite,
		},
	}
	resp, err := n.dartReceiver.PublishNotification(ctx, req)

	if err != nil {
		reportErr := n.stats.TimingDuration("notification_squadstreaminvite_failure", time.Since(now), 1.0)
		if reportErr != nil {
			n.log.LogCtx(ctx, "report_err", reportErr, "error recording stat")
		}
		return nil, err
	}
	reportErr := n.stats.TimingDuration("notification_squadstreaminvite_success", time.Since(now), 1.0)
	if reportErr != nil {
		n.log.LogCtx(ctx, "report_err", reportErr, "error recording stat")
	}
	return resp, nil
}
