package clients

import (
	"context"
	"encoding/json"
	"testing"

	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/twitch-events/meepo/internal/models"
	"code.justin.tv/twitch-events/meepo/internal/util"

	. "github.com/smartystreets/goconvey/convey"
)

func TestPubsubClient(t *testing.T) {

	Convey("PubsubClient", t, func() {
		ctx := context.Background()
		baseClient := newPubsubStub()

		pubsubClient := pubsubImpl{
			baseClient: baseClient,
		}

		Convey("PublishSquadUpdate", func() {
			Convey("should send messages when the squad is populated", func() {
				squadID, err := util.NewID()
				So(err, ShouldBeNil)
				memberID := util.NewUserID()
				ownerID := util.NewUserID()

				memberIDs := []string{ownerID, memberID}
				squad := &models.ManagedSquad{
					ID:        squadID,
					MemberIDs: memberIDs,
					OwnerID:   &ownerID,
					Status:    models.SquadStatusPending,
				}
				pubsubUsers := []models.PubsubUser{
					{ID: memberID},
					{ID: ownerID},
				}

				managedSquad := models.NewPubsubManagedSquad(squad, pubsubUsers, nil)
				err = pubsubClient.PublishSquadUpdate(ctx, managedSquad)
				So(err, ShouldBeNil)

				// Squad and ManagedSquad messages should have be sent.
				So(baseClient.msgTypeToTopics, ShouldContainKey, PubsubTypeSquad)
				So(baseClient.msgTypeToTopics, ShouldContainKey, PubsubTypeManagedSquad)
				So(baseClient.msgTypeToTopics, ShouldNotContainKey, PubsubTypeReceivedSquadInvites)

				squadTopics := baseClient.msgTypeToTopics[PubsubTypeSquad]
				So(squadTopics, ShouldHaveLength, 4)

				So(squadTopics, ShouldContain, getSquadUpdatesBySquadIDTopic(squadID))

				So(squadTopics, ShouldContain, getSquadUpdatesByChannelIDTopic(memberID))
				So(squadTopics, ShouldContain, getSquadUpdatesByChannelIDTopic(ownerID))

				So(squadTopics, ShouldContain, getCreatorUpdatesByChannelIDTopic(memberID))
				So(squadTopics, ShouldNotContain, getCreatorUpdatesByChannelIDTopic(ownerID))

				managedSquadTopics := baseClient.msgTypeToTopics[PubsubTypeManagedSquad]
				So(managedSquadTopics, ShouldHaveLength, 1)
				So(managedSquadTopics, ShouldContain, getCreatorUpdatesByChannelIDTopic(ownerID))
			})

			Convey("should not send messages when squad is nil", func() {
				err := pubsubClient.PublishSquadUpdate(ctx, nil)
				So(err, ShouldBeNil)
				So(baseClient.msgTypeToTopics, ShouldBeEmpty)
			})
		})

		Convey("PublishChannelNotInSquad", func() {
			channelID := util.NewUserID()

			err := pubsubClient.PublishChannelNotInSquad(ctx, channelID)
			So(err, ShouldBeNil)

			So(baseClient.msgTypeToTopics, ShouldContainKey, PubsubTypeSquad)
			So(baseClient.msgTypeToTopics, ShouldNotContainKey, PubsubTypeManagedSquad)
			So(baseClient.msgTypeToTopics, ShouldNotContainKey, PubsubTypeReceivedSquadInvites)

			squadTopics := baseClient.msgTypeToTopics[PubsubTypeSquad]
			So(squadTopics, ShouldHaveLength, 2)
			So(squadTopics, ShouldContain, getSquadUpdatesByChannelIDTopic(channelID))
			So(squadTopics, ShouldContain, getCreatorUpdatesByChannelIDTopic(channelID))
		})
	})
}

type pubsubMessage struct {
	Type string `json:"type"`
}

func newPubsubStub() *pubsubStub {
	return &pubsubStub{
		msgTypeToTopics: make(map[string][]string),
	}
}

type pubsubStub struct {
	msgTypeToTopics map[string][]string
}

var _ PubSub = &pubsubStub{}

func (p *pubsubStub) Publish(ctx context.Context, topics []string, message string, opts *twitchclient.ReqOpts) error {
	pubsubMsg := pubsubMessage{}
	err := json.Unmarshal([]byte(message), &pubsubMsg)
	if err != nil {
		return err
	}

	msgType := pubsubMsg.Type
	p.msgTypeToTopics[msgType] = append(p.msgTypeToTopics[msgType], topics...)
	return nil
}
