package clients

import (
	"context"
	"time"

	"code.justin.tv/feeds/errors"
	service_common "code.justin.tv/feeds/service-common"
	"code.justin.tv/foundation/twitchclient"
	ripley "code.justin.tv/revenue/ripley/rpc"
)

// UserPayoutType represents the payout properties of a user.
type UserPayoutType struct {
	IsAffiliate bool
	IsPartner   bool
}

// Ripley is an interface based on revenues' ripley client
type Ripley interface {
	GetPayoutType(ctx context.Context, request *ripley.GetPayoutTypeRequest) (*ripley.GetPayoutTypeResponse, error)
}

// RipleyClient used within meepo
type RipleyClient interface {
	GetUserPayoutType(ctx context.Context, channelID string) (*UserPayoutType, error)
}

type ripleyImpl struct {
	baseClient Ripley
	stats      *service_common.StatSender
}

// NewRipleyClient creates a new client for use within meepo
func NewRipleyClient(host string, stats *service_common.StatSender) RipleyClient {
	httpClient := twitchclient.NewHTTPClient(twitchclient.ClientConf{
		Transport: twitchclient.TransportConf{
			MaxIdleConnsPerHost: 500,
		},
		Host:           host,
		TimingXactName: "ripley",
		Stats:          stats,
	})

	ripleyClient := ripley.NewRipleyProtobufClient(host, httpClient)

	return &ripleyImpl{
		baseClient: ripleyClient,
		stats:      stats,
	}
}

func (u *ripleyImpl) GetUserPayoutType(ctx context.Context, channelID string) (*UserPayoutType, error) {
	now := time.Now()

	resp, err := u.baseClient.GetPayoutType(ctx, &ripley.GetPayoutTypeRequest{
		ChannelId: channelID,
	})

	if err != nil {
		u.stats.TimingDurationC("service.ripley.get_payout_type.error", time.Since(now), 1.0)
		return nil, errors.Wrapf(err, "ripley cannot fetch user payout type for user %v", channelID)
	}
	u.stats.TimingDurationC("service.ripley.get_payout_type.success", time.Since(now), 1.0)

	payoutType := resp.GetPayoutType()
	return &UserPayoutType{
		IsPartner:   payoutType.GetIsPartner(),
		IsAffiliate: payoutType.GetIsAffiliate(),
	}, nil
}
