package clients

import (
	"context"
	"fmt"
	"time"

	"code.justin.tv/feeds/errors"
	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/web/users-service/client"
	usersclient "code.justin.tv/web/users-service/client/usersclient_internal"
	"code.justin.tv/web/users-service/models"
)

const (
	usersGetUserByIDAndParamsTimeout = 150 * time.Millisecond
	usersGetUsersTimeout             = 50 * time.Millisecond
)

// Users is an interface based on web's usersclient
type Users interface {
	GetUserByIDAndParams(ctx context.Context, userID string, params *models.FilterParams, reqOpts *twitchclient.ReqOpts) (*models.Properties, error)
	GetUsers(ctx context.Context, params *models.FilterParams, reqOpts *twitchclient.ReqOpts) (*models.PropertiesResult, error)
}

// UsersClient used within meepo
type UsersClient interface {
	GetUserByID(ctx context.Context, userID string) (*models.Properties, error)
	GetUsersByIDs(ctx context.Context, userIDs []string) ([]*models.Properties, error)
	ValidateUser(ctx context.Context, userID string) (bool, error)
}

type usersImpl struct {
	baseClient Users
}

type UserValidationError struct {
	innerErr client.Error
}

func (e *UserValidationError) Error() string {
	return fmt.Sprintf("user validation failed: status code=%v, err=%v", e.innerErr.StatusCode(), e.innerErr.Error())
}

func (e *UserValidationError) StatusCode() int {
	return e.innerErr.StatusCode()
}

type UserInternalServerError struct {
	innerErr error
}

func (e *UserInternalServerError) Error() string {
	return fmt.Sprintf("user validation failed with internal server error: err=%v", e.innerErr.Error())
}

// NewUsersClient creates a new client for use within meepo
func NewUsersClient(host string, stats twitchclient.Statter) (UsersClient, error) {
	clientConf := twitchclient.ClientConf{
		Transport: twitchclient.TransportConf{
			MaxIdleConnsPerHost: 100,
		},
		Host:  host,
		Stats: stats,
	}

	usersClient, err := usersclient.NewClient(clientConf)
	if err != nil {
		return nil, errors.New("failed to start users-service client")
	}

	return &usersImpl{baseClient: usersClient}, nil
}

func (u *usersImpl) GetUsersByIDs(ctx context.Context, userIDs []string) ([]*models.Properties, error) {
	params := &models.FilterParams{
		IDs: userIDs,
	}

	ctx, cancel := context.WithTimeout(ctx, usersGetUsersTimeout)
	defer cancel()

	result, err := u.baseClient.GetUsers(ctx, params, nil)
	if err != nil {
		return nil, err
	}

	if len(result.BadIdentifiers) > 0 {
		return nil, errors.Errorf("users service bad identifiers: %v", result.BadIdentifiers)
	}

	if len(result.Results) != len(userIDs) {
		return nil, errors.Errorf("users service cannot find all users")
	}

	for _, userID := range userIDs {
		var found bool
		for _, property := range result.Results {
			if property == nil {
				return nil, errors.Errorf("users service cannot find all users")
			}

			if property.ID == userID {
				found = true
				break
			}
		}
		if !found {
			return nil, errors.Errorf("users service cannot find all users")
		}
	}

	return result.Results, nil
}

// ValidateUser will return true if the userID is found and passes our criteria. The error indicates why the
// userID failed to validate, and can be either a ValidationError or an InternalServerError.
func (u *usersImpl) ValidateUser(ctx context.Context, userID string) (bool, error) {
	params := &models.FilterParams{
		IDs:             []string{userID},
		NotDeleted:      true,
		NoTOSViolation:  true,
		NoDMCAViolation: true,
	}

	ctx, cancel := context.WithTimeout(ctx, usersGetUserByIDAndParamsTimeout)
	defer cancel()

	_, err := u.baseClient.GetUserByIDAndParams(ctx, userID, params, nil)
	if err != nil {
		if uerr, ok := err.(client.Error); ok {
			if uerr.StatusCode() < 500 {
				return false, &UserValidationError{innerErr: uerr}
			}
		}
		return false, &UserInternalServerError{innerErr: err}
	}
	return true, nil
}

func (u *usersImpl) GetUserByID(ctx context.Context, userID string) (*models.Properties, error) {
	params := &models.FilterParams{
		IDs:             []string{userID},
		NotDeleted:      true,
		NoTOSViolation:  true,
		NoDMCAViolation: true,
	}

	ctx, cancel := context.WithTimeout(ctx, usersGetUserByIDAndParamsTimeout)
	defer cancel()

	user, err := u.baseClient.GetUserByIDAndParams(ctx, userID, params, nil)
	if err != nil {
		return nil, errors.Wrapf(err, "users service cannot find user %v", userID)
	}

	return user, nil
}
