// +build integration

package main

import (
	"context"
	"testing"
	"time"

	"code.justin.tv/twitch-events/meepo/internal/mocks"
	"code.justin.tv/twitch-events/meepo/internal/stubs"
	"code.justin.tv/twitch-events/meepo/internal/util"
	"code.justin.tv/twitch-events/meepo/rpc/meepo"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

func TestChannelLivecheck(t *testing.T) {
	adminUser := util.NewUserID()

	injectables := newDefaultInjectables()

	followsClient := &mocks.FollowsClient{}
	friendshipClient := &mocks.FriendshipClient{}
	rosterClient := &mocks.RosterClient{}
	livelineClient := &stubs.LivelineStub{}
	authorizer := (injectables.authorizer).(*mocks.Authorizer)

	followsClient.On("IsFollowing", mock.Anything, mock.Anything, mock.Anything).Maybe().Return(true, nil)
	friendshipClient.On("IsFriend", mock.Anything, mock.Anything, mock.Anything).Maybe().Return(false, nil)
	rosterClient.On("IsTeammate", mock.Anything, mock.Anything, mock.Anything).Maybe().Return(false, nil)
	authorizer.On("CanUpdateSquad", mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(true)
	authorizer.On("CanGetInvitationsBySquadID", mock.Anything, mock.Anything, mock.Anything).Return(true)

	injectables.followsClient = followsClient
	injectables.friendshipClient = friendshipClient
	injectables.rosterClient = rosterClient
	injectables.livelineClient = livelineClient

	ts := startServer(t, injectables, map[string][]byte{
		// Do not hit AWS Parameter Store
		// This is because tests are run concurrently making it likely
		// that the Parameter store API limit is reached, thus making the test fail
		"ssm.region":        []byte(""),
		"meepo.admin_users": []byte(adminUser),
		// Set to 0 to test ChannelLiveCheck without new squads livecheck exemption period
		"meepo.livecheck_new_squad_grace_period": []byte("0m"),
	})
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		internalClient := ts.internalMeepoClient
		meepoClient := ts.meepoClient
		ctx := context.Background()

		ownerID := util.NewUserID()
		memberID := util.NewUserID()
		memberIDs := []string{ownerID, memberID}
		livelineClient.SetLiveChannels(memberIDs)
		squad := CreateTestSquadWithMemberAndInvitation(ctx, t, meepoClient, injectables, ownerID, memberID, util.NewUserID())
		squadID := squad.Id

		// Set squad as live
		liveSquadRes, err := meepoClient.UpdateSquad(ctx, &meepo.UpdateSquadRequest{
			Id:       squadID,
			Status:   meepo.Squad_LIVE,
			CallerId: ownerID,
		})
		So(err, ShouldBeNil)
		liveSquad := liveSquadRes.Squad

		// reset liveline stub
		livelineClient.SetShouldError(false)
		livelineClient.ClearLiveChannels()

		Convey("Should throw an error and not delete memberships if Liveline fails", func() {
			livelineClient.SetLiveChannels(memberIDs)
			livelineClient.SetShouldError(true)

			req := &meepo.LivecheckChannelsRequest{}
			_, err = internalClient.LivecheckChannels(ctx, req)
			So(err, ShouldNotBeNil)

			res, err := meepoClient.GetSquadByID(ctx, &meepo.GetSquadByIDRequest{
				Id: squadID,
			})
			So(err, ShouldBeNil)
			So(res, ShouldNotBeNil)
			So(res.Squad, ShouldNotBeNil)
			So(res.Squad.Id, ShouldEqual, liveSquad.Id)
			So(res.Squad.OwnerId, ShouldEqual, liveSquad.OwnerId)
			So(res.Squad.Status, ShouldEqual, liveSquad.Status)
			So(res.Squad.MemberIds, ShouldResemble, liveSquad.MemberIds)
		})

		Convey("Should not delete memberships that are live", func() {
			livelineClient.SetLiveChannels(memberIDs)

			// we call livecheck endpoint twice, since the first call changes a offline member's status to pending delete,
			// and the second call deletes the member
			req := &meepo.LivecheckChannelsRequest{}
			_, err = internalClient.LivecheckChannels(ctx, req)
			So(err, ShouldBeNil)

			res, err := meepoClient.GetSquadByID(ctx, &meepo.GetSquadByIDRequest{
				Id: squadID,
			})
			So(err, ShouldBeNil)
			So(res, ShouldNotBeNil)
			So(res.Squad, ShouldNotBeNil)
			So(res.Squad.Id, ShouldEqual, liveSquad.Id)
			So(res.Squad.OwnerId, ShouldEqual, liveSquad.OwnerId)
			So(res.Squad.Status, ShouldEqual, liveSquad.Status)
			So(res.Squad.MemberIds, ShouldResemble, liveSquad.MemberIds)

			_, err = internalClient.LivecheckChannels(ctx, req)
			So(err, ShouldBeNil)

			res, err = meepoClient.GetSquadByID(ctx, &meepo.GetSquadByIDRequest{
				Id: squadID,
			})
			So(err, ShouldBeNil)
			So(res, ShouldNotBeNil)
			So(res.Squad, ShouldNotBeNil)
			So(res.Squad.Id, ShouldEqual, liveSquad.Id)
			So(res.Squad.OwnerId, ShouldEqual, liveSquad.OwnerId)
			So(res.Squad.Status, ShouldEqual, liveSquad.Status)
			So(res.Squad.MemberIds, ShouldResemble, liveSquad.MemberIds)
		})

		Convey("Should not delete memberships that went offline then online", func() {
			livelineClient.SetLiveChannels(memberIDs)

			// member goes offline
			livelineClient.RemoveLiveChannels([]string{memberID})

			// first call should set memberID as pending delete
			req := &meepo.LivecheckChannelsRequest{}
			_, err = internalClient.LivecheckChannels(ctx, req)
			So(err, ShouldBeNil)

			res, err := meepoClient.GetSquadByID(ctx, &meepo.GetSquadByIDRequest{
				Id: squadID,
			})
			So(err, ShouldBeNil)
			So(res, ShouldNotBeNil)
			So(res.Squad, ShouldNotBeNil)
			So(res.Squad.Id, ShouldEqual, liveSquad.Id)
			So(res.Squad.OwnerId, ShouldEqual, liveSquad.OwnerId)
			So(res.Squad.Status, ShouldEqual, liveSquad.Status)
			So(res.Squad.MemberIds, ShouldResemble, liveSquad.MemberIds)

			// member comes back online
			livelineClient.AddLiveChannels([]string{memberID})

			// second call should set memberID as active
			_, err = internalClient.LivecheckChannels(ctx, req)
			So(err, ShouldBeNil)

			res, err = meepoClient.GetSquadByID(ctx, &meepo.GetSquadByIDRequest{
				Id: squadID,
			})
			So(err, ShouldBeNil)
			So(res, ShouldNotBeNil)
			So(res.Squad, ShouldNotBeNil)
			So(res.Squad.Id, ShouldEqual, liveSquad.Id)
			So(res.Squad.OwnerId, ShouldEqual, liveSquad.OwnerId)
			So(res.Squad.Status, ShouldEqual, liveSquad.Status)
			So(res.Squad.MemberIds, ShouldResemble, liveSquad.MemberIds)
		})

		Convey("Should delete memberships that went offline", func() {
			// member is offline
			livelineClient.SetLiveChannels([]string{ownerID})

			// we call livecheck endpoint twice, since the first call changes a offline member's status to pending delete,
			// and the second call deletes the member
			req := &meepo.LivecheckChannelsRequest{}
			_, err = internalClient.LivecheckChannels(ctx, req)
			So(err, ShouldBeNil)

			res, err := meepoClient.GetSquadByID(ctx, &meepo.GetSquadByIDRequest{
				Id: squadID,
			})
			So(err, ShouldBeNil)
			So(res, ShouldNotBeNil)
			So(res.Squad, ShouldNotBeNil)
			So(res.Squad.Id, ShouldEqual, liveSquad.Id)
			So(res.Squad.OwnerId, ShouldEqual, liveSquad.OwnerId)
			So(res.Squad.Status, ShouldEqual, liveSquad.Status)
			So(res.Squad.MemberIds, ShouldResemble, liveSquad.MemberIds)

			_, err = internalClient.LivecheckChannels(ctx, req)
			So(err, ShouldBeNil)

			res, err = meepoClient.GetSquadByID(ctx, &meepo.GetSquadByIDRequest{
				Id: squadID,
			})
			So(err, ShouldBeNil)
			So(res.Squad, ShouldNotBeNil)
			So(res.Squad.Id, ShouldEqual, squadID)
			So(res.Squad.OwnerId, ShouldEqual, ownerID)
			So(res.Squad.Status, ShouldEqual, meepo.Squad_LIVE)
			So(res.Squad.MemberIds, ShouldResemble, []string{ownerID})
		})

		Convey("Should delete owner that went offline and set a new owner", func() {
			// owner is offline
			livelineClient.SetLiveChannels([]string{memberID})

			// we call livecheck endpoint twice, since the first call changes a offline member's status to pending delete,
			// and the second call deletes the member
			req := &meepo.LivecheckChannelsRequest{}
			_, err = internalClient.LivecheckChannels(ctx, req)
			So(err, ShouldBeNil)

			res, err := meepoClient.GetSquadByID(ctx, &meepo.GetSquadByIDRequest{
				Id: squadID,
			})
			So(err, ShouldBeNil)
			So(res, ShouldNotBeNil)
			So(res.Squad, ShouldNotBeNil)
			So(res.Squad.Id, ShouldEqual, liveSquad.Id)
			So(res.Squad.OwnerId, ShouldEqual, liveSquad.OwnerId)
			So(res.Squad.Status, ShouldEqual, liveSquad.Status)
			So(res.Squad.MemberIds, ShouldResemble, liveSquad.MemberIds)

			_, err = internalClient.LivecheckChannels(ctx, req)
			So(err, ShouldBeNil)

			res, err = meepoClient.GetSquadByID(ctx, &meepo.GetSquadByIDRequest{
				Id: squadID,
			})
			So(err, ShouldBeNil)
			So(res.Squad, ShouldNotBeNil)
			So(res.Squad.Id, ShouldEqual, squadID)
			So(res.Squad.OwnerId, ShouldEqual, memberID)
			So(res.Squad.Status, ShouldEqual, meepo.Squad_LIVE)
			So(res.Squad.MemberIds, ShouldResemble, []string{memberID})
		})

		Convey("Should delete members and end the squad when all members are offline", func() {
			// we call livecheck endpoint twice, since the first call changes a offline member's status to pending delete,
			// and the second call deletes the member
			req := &meepo.LivecheckChannelsRequest{}
			_, err = internalClient.LivecheckChannels(ctx, req)
			So(err, ShouldBeNil)

			res, err := meepoClient.GetSquadByID(ctx, &meepo.GetSquadByIDRequest{
				Id: squadID,
			})
			So(err, ShouldBeNil)
			So(res, ShouldNotBeNil)
			So(res.Squad, ShouldNotBeNil)
			So(res.Squad.Id, ShouldEqual, liveSquad.Id)
			So(res.Squad.OwnerId, ShouldEqual, liveSquad.OwnerId)
			So(res.Squad.Status, ShouldEqual, liveSquad.Status)
			So(res.Squad.MemberIds, ShouldResemble, liveSquad.MemberIds)

			_, err = internalClient.LivecheckChannels(ctx, req)
			So(err, ShouldBeNil)

			res, err = meepoClient.GetSquadByID(ctx, &meepo.GetSquadByIDRequest{
				Id: squadID,
			})
			So(err, ShouldBeNil)
			So(res.Squad, ShouldNotBeNil)
			So(res.Squad.Id, ShouldEqual, squadID)
			So(res.Squad.OwnerId, ShouldBeBlank)
			So(res.Squad.Status, ShouldEqual, meepo.Squad_ENDED)
			So(res.Squad.MemberIds, ShouldBeEmpty)

			getInvitationsReq := &meepo.GetInvitationsBySquadIDRequest{
				SquadId:  squadID,
				CallerId: adminUser,
				Status:   meepo.Invitation_PENDING,
			}
			i, err := meepoClient.GetInvitationsBySquadID(ctx, getInvitationsReq)
			So(err, ShouldBeNil)
			So(i, ShouldNotBeNil)
			So(i.Invitations, ShouldBeEmpty)
		})

		Convey("Should not delete memberships from pending squads", func() {
			// create a pending squad
			ownerID2 := util.NewUserID()
			memberID2 := util.NewUserID()

			squad2 := CreateTestSquadWithMemberAndInvitation(ctx, t, meepoClient, injectables, ownerID2, memberID2, util.NewUserID())
			squadID2 := squad2.Id

			// we call livecheck endpoint twice, since the first call changes a offline member's status to pending delete,
			// and the second call deletes the member
			req := &meepo.LivecheckChannelsRequest{}
			_, err := internalClient.LivecheckChannels(ctx, req)
			So(err, ShouldBeNil)

			res, err := meepoClient.GetSquadByID(ctx, &meepo.GetSquadByIDRequest{
				Id: squadID2,
			})
			So(err, ShouldBeNil)
			So(res, ShouldNotBeNil)
			So(res.Squad, ShouldNotBeNil)
			So(res.Squad.Id, ShouldEqual, squad2.Id)
			So(res.Squad.OwnerId, ShouldEqual, squad2.OwnerId)
			So(res.Squad.Status, ShouldEqual, squad2.Status)
			So(res.Squad.MemberIds, ShouldResemble, squad2.MemberIds)

			_, err = internalClient.LivecheckChannels(ctx, req)
			So(err, ShouldBeNil)

			res, err = meepoClient.GetSquadByID(ctx, &meepo.GetSquadByIDRequest{
				Id: squadID2,
			})
			So(err, ShouldBeNil)
			So(res, ShouldNotBeNil)
			So(res.Squad, ShouldNotBeNil)
			So(res.Squad.Id, ShouldEqual, squad2.Id)
			So(res.Squad.OwnerId, ShouldEqual, squad2.OwnerId)
			So(res.Squad.Status, ShouldEqual, squad2.Status)
			So(res.Squad.MemberIds, ShouldResemble, squad2.MemberIds)
		})
	})

	ts.onFinish(serverShutdownWaitTime)
}

func TestChannelLivecheckWithLiveSquadExemption(t *testing.T) {
	adminUser := util.NewUserID()

	injectables := newDefaultInjectables()

	followsClient := &mocks.FollowsClient{}
	friendshipClient := &mocks.FriendshipClient{}
	rosterClient := &mocks.RosterClient{}
	livelineClient := &stubs.LivelineStub{}
	clock := &stubs.ClockStub{}
	authorizer := (injectables.authorizer).(*mocks.Authorizer)

	followsClient.On("IsFollowing", mock.Anything, mock.Anything, mock.Anything).Maybe().Return(true, nil)
	friendshipClient.On("IsFriend", mock.Anything, mock.Anything, mock.Anything).Maybe().Return(false, nil)
	rosterClient.On("IsTeammate", mock.Anything, mock.Anything, mock.Anything).Maybe().Return(false, nil)
	authorizer.On("CanUpdateSquad", mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(true)

	injectables.followsClient = followsClient
	injectables.friendshipClient = friendshipClient
	injectables.rosterClient = rosterClient
	injectables.livelineClient = livelineClient
	injectables.clock = clock

	ts := startServer(t, injectables, map[string][]byte{
		// Do not hit AWS Parameter Store
		// This is because tests are run concurrently making it likely
		// that the Parameter store API limit is reached, thus making the test fail
		"ssm.region":                             []byte(""),
		"meepo.admin_users":                      []byte(adminUser),
		"meepo.livecheck_new_squad_grace_period": []byte("10m"),
	})
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		internalClient := ts.internalMeepoClient
		meepoClient := ts.meepoClient
		ctx := context.Background()
		clock.SetNow(time.Now())

		ownerID := util.NewUserID()
		memberID := util.NewUserID()
		memberIDs := []string{ownerID, memberID}
		livelineClient.SetLiveChannels(memberIDs)
		squad := CreateTestSquadWithMemberAndInvitation(ctx, t, meepoClient, injectables, ownerID, memberID, util.NewUserID())
		squadID := squad.Id

		// Set squad as live
		liveSquadRes, err := meepoClient.UpdateSquad(ctx, &meepo.UpdateSquadRequest{
			Id:       squadID,
			Status:   meepo.Squad_LIVE,
			CallerId: ownerID,
		})
		So(err, ShouldBeNil)
		liveSquad := liveSquadRes.Squad

		// reset liveline stub
		livelineClient.SetShouldError(false)
		livelineClient.ClearLiveChannels()

		Convey("Should not delete new squads", func() {
			clock.AddNow(time.Minute)

			// member is offline
			livelineClient.SetLiveChannels([]string{ownerID})

			// we call livecheck endpoint twice
			req := &meepo.LivecheckChannelsRequest{}
			_, err = internalClient.LivecheckChannels(ctx, req)
			So(err, ShouldBeNil)

			res, err := meepoClient.GetSquadByID(ctx, &meepo.GetSquadByIDRequest{
				Id: squadID,
			})
			So(err, ShouldBeNil)
			So(res, ShouldNotBeNil)
			So(res.Squad, ShouldNotBeNil)
			So(res.Squad.Id, ShouldEqual, liveSquad.Id)
			So(res.Squad.OwnerId, ShouldEqual, liveSquad.OwnerId)
			So(res.Squad.Status, ShouldEqual, liveSquad.Status)
			So(res.Squad.MemberIds, ShouldResemble, liveSquad.MemberIds)

			_, err = internalClient.LivecheckChannels(ctx, req)
			So(err, ShouldBeNil)

			// Member should still not be deleted
			res, err = meepoClient.GetSquadByID(ctx, &meepo.GetSquadByIDRequest{
				Id: squadID,
			})
			So(err, ShouldBeNil)
			So(res.Squad, ShouldNotBeNil)
			So(res.Squad.Id, ShouldEqual, squadID)
			So(res.Squad.OwnerId, ShouldEqual, ownerID)
			So(res.Squad.Status, ShouldEqual, meepo.Squad_LIVE)
			So(res.Squad.MemberIds, ShouldResemble, memberIDs)
		})

		Convey("Should delete old squads", func() {
			clock.AddNow(time.Minute * 15)

			// member is offline
			livelineClient.SetLiveChannels([]string{ownerID})

			// we call livecheck endpoint twice, since the first call changes a offline member's status to pending delete,
			// and the second call deletes the member
			req := &meepo.LivecheckChannelsRequest{}
			_, err = internalClient.LivecheckChannels(ctx, req)
			So(err, ShouldBeNil)

			res, err := meepoClient.GetSquadByID(ctx, &meepo.GetSquadByIDRequest{
				Id: squadID,
			})
			So(err, ShouldBeNil)
			So(res, ShouldNotBeNil)
			So(res.Squad, ShouldNotBeNil)
			So(res.Squad.Id, ShouldEqual, liveSquad.Id)
			So(res.Squad.OwnerId, ShouldEqual, liveSquad.OwnerId)
			So(res.Squad.Status, ShouldEqual, liveSquad.Status)
			So(res.Squad.MemberIds, ShouldResemble, liveSquad.MemberIds)

			_, err = internalClient.LivecheckChannels(ctx, req)
			So(err, ShouldBeNil)

			res, err = meepoClient.GetSquadByID(ctx, &meepo.GetSquadByIDRequest{
				Id: squadID,
			})
			So(err, ShouldBeNil)
			So(res.Squad, ShouldNotBeNil)
			So(res.Squad.Id, ShouldEqual, squadID)
			So(res.Squad.OwnerId, ShouldEqual, ownerID)
			So(res.Squad.Status, ShouldEqual, meepo.Squad_LIVE)
			So(res.Squad.MemberIds, ShouldResemble, []string{ownerID})
		})
	})

	ts.onFinish(serverShutdownWaitTime)
}
