// +build integration

package main

import (
	"context"
	"testing"

	"code.justin.tv/twitch-events/meepo/internal/mocks"
	"code.justin.tv/twitch-events/meepo/internal/util"
	"code.justin.tv/twitch-events/meepo/rpc/meepo"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

func TestDestroyUser(t *testing.T) {
	adminUser := util.NewUserID()

	injectables := newDefaultInjectables()

	followsClient := &mocks.FollowsClient{}
	friendshipClient := &mocks.FriendshipClient{}
	rosterClient := &mocks.RosterClient{}
	authorizer := (injectables.authorizer).(*mocks.Authorizer)

	followsClient.On("IsFollowing", mock.Anything, mock.Anything, mock.Anything).Maybe().Return(true, nil)
	friendshipClient.On("IsFriend", mock.Anything, mock.Anything, mock.Anything).Maybe().Return(false, nil)
	rosterClient.On("IsTeammate", mock.Anything, mock.Anything, mock.Anything).Maybe().Return(false, nil)
	authorizer.On("CanUpdateSquad", mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(true)
	authorizer.On("CanGetInvitationsBySquadID", mock.Anything, mock.Anything, mock.Anything).Return(true)
	authorizer.On("CanGetPendingInvitationsByRecipientID", mock.Anything, mock.Anything, mock.Anything).Return(true)
	authorizer.On("CanRejectInvitation", mock.Anything, mock.Anything, mock.Anything).Return(true)

	injectables.followsClient = followsClient
	injectables.friendshipClient = friendshipClient
	injectables.rosterClient = rosterClient

	ts := startServer(t, injectables, map[string][]byte{
		// Do not hit AWS Parameter Store
		// This is because tests are run concurrently making it likely
		// that the Parameter store API limit is reached, thus making the test fail
		"ssm.region":        []byte(""),
		"meepo.admin_users": []byte(adminUser),
	})
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		internalClient := ts.internalMeepoClient
		meepoClient := ts.meepoClient
		ctx := context.Background()

		userID := util.NewUserID()
		req := &meepo.DestroyUserRequest{
			ChannelId: userID,
		}

		Convey("Should succeed on user without any squad stream data", func() {
			_, err := internalClient.DestroyUser(ctx, req)
			So(err, ShouldBeNil)
		})

		Convey("Should reset invite policy to only in network", func() {
			invitePolicy := meepo.InvitePolicy_ALL
			_, err := meepoClient.UpdateInvitePolicyByChannelID(ctx, &meepo.UpdateInvitePolicyByChannelIDRequest{
				ChannelId:    userID,
				CallerId:     adminUser,
				InvitePolicy: invitePolicy,
			})
			So(err, ShouldBeNil)

			res, err := meepoClient.GetInvitePolicyByChannelID(ctx, &meepo.GetInvitePolicyByChannelIDRequest{
				ChannelId: userID,
				CallerId:  adminUser,
			})
			So(err, ShouldBeNil)
			So(res, ShouldNotBeNil)
			So(res.InvitePolicy, ShouldEqual, invitePolicy)

			_, err = internalClient.DestroyUser(ctx, req)
			So(err, ShouldBeNil)

			res, err = meepoClient.GetInvitePolicyByChannelID(ctx, &meepo.GetInvitePolicyByChannelIDRequest{
				ChannelId: userID,
				CallerId:  adminUser,
			})
			So(err, ShouldBeNil)
			So(res.GetInvitePolicy(), ShouldEqual, meepo.InvitePolicy_NETWORK)
		})

		Convey("Should leave current squad if the user is a member", func() {
			ownerID := util.NewUserID()
			memberIDs := []string{ownerID}
			squad := CreateTestSquadWithMember(ctx, t, meepoClient, injectables, ownerID, userID)
			squadID := squad.Id

			_, err := internalClient.DestroyUser(ctx, req)
			So(err, ShouldBeNil)

			res, err := meepoClient.GetSquadByID(ctx, &meepo.GetSquadByIDRequest{
				Id: squadID,
			})
			So(err, ShouldBeNil)
			So(res.GetSquad().GetOwnerId(), ShouldEqual, ownerID)
			So(res.GetSquad().GetMemberIds(), ShouldResemble, memberIDs)
			So(res.GetSquad().GetStatus(), ShouldEqual, squad.GetStatus())
		})

		Convey("Should leave current squad if the user is an owner", func() {
			memberID := util.NewUserID()
			memberIDs := []string{memberID}
			squad := CreateTestSquadWithMember(ctx, t, meepoClient, injectables, userID, memberID)
			squadID := squad.Id
			squad.MemberIds = memberIDs
			squad.OwnerId = memberID

			_, err := internalClient.DestroyUser(ctx, req)
			So(err, ShouldBeNil)

			res, err := meepoClient.GetSquadByID(ctx, &meepo.GetSquadByIDRequest{
				Id: squadID,
			})
			So(err, ShouldBeNil)
			So(res.GetSquad().GetOwnerId(), ShouldEqual, memberID)
			So(res.GetSquad().GetMemberIds(), ShouldResemble, memberIDs)
			So(res.GetSquad().GetStatus(), ShouldEqual, squad.GetStatus())
		})

		Convey("Should delete pending and rejected incoming invitations", func() {
			ownerID := util.NewUserID()
			memberID := util.NewUserID()
			squad := CreateTestSquadWithMemberAndInvitation(ctx, t, meepoClient, injectables, ownerID, memberID, userID)
			squadID := squad.Id

			ownerID2 := util.NewUserID()
			memberID2 := util.NewUserID()
			squad2 := CreateTestSquadWithMemberAndInvitation(ctx, t, meepoClient, injectables, ownerID2, memberID2, userID)
			squadID2 := squad2.Id

			invitationsRes, err := meepoClient.GetIncomingInvitationsByChannelID(ctx, &meepo.GetIncomingInvitationsByChannelIDRequest{
				ChannelId: userID,
				CallerId:  adminUser,
			})
			So(err, ShouldBeNil)
			So(invitationsRes.GetInvitations(), ShouldHaveLength, 2)
			rejectInvitation := invitationsRes.GetInvitations()[1]
			_, err = meepoClient.RejectInvitation(ctx, &meepo.RejectInvitationRequest{
				InvitationId: rejectInvitation.Id,
				CallerId:     adminUser,
			})
			So(err, ShouldBeNil)

			_, err = internalClient.DestroyUser(ctx, req)
			So(err, ShouldBeNil)

			res, err := meepoClient.GetInvitationsBySquadID(ctx, &meepo.GetInvitationsBySquadIDRequest{
				SquadId:  squadID,
				CallerId: adminUser,
				Status:   meepo.Invitation_PENDING,
			})
			So(err, ShouldBeNil)
			So(res.GetInvitations(), ShouldHaveLength, 0)

			res, err = meepoClient.GetInvitationsBySquadID(ctx, &meepo.GetInvitationsBySquadIDRequest{
				SquadId:  squadID,
				CallerId: adminUser,
				Status:   meepo.Invitation_REJECTED,
			})
			So(err, ShouldBeNil)
			So(res.GetInvitations(), ShouldHaveLength, 0)

			res, err = meepoClient.GetInvitationsBySquadID(ctx, &meepo.GetInvitationsBySquadIDRequest{
				SquadId:  squadID2,
				CallerId: adminUser,
				Status:   meepo.Invitation_PENDING,
			})
			So(err, ShouldBeNil)
			So(res.GetInvitations(), ShouldHaveLength, 0)

			res, err = meepoClient.GetInvitationsBySquadID(ctx, &meepo.GetInvitationsBySquadIDRequest{
				SquadId:  squadID2,
				CallerId: adminUser,
				Status:   meepo.Invitation_REJECTED,
			})
			So(err, ShouldBeNil)
			So(res.GetInvitations(), ShouldHaveLength, 0)
		})
	})

	ts.onFinish(serverShutdownWaitTime)
}

func TestDeleteUser(t *testing.T) {
	adminUser := util.NewUserID()

	injectables := newDefaultInjectables()

	followsClient := &mocks.FollowsClient{}
	friendshipClient := &mocks.FriendshipClient{}
	rosterClient := &mocks.RosterClient{}
	authorizer := (injectables.authorizer).(*mocks.Authorizer)

	followsClient.On("IsFollowing", mock.Anything, mock.Anything, mock.Anything).Maybe().Return(true, nil)
	friendshipClient.On("IsFriend", mock.Anything, mock.Anything, mock.Anything).Maybe().Return(false, nil)
	rosterClient.On("IsTeammate", mock.Anything, mock.Anything, mock.Anything).Maybe().Return(false, nil)
	authorizer.On("CanUpdateSquad", mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(true)
	authorizer.On("CanGetInvitationsBySquadID", mock.Anything, mock.Anything, mock.Anything).Return(true)
	authorizer.On("CanGetPendingInvitationsByRecipientID", mock.Anything, mock.Anything, mock.Anything).Return(true)
	authorizer.On("CanRejectInvitation", mock.Anything, mock.Anything, mock.Anything).Return(true)

	injectables.followsClient = followsClient
	injectables.friendshipClient = friendshipClient
	injectables.rosterClient = rosterClient

	ts := startServer(t, injectables, map[string][]byte{
		// Do not hit AWS Parameter Store
		// This is because tests are run concurrently making it likely
		// that the Parameter store API limit is reached, thus making the test fail
		"ssm.region":        []byte(""),
		"meepo.admin_users": []byte(adminUser),
	})
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		internalClient := ts.internalMeepoClient
		meepoClient := ts.meepoClient
		ctx := context.Background()

		userID := util.NewUserID()
		req := &meepo.DeleteUserRequest{
			ChannelId: userID,
		}

		Convey("Should succeed on user without any squad stream data", func() {
			_, err := internalClient.DeleteUser(ctx, req)
			So(err, ShouldBeNil)
		})

		Convey("Should leave current squad if the user is a member", func() {
			ownerID := util.NewUserID()
			memberIDs := []string{ownerID}
			squad := CreateTestSquadWithMember(ctx, t, meepoClient, injectables, ownerID, userID)
			squadID := squad.Id

			_, err := internalClient.DeleteUser(ctx, req)
			So(err, ShouldBeNil)

			res, err := meepoClient.GetSquadByID(ctx, &meepo.GetSquadByIDRequest{
				Id: squadID,
			})
			So(err, ShouldBeNil)
			So(res.GetSquad().GetOwnerId(), ShouldEqual, ownerID)
			So(res.GetSquad().GetMemberIds(), ShouldResemble, memberIDs)
			So(res.GetSquad().GetStatus(), ShouldEqual, squad.GetStatus())
		})

		Convey("Should leave current squad if the user is an owner", func() {
			memberID := util.NewUserID()
			memberIDs := []string{memberID}
			squad := CreateTestSquadWithMember(ctx, t, meepoClient, injectables, userID, memberID)
			squadID := squad.Id
			squad.MemberIds = memberIDs
			squad.OwnerId = memberID

			_, err := internalClient.DeleteUser(ctx, req)
			So(err, ShouldBeNil)

			res, err := meepoClient.GetSquadByID(ctx, &meepo.GetSquadByIDRequest{
				Id: squadID,
			})
			So(err, ShouldBeNil)
			So(res.GetSquad().GetOwnerId(), ShouldEqual, memberID)
			So(res.GetSquad().GetMemberIds(), ShouldResemble, memberIDs)
			So(res.GetSquad().GetStatus(), ShouldEqual, squad.GetStatus())
		})

		Convey("Should delete pending and rejected incoming invitations", func() {
			ownerID := util.NewUserID()
			memberID := util.NewUserID()
			squad := CreateTestSquadWithMemberAndInvitation(ctx, t, meepoClient, injectables, ownerID, memberID, userID)
			squadID := squad.Id

			ownerID2 := util.NewUserID()
			memberID2 := util.NewUserID()
			squad2 := CreateTestSquadWithMemberAndInvitation(ctx, t, meepoClient, injectables, ownerID2, memberID2, userID)
			squadID2 := squad2.Id

			invitationsRes, err := meepoClient.GetIncomingInvitationsByChannelID(ctx, &meepo.GetIncomingInvitationsByChannelIDRequest{
				ChannelId: userID,
				CallerId:  adminUser,
			})
			So(err, ShouldBeNil)
			So(invitationsRes.GetInvitations(), ShouldHaveLength, 2)
			rejectInvitation := invitationsRes.GetInvitations()[1]
			_, err = meepoClient.RejectInvitation(ctx, &meepo.RejectInvitationRequest{
				InvitationId: rejectInvitation.Id,
				CallerId:     adminUser,
			})
			So(err, ShouldBeNil)

			_, err = internalClient.DeleteUser(ctx, req)
			So(err, ShouldBeNil)

			res, err := meepoClient.GetInvitationsBySquadID(ctx, &meepo.GetInvitationsBySquadIDRequest{
				SquadId:  squadID,
				CallerId: adminUser,
				Status:   meepo.Invitation_PENDING,
			})
			So(err, ShouldBeNil)
			So(res.GetInvitations(), ShouldHaveLength, 0)

			res, err = meepoClient.GetInvitationsBySquadID(ctx, &meepo.GetInvitationsBySquadIDRequest{
				SquadId:  squadID,
				CallerId: adminUser,
				Status:   meepo.Invitation_REJECTED,
			})
			So(err, ShouldBeNil)
			So(res.GetInvitations(), ShouldHaveLength, 0)

			res, err = meepoClient.GetInvitationsBySquadID(ctx, &meepo.GetInvitationsBySquadIDRequest{
				SquadId:  squadID2,
				CallerId: adminUser,
				Status:   meepo.Invitation_PENDING,
			})
			So(err, ShouldBeNil)
			So(res.GetInvitations(), ShouldHaveLength, 0)

			res, err = meepoClient.GetInvitationsBySquadID(ctx, &meepo.GetInvitationsBySquadIDRequest{
				SquadId:  squadID2,
				CallerId: adminUser,
				Status:   meepo.Invitation_REJECTED,
			})
			So(err, ShouldBeNil)
			So(res.GetInvitations(), ShouldHaveLength, 0)
		})
	})

	ts.onFinish(serverShutdownWaitTime)
}
