package main

import (
	"errors"
	"net"
	"os"
	"os/signal"

	"code.justin.tv/twitch-events/meepo/internal/auth"
	"code.justin.tv/vod/eventbus"

	"code.justin.tv/feeds/distconf"
	service_common "code.justin.tv/feeds/service-common"
	"code.justin.tv/twitch-events/meepo/clients"
	"code.justin.tv/twitch-events/meepo/internal/api"
	"code.justin.tv/twitch-events/meepo/internal/backend"
	"code.justin.tv/twitch-events/meepo/internal/cache"
	"code.justin.tv/twitch-events/meepo/internal/clock"
	"code.justin.tv/twitch-events/meepo/internal/datastore"
	"code.justin.tv/twitch-events/meepo/internal/playerstore"
	"github.com/aws/aws-sdk-go/service/sns"
)

const (
	teamName    = "events"
	serviceName = "meepo"
)

// CodeVersion is set by the build script.
var CodeVersion string

var instance = service{
	osExit: os.Exit,
	serviceCommon: service_common.ServiceCommon{
		ConfigCommon: service_common.ConfigCommon{
			Team:       teamName,
			Service:    serviceName,
			OsGetenv:   os.Getenv,
			OsHostname: os.Hostname,
		},
		CodeVersion: CodeVersion,
	},
}

// nolint: structcheck
type injectables struct {
	authorizer                  auth.Authorizer
	followsClient               clients.FollowsClient
	friendshipClient            clients.FriendshipClient
	hallpassClient              clients.HallpassClient
	livelineClient              clients.LivelineClient
	pubsubClient                clients.PubsubClient
	ripleyClient                clients.RipleyClient
	rosterClient                clients.RosterClient
	spadeClient                 clients.SpadeClient
	usersClient                 clients.UsersClient
	channelStatePublisherClient clients.ChannelStatePublisher
	dartClient                  clients.NotifierClient
	pdmsClient                  clients.PDMSClient
	clock                       clock.Clock
}

type clientConfig struct {
	followsURL     *distconf.Str
	friendshipURL  *distconf.Str
	dartURL        *distconf.Str
	hallpassURL    *distconf.Str
	livelineURL    *distconf.Str
	pubsubURL      *distconf.Str
	ripleyURL      *distconf.Str
	rosterURL      *distconf.Str
	usersURL       *distconf.Str
	pdmsCallerRole *distconf.Str
	pdmsLambdaARN  *distconf.Str
}

func (c *clientConfig) Load(dconf *distconf.Distconf) error {
	c.pubsubURL = dconf.Str("meepo.clients.pubsub.url", "")
	if c.pubsubURL.Get() == "" {
		return errors.New("unable to find a valid pubsub url")
	}

	c.usersURL = dconf.Str("meepo.clients.users.url", "")
	if c.usersURL.Get() == "" {
		return errors.New("unable to find a valid users url")
	}

	c.livelineURL = dconf.Str("meepo.clients.liveline.url", "")
	if c.livelineURL.Get() == "" {
		return errors.New("unable to find a valid liveline url")
	}

	c.followsURL = dconf.Str("meepo.clients.follows.url", "")
	if c.followsURL.Get() == "" {
		return errors.New("unable to find a valid follows url")
	}

	c.friendshipURL = dconf.Str("meepo.clients.friendship.url", "")
	if c.friendshipURL.Get() == "" {
		return errors.New("unable to find a valid friendship url")
	}

	c.rosterURL = dconf.Str("meepo.clients.roster.url", "")
	if c.rosterURL.Get() == "" {
		return errors.New("unable to find a valid roster url")
	}

	c.ripleyURL = dconf.Str("meepo.clients.ripley.url", "")
	if c.ripleyURL.Get() == "" {
		return errors.New("unable to find a valid ripley url")
	}

	c.hallpassURL = dconf.Str("meepo.clients.hallpass.url", "")
	if c.hallpassURL.Get() == "" {
		return errors.New("unable to find a valid hallpass url")
	}

	c.dartURL = dconf.Str("meepo.clients.dart.url", "")
	if c.dartURL.Get() == "" {
		return errors.New("unable to find a valid dart url")
	}

	c.pdmsCallerRole = dconf.Str("meepo.clients.pdms.callerrole", "")
	if c.pdmsCallerRole.Get() == "" {
		return errors.New("unable to find a valid pdms caller role")
	}

	c.pdmsLambdaARN = dconf.Str("meepo.clients.pdms.lambdaarn", "")
	if c.pdmsLambdaARN.Get() == "" {
		return errors.New("unable to find a valid pdms lambda arn")
	}

	return nil
}

type service struct {
	injectables
	osExit        func(code int)
	onListen      func(net.Addr)
	sigChan       chan os.Signal
	serviceCommon service_common.ServiceCommon
	runner        service_common.ServiceRunner

	configs struct {
		httpConfig                  api.HTTPConfig
		meepoDBConfig               datastore.DBConfig
		cacheConfig                 cache.Config
		authConfig                  auth.Config
		backendConfig               backend.Config
		playerstoreConfig           playerstore.Config
		channelStatePublisherConfig clients.ChannelStatePublisherConfig

		clientConfig clientConfig
	}

	server       api.HTTPServer
	cacheService *cache.Service
	auth         auth.Auth
	backend      backend.Backender
	datastore    datastore.Datastore
	playerstore  *playerstore.PlayerStore
}

func (s *service) setup() error {
	if err := s.serviceCommon.Setup(); err != nil {
		return err
	}
	if err := service_common.LoadConfigs(
		s.serviceCommon.Config,
		&s.configs.httpConfig,
		&s.configs.meepoDBConfig,
		&s.configs.cacheConfig,
		&s.configs.authConfig,
		&s.configs.backendConfig,
		&s.configs.playerstoreConfig,
		&s.configs.channelStatePublisherConfig,
		&s.configs.clientConfig,
	); err != nil {
		return err
	}

	if err := s.configs.meepoDBConfig.LoadSecrets(s.serviceCommon.Secrets); err != nil {
		return err
	}

	clientsStatsSender := &service_common.StatSender{
		SubStatter:   s.serviceCommon.Statsd.NewSubStatter("clients"),
		ErrorTracker: &s.serviceCommon.ErrorTracker,
	}

	if s.pubsubClient == nil {
		pubsubClient, err := clients.NewPubSubClient(s.configs.clientConfig.pubsubURL.Get(), clientsStatsSender)
		if err != nil {
			return err
		}
		s.pubsubClient = pubsubClient
	}

	if s.usersClient == nil {
		usersClient, err := clients.NewUsersClient(s.configs.clientConfig.usersURL.Get(), clientsStatsSender)
		if err != nil {
			return err
		}
		s.usersClient = usersClient
	}

	if s.livelineClient == nil {
		s.livelineClient = clients.NewLivelineClient(s.configs.clientConfig.livelineURL.Get(), clientsStatsSender)
	}

	if s.followsClient == nil {
		s.followsClient = clients.NewFollowsClient(s.configs.clientConfig.followsURL.Get(), clientsStatsSender)
	}

	if s.friendshipClient == nil {
		friendshipClient, err := clients.NewFriendshipClient(s.configs.clientConfig.friendshipURL.Get(), clientsStatsSender)
		if err != nil {
			return err
		}
		s.friendshipClient = friendshipClient
	}

	if s.rosterClient == nil {
		rosterClient, err := clients.NewRosterClient(s.configs.clientConfig.rosterURL.Get(), clientsStatsSender)
		if err != nil {
			return err
		}
		s.rosterClient = rosterClient
	}

	if s.ripleyClient == nil {
		ripleyClient := clients.NewRipleyClient(s.configs.clientConfig.ripleyURL.Get(), clientsStatsSender)
		s.ripleyClient = ripleyClient
	}

	if s.spadeClient == nil {
		spadeClient, err := clients.NewSpadeClient()
		if err != nil {
			return err
		}
		s.spadeClient = spadeClient
	}

	if s.clock == nil {
		s.clock = &clock.RealClock{}
	}

	if s.hallpassClient == nil {
		hallpassClient, err := clients.NewHallpassClient(s.configs.clientConfig.hallpassURL.Get(), clientsStatsSender)
		if err != nil {
			return err
		}
		s.hallpassClient = hallpassClient
	}

	dbStatsSender := &service_common.StatSender{
		SubStatter:   s.serviceCommon.Statsd.NewSubStatter("db"),
		ErrorTracker: &s.serviceCommon.ErrorTracker,
	}

	datastore, err := datastore.NewDatastore(&s.configs.meepoDBConfig, s.serviceCommon.Log, dbStatsSender, s.clock)
	if err != nil {
		return err
	}
	s.datastore = datastore

	cacheService, err := cache.NewCacheService(&s.configs.cacheConfig)
	if err != nil {
		return err
	}
	s.cacheService = cacheService

	session, awsConf := service_common.CreateAWSSession(s.serviceCommon.Config)
	if s.channelStatePublisherClient == nil {
		s.channelStatePublisherClient = clients.NewChannelStatePublisher(
			sns.New(session, awsConf...),
			&s.configs.channelStatePublisherConfig,
			s.serviceCommon.Log,
		)
	}
	if s.dartClient == nil {
		s.dartClient = clients.NewNotifier(s.configs.clientConfig.dartURL.Get(), s.serviceCommon.Log, clientsStatsSender)
	}
	if s.pdmsClient == nil {
		s.pdmsClient = clients.NewPDMSClient(s.configs.clientConfig.pdmsCallerRole.Get(), s.configs.clientConfig.pdmsLambdaARN.Get())
	}

	clients := &backend.Clients{
		Follows:               s.followsClient,
		Friendship:            s.friendshipClient,
		Liveline:              s.livelineClient,
		Pubsub:                s.pubsubClient,
		Ripley:                s.ripleyClient,
		Roster:                s.rosterClient,
		Spade:                 s.spadeClient,
		Users:                 s.usersClient,
		ChannelStatePublisher: s.channelStatePublisherClient,
		PDMS:                  s.pdmsClient,
		Dart:                  s.dartClient,
	}

	backendStatsSender := &service_common.StatSender{
		SubStatter:   s.serviceCommon.Statsd.NewSubStatter("backend"),
		ErrorTracker: &s.serviceCommon.ErrorTracker,
	}

	backend, err := backend.NewBackend(&s.configs.backendConfig, s.datastore, clients, s.cacheService, s.serviceCommon.Log, backendStatsSender, s.clock)
	if err != nil {
		return err
	}
	s.backend = backend

	playerStore, err := playerstore.NewPlayerStore(&s.configs.playerstoreConfig, s.cacheService, s.serviceCommon.Log)
	if err != nil {
		return err
	}
	s.playerstore = playerStore

	if s.authorizer == nil {
		authClients := &auth.Clients{
			Hallpass: s.hallpassClient,
			Users:    s.usersClient,
		}
		authUtil := auth.NewUtils(&s.configs.authConfig, authClients, s.serviceCommon.Log)
		authorizer, err := auth.NewAuthorizer(s.backend, authUtil, s.serviceCommon.Log)
		if err != nil {
			return err
		}
		s.authorizer = authorizer
	}

	auth, err := auth.NewAuth(s.backend, s.authorizer)
	if err != nil {
		return err
	}
	s.auth = auth

	return nil
}

func (s *service) inject() {
	s.server = api.HTTPServer{
		BaseHTTPServer: service_common.BaseHTTPServer{
			Config: &s.configs.httpConfig.BaseHTTPServerConfig,
			Stats: &service_common.StatSender{
				SubStatter:   s.serviceCommon.Statsd.NewSubStatter("http"),
				ErrorTracker: &s.serviceCommon.ErrorTracker,
			},
			Dims:        &s.serviceCommon.CtxDimensions,
			Log:         s.serviceCommon.Log,
			ElevateKey:  s.serviceCommon.ElevateLogKey,
			Ctxlog:      &s.serviceCommon.Ctxlog,
			OnListen:    s.onListen,
			PanicLogger: s.serviceCommon.PanicLogger,
			XRay:        s.serviceCommon.XRay,
		},
		Backend:           s.backend,
		Config:            &s.configs.httpConfig,
		PlayerStore:       s.playerstore,
		Auth:              s.auth,
		EventBusProcessor: eventbus.NewDefaultEventBusProcessor(),
	}
	s.server.SetupRoutes = s.server.Routes

	s.runner = service_common.ServiceRunner{
		Log: s.serviceCommon.Log,
		Services: []service_common.Service{
			&s.server,
			&s.serviceCommon,
			s.cacheService,
		},
		SigChan:      s.sigChan,
		SignalNotify: signal.Notify,
	}
}

func (s *service) main() {
	if err := s.setup(); err != nil {
		if s.serviceCommon.Log != nil {
			s.serviceCommon.Log.Log("err", err, "Unable to load initial config")
		}
		service_common.SetupLogger.Log("err", err, "Unable to load initial config")
		s.osExit(1)
		return
	}
	s.inject()
	if err := s.runner.Execute(); err != nil {
		s.serviceCommon.Log.Log("err", err, "wait to end finished with an error")
		s.osExit(1)
		return
	}
	s.serviceCommon.Log.Log("Finished Main")
}

func main() {
	instance.main()
}
