// +build integration

package main

import (
	"context"
	"errors"
	"fmt"
	"math/rand"
	"net"
	"net/http"
	"os"
	"strconv"
	"syscall"
	"testing"
	"time"

	"code.justin.tv/twitch-events/meepo/internal/backend"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/log"
	service_common "code.justin.tv/feeds/service-common"
	"code.justin.tv/twitch-events/meepo/clients"
	meepo_errors "code.justin.tv/twitch-events/meepo/errors"
	"code.justin.tv/twitch-events/meepo/internal/clock"
	"code.justin.tv/twitch-events/meepo/internal/mocks"
	"code.justin.tv/twitch-events/meepo/internal/models"
	"code.justin.tv/twitch-events/meepo/internal/stubs"
	"code.justin.tv/twitch-events/meepo/internal/testutil"
	"code.justin.tv/twitch-events/meepo/internal/util"
	"code.justin.tv/twitch-events/meepo/rpc/meepo"
	usermodels "code.justin.tv/web/users-service/models"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

const (
	serverShutdownWaitTime = 3 * time.Second
	testWaitTime           = 20 * time.Second
	adminUserIDs           = "153817938,267332047"
)

func NewDeviceID() string {
	id := rand.Int63()
	return "dev-" + strconv.FormatInt(id, 10)
}

// CreateTestSquad creates a squad with an owner
func CreateTestSquad(ctx context.Context, t *testing.T, client meepo.Meepo, injectables injectables, ownerID string) *meepo.Squad {
	authorizer := (injectables.authorizer).(*mocks.Authorizer)
	authorizer.On("CanCreateSquad", mock.Anything, mock.Anything, mock.Anything).Return(true)

	createSquadReq := &meepo.CreateSquadRequest{
		CallerId: ownerID,
		OwnerId:  ownerID,
	}
	createSquadRes, err := client.CreateSquad(ctx, createSquadReq)
	So(err, ShouldBeNil)
	So(createSquadRes, ShouldNotBeNil)
	So(createSquadRes.Squad, ShouldNotBeNil)

	getSquadByIDReq := &meepo.GetSquadByIDRequest{
		Id: createSquadRes.Squad.Id,
	}
	squadRes, err := client.GetSquadByID(ctx, getSquadByIDReq)
	So(err, ShouldBeNil)
	So(squadRes, ShouldNotBeNil)
	So(squadRes.Squad, ShouldNotBeNil)

	return squadRes.Squad
}

// CreateTestSquadWithMemberAndInvitation creates a squad with an owner, an additional member, and an invitation.
func CreateTestSquadWithMemberAndInvitation(ctx context.Context, t *testing.T, client meepo.Meepo, injectables injectables, ownerID, memberID, recipientID string) *meepo.Squad {
	authorizer := (injectables.authorizer).(*mocks.Authorizer)
	authorizer.On("CanCreateInvitation", mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(true)
	authorizer.On("CanCreateMembership", mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(true)

	// send an invitation from the owner to the recipient, which causes a squad to be created
	createInviteReq := &meepo.CreateInvitationRequest{
		SenderId:    ownerID,
		RecipientId: recipientID,
		CallerId:    ownerID,
	}
	createInviteRes, err := client.CreateInvitation(ctx, createInviteReq)
	So(err, ShouldBeNil)
	So(createInviteRes, ShouldNotBeNil)
	So(createInviteRes.Invitation, ShouldNotBeNil)

	squadID := createInviteRes.Invitation.SquadId

	// create the member in the squad
	createMemberReq := &meepo.CreateMembershipRequest{
		TargetUserId: memberID,
		SquadId:      squadID,
		CallerId:     ownerID,
	}
	createMemberRes, err := client.CreateMembership(ctx, createMemberReq)
	So(err, ShouldBeNil)
	So(createMemberRes, ShouldNotBeNil)
	So(createMemberRes.Membership, ShouldNotBeNil)

	getSquadByIDReq := &meepo.GetSquadByIDRequest{
		Id: squadID,
	}
	squadRes, err := client.GetSquadByID(ctx, getSquadByIDReq)
	So(err, ShouldBeNil)
	So(squadRes, ShouldNotBeNil)
	So(squadRes.Squad, ShouldNotBeNil)
	return squadRes.Squad
}

// CreateTestSquadWithMember creates a squad with an owner and an additional member.
func CreateTestSquadWithMember(ctx context.Context, t *testing.T, client meepo.Meepo, injectables injectables, ownerID, memberID string) *meepo.Squad {
	authorizer := (injectables.authorizer).(*mocks.Authorizer)
	authorizer.On("CanCreateMembership", mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(true)
	authorizer.On("CanCreateSquad", mock.Anything, mock.Anything, mock.Anything).Return(true)

	createSquadReq := &meepo.CreateSquadRequest{
		CallerId: ownerID,
		OwnerId:  ownerID,
	}
	createSquadRes, err := client.CreateSquad(ctx, createSquadReq)
	So(err, ShouldBeNil)
	So(createSquadRes, ShouldNotBeNil)
	So(createSquadRes.Squad, ShouldNotBeNil)

	squadID := createSquadRes.Squad.Id

	// create the member in the squad
	createMemberReq := &meepo.CreateMembershipRequest{
		TargetUserId: memberID,
		SquadId:      squadID,
		CallerId:     ownerID,
	}
	createMemberRes, err := client.CreateMembership(ctx, createMemberReq)
	So(err, ShouldBeNil)
	So(createMemberRes, ShouldNotBeNil)
	So(createMemberRes.Membership, ShouldNotBeNil)

	getSquadByIDReq := &meepo.GetSquadByIDRequest{
		Id: squadID,
	}
	squadRes, err := client.GetSquadByID(ctx, getSquadByIDReq)
	So(err, ShouldBeNil)
	So(squadRes, ShouldNotBeNil)
	So(squadRes.Squad, ShouldNotBeNil)
	return squadRes.Squad
}

// CreateTestSquadWithMember creates a squad with an owner and an additional member.
func CreateTestLiveSquadWithMember(ctx context.Context, t *testing.T, client meepo.Meepo, injectables injectables, ownerID, memberID string) *meepo.Squad {
	squad := CreateTestSquadWithMemberAndInvitation(ctx, t, client, injectables, ownerID, memberID, util.NewUserID())
	So(squad.Status, ShouldEqual, meepo.Squad_PENDING)

	authorizer := (injectables.authorizer).(*mocks.Authorizer)
	authorizer.On("CanUpdateSquad", mock.Anything, mock.Anything, mock.Anything).Return(true)

	// Change the squad's status.  This is expected to update the cache with the updated squad.
	resp, e := client.UpdateSquad(ctx, &meepo.UpdateSquadRequest{
		Id:       squad.Id,
		Status:   meepo.Squad_LIVE,
		CallerId: ownerID,
	})
	So(e, ShouldBeNil)
	So(resp, ShouldNotBeNil)
	So(resp.Squad, ShouldNotBeNil)
	So(resp.Squad.Status, ShouldEqual, meepo.Squad_LIVE)

	return resp.Squad
}

// CreatePendingInvitations creates pending invitation for squad that ownerID owns
func CreatePendingInvitations(ctx context.Context, t *testing.T, client meepo.Meepo, injectables injectables, invitationsToCreate ...*meepo.CreateInvitationRequest) []*meepo.Invitation {
	authorizer := (injectables.authorizer).(*mocks.Authorizer)
	authorizer.On("CanCreateInvitation", mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(true)

	createdInvitations := make([]*meepo.Invitation, 0, len(invitationsToCreate))
	for _, req := range invitationsToCreate {
		res, err := client.CreateInvitation(ctx, req)
		So(err, ShouldBeNil)
		So(res, ShouldNotBeNil)
		So(res.Invitation, ShouldNotBeNil)
		createdInvitations = append(createdInvitations, res.Invitation)
	}

	return createdInvitations
}

func TestGetSquadByChannelID(t *testing.T) {

	injectables := newDefaultInjectables()
	ts := startServer(t, injectables)
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		client := ts.meepoClient
		ctx := ts.ctx

		ownerID := util.NewUserID()
		memberID := util.NewUserID()

		Convey("No channel ID param should throw error", func() {
			req := &meepo.GetSquadByChannelIDRequest{}
			_, err := client.GetSquadByChannelID(ctx, req)
			So(err, ShouldNotBeNil)
		})

		Convey("Should return no squad for channels not in a squad", func() {
			req := &meepo.GetSquadByChannelIDRequest{ChannelId: util.NewUserID()}
			r, err := client.GetSquadByChannelID(ctx, req)
			So(err, ShouldBeNil)
			So(r, ShouldNotBeNil)
			So(r.Squad, ShouldBeNil)
		})

		Convey("Should return a squad for owner of a squad", func() {
			req := &meepo.GetSquadByChannelIDRequest{ChannelId: ownerID}
			r1, e1 := client.GetSquadByChannelID(ctx, req)
			So(e1, ShouldBeNil)
			So(r1, ShouldNotBeNil)
			So(r1.Squad, ShouldBeNil)

			squad := CreateTestSquadWithMemberAndInvitation(ctx, t, client, injectables, ownerID, memberID, util.NewUserID())

			r2, e2 := client.GetSquadByChannelID(ctx, req)
			So(e2, ShouldBeNil)
			So(r2, ShouldNotBeNil)
			So(r2.Squad, ShouldNotBeNil)
			So(r2.Squad.Id, ShouldEqual, squad.Id)
			So(r2.Squad.MemberIds, ShouldResemble, []string{ownerID, memberID})
			So(r2.Squad.OwnerId, ShouldEqual, ownerID)
			So(r2.Squad.Status, ShouldEqual, meepo.Squad_PENDING)
		})

		Convey("Should return a squad for member of a squad", func() {
			req := &meepo.GetSquadByChannelIDRequest{ChannelId: memberID}
			r1, e1 := client.GetSquadByChannelID(ctx, req)
			So(e1, ShouldBeNil)
			So(r1, ShouldNotBeNil)
			So(r1.Squad, ShouldBeNil)

			squad := CreateTestSquadWithMemberAndInvitation(ctx, t, client, injectables, ownerID, memberID, util.NewUserID())

			r2, e2 := client.GetSquadByChannelID(ctx, req)
			So(e2, ShouldBeNil)
			So(r2, ShouldNotBeNil)
			So(r2.Squad, ShouldNotBeNil)
			So(r2.Squad.Id, ShouldEqual, squad.Id)
			So(r2.Squad.MemberIds, ShouldResemble, []string{ownerID, memberID})
			So(r2.Squad.OwnerId, ShouldEqual, ownerID)
			So(r2.Squad.Status, ShouldEqual, meepo.Squad_PENDING)
		})
	})

	ts.onFinish(serverShutdownWaitTime)
}

func TestGetSquadByID(t *testing.T) {

	injectables := newDefaultInjectables()
	ts := startServer(t, injectables, map[string][]byte{
		"meepo.squad_enabled_for_partners": []byte("true"),
	})
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		client := ts.meepoClient
		ctx := ts.ctx

		Convey("No ID param should throw error", func() {
			req := &meepo.GetSquadByIDRequest{}
			_, err := client.GetSquadByID(ctx, req)
			So(err, ShouldNotBeNil)
		})

		Convey("Should return no squad for ID that doesn't belong to a squad", func() {
			req := &meepo.GetSquadByIDRequest{Id: testutil.MustNewID()}
			r, err := client.GetSquadByID(ctx, req)
			So(err, ShouldBeNil)
			So(r, ShouldNotBeNil)
			So(r.Squad, ShouldBeNil)
		})

		Convey("Should return a squad for a valid squad", func() {
			ownerID := util.NewUserID()
			memberID := util.NewUserID()
			squad := CreateTestSquadWithMemberAndInvitation(ctx, t, client, injectables, ownerID, memberID, util.NewUserID())

			req := &meepo.GetSquadByIDRequest{Id: squad.Id}
			r, err := client.GetSquadByID(ctx, req)
			testSquadMemberIDs := []string{ownerID, memberID}

			So(err, ShouldBeNil)
			So(r, ShouldNotBeNil)
			So(r.Squad.Id, ShouldNotBeBlank)
			So(r.Squad.Id, ShouldEqual, squad.Id)
			So(r.Squad.MemberIds, ShouldResemble, testSquadMemberIDs)
			So(r.Squad.OwnerId, ShouldEqual, ownerID)
			So(r.Squad.Status, ShouldEqual, meepo.Squad_PENDING)
		})
	})

	ts.onFinish(serverShutdownWaitTime)
}

func TestCache(t *testing.T) {

	injectables := newDefaultInjectables()
	ts := startServer(t, injectables)
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		client := ts.meepoClient
		ctx := ts.ctx

		// This resets TestSkipCacheRead
		ts.thisInstance.configs.backendConfig.TestSkipCacheRead = false

		Convey("GetSquadByID should return cached squads", func() {
			ownerID := util.NewUserID()
			memberID := util.NewUserID()
			member2ID := util.NewUserID()
			squad := CreateTestSquadWithMemberAndInvitation(ctx, t, client, injectables, ownerID, memberID, util.NewUserID())

			// This bypasses cache invalidation by update the db directly
			ts.thisInstance.datastore.CreateMember(ctx, &models.CreateMemberInput{
				SquadID:  squad.Id,
				MemberID: member2ID,
				Status:   models.MemberStatusActive,
			})

			// This should fetch the cached version
			req := &meepo.GetSquadByIDRequest{Id: squad.Id}
			r1, e1 := client.GetSquadByID(ctx, req)
			So(e1, ShouldBeNil)
			So(r1, ShouldNotBeNil)
			So(r1.Squad.MemberIds, ShouldResemble, []string{ownerID, memberID})

			ts.thisInstance.configs.backendConfig.TestSkipCacheRead = true

			// This should fetch the db version
			r2, e2 := client.GetSquadByID(ctx, req)
			So(e2, ShouldBeNil)
			So(r2, ShouldNotBeNil)
			So(r2.Squad.MemberIds, ShouldResemble, []string{ownerID, memberID, member2ID})
		})

		Convey("GetSquadByChannelID should return cached squads", func() {
			ownerID := util.NewUserID()
			memberID := util.NewUserID()
			member2ID := util.NewUserID()
			squad := CreateTestSquadWithMemberAndInvitation(ctx, t, client, injectables, ownerID, memberID, util.NewUserID())

			// This should cache the empty channel
			reqMember2 := &meepo.GetSquadByChannelIDRequest{ChannelId: member2ID}
			rNoSquad, err := client.GetSquadByChannelID(ctx, reqMember2)
			So(err, ShouldBeNil)
			So(rNoSquad, ShouldNotBeNil)
			So(rNoSquad.Squad, ShouldBeNil)

			// This bypasses cache invalidation by update the db directly
			ts.thisInstance.datastore.CreateMember(ctx, &models.CreateMemberInput{
				SquadID:  squad.Id,
				MemberID: member2ID,
				Status:   models.MemberStatusActive,
			})

			// This should fetch the cached version
			rCachedMember2, err := client.GetSquadByChannelID(ctx, reqMember2)
			So(err, ShouldBeNil)
			So(rCachedMember2, ShouldNotBeNil)
			So(rCachedMember2.Squad, ShouldBeNil)

			ts.thisInstance.configs.backendConfig.TestSkipCacheRead = true

			// This should fetch the db version
			rSkipCacheMember2, err := client.GetSquadByChannelID(ctx, reqMember2)
			So(err, ShouldBeNil)
			So(rSkipCacheMember2, ShouldNotBeNil)
			So(rSkipCacheMember2.Squad, ShouldNotBeNil)
			So(rSkipCacheMember2.Squad.MemberIds, ShouldResemble, []string{ownerID, memberID, member2ID})
		})
	})

	ts.onFinish(serverShutdownWaitTime)
}

func TestCache_WithImmediateRefresh(t *testing.T) {

	injectables := newDefaultInjectables()
	ts := startServer(t, injectables, map[string][]byte{
		"meepo.backend.cache_refresh_duration":    []byte("0s"),
		"meepo.backend.cache_refresh_probability": []byte("1.0"),
	})
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		client := ts.meepoClient
		ctx := ts.ctx

		Convey("GetSquadByID should always refresh", func() {
			ownerID := util.NewUserID()
			memberID := util.NewUserID()
			member2ID := util.NewUserID()
			squad := CreateTestSquadWithMemberAndInvitation(ctx, t, client, injectables, ownerID, memberID, util.NewUserID())

			// This bypasses cache invalidation by update the db directly
			ts.thisInstance.datastore.CreateMember(ctx, &models.CreateMemberInput{
				SquadID:  squad.Id,
				MemberID: member2ID,
				Status:   models.MemberStatusActive,
			})

			// This should fetch the db version
			req := &meepo.GetSquadByIDRequest{Id: squad.Id}
			r, e := client.GetSquadByID(ctx, req)
			So(e, ShouldBeNil)
			So(r, ShouldNotBeNil)
			So(r.Squad.MemberIds, ShouldResemble, []string{ownerID, memberID, member2ID})
		})
	})

	ts.onFinish(serverShutdownWaitTime)
}

func TestCache_Updates(t *testing.T) {

	injectables := newDefaultInjectables()
	ts := startServer(t, injectables)
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	authorizer := (injectables.authorizer).(*mocks.Authorizer)

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		client := ts.meepoClient
		ctx := ts.ctx

		authorizer.On("CanLeaveSquad", mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(true)
		authorizer.On("CanRemoveMember", mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(true)
		authorizer.On("CanUpdateSquad", mock.Anything, mock.Anything, mock.Anything).Return(true)

		Convey("CreateMembership should update the cache with new values", func() {
			ownerID := util.NewUserID()
			memberID := util.NewUserID()
			squad := CreateTestSquadWithMemberAndInvitation(ctx, t, client, injectables, ownerID, memberID, util.NewUserID())
			squadID := squad.Id

			// Invite a member to the squad.  This is expected to update the cache with the updated squad.
			memberID2 := util.NewUserID()

			_, e := client.CreateMembership(ctx, &meepo.CreateMembershipRequest{
				CallerId:     ownerID,
				TargetUserId: memberID2,
				SquadId:      squadID,
			})
			So(e, ShouldBeNil)

			// Verify that we if we refetch the squad, we get the updated squad.
			req := &meepo.GetSquadByIDRequest{Id: squadID}
			r, e2 := client.GetSquadByID(ctx, req)
			So(e2, ShouldBeNil)
			So(r, ShouldNotBeNil)
			So(r.Squad, ShouldNotBeNil)
			So(r.Squad.MemberIds, ShouldHaveLength, 3)
			So(r.Squad.MemberIds, ShouldContain, memberID2)
		})

		Convey("UpdateSquad should update the cache with new values", func() {
			ownerID := util.NewUserID()
			memberID := util.NewUserID()
			squad := CreateTestSquadWithMemberAndInvitation(ctx, t, client, injectables, ownerID, memberID, util.NewUserID())
			So(squad.Status, ShouldEqual, meepo.Squad_PENDING)

			// Change the squad's status.  This is expected to update the cache with the updated squad.
			_, e := client.UpdateSquad(ctx, &meepo.UpdateSquadRequest{
				Id:       squad.Id,
				Status:   meepo.Squad_LIVE,
				CallerId: ownerID,
			})
			So(e, ShouldBeNil)

			// Verify that we if we refetch the squad, we get the updated squad.
			req := &meepo.GetSquadByIDRequest{Id: squad.Id}
			r, e2 := client.GetSquadByID(ctx, req)
			So(e2, ShouldBeNil)
			So(r, ShouldNotBeNil)
			So(r.Squad, ShouldNotBeNil)
			So(r.Squad.Status, ShouldEqual, meepo.Squad_LIVE)
		})

		Convey("LeaveSquad should update the cache with new values", func() {
			ownerID := util.NewUserID()
			memberID := util.NewUserID()
			squad := CreateTestSquadWithMemberAndInvitation(ctx, t, client, injectables, ownerID, memberID, util.NewUserID())
			So(squad, ShouldNotBeNil)
			squadID := squad.Id
			So(squadID, ShouldNotBeEmpty)

			// Remove a member.  This is expected to update the cache.
			_, e := client.LeaveSquad(ctx, &meepo.LeaveSquadRequest{
				MemberId: memberID,
				CallerId: memberID,
				SquadId:  squadID,
			})
			So(e, ShouldBeNil)

			Convey("Fetching the squad by ID should return the updated squad", func() {
				r, err := client.GetSquadByID(ctx, &meepo.GetSquadByIDRequest{
					Id: squadID,
				})

				So(err, ShouldBeNil)
				So(r, ShouldNotBeNil)
				So(r.Squad, ShouldNotBeNil)
				So(r.Squad.MemberIds, ShouldHaveLength, 1)
				So(r.Squad.MemberIds[0], ShouldEqual, ownerID)
			})

			Convey("Fetching the squad by the remaining member should return the updated squad", func() {
				r, err := client.GetSquadByChannelID(ctx, &meepo.GetSquadByChannelIDRequest{
					ChannelId: ownerID,
				})

				So(err, ShouldBeNil)
				So(r, ShouldNotBeNil)
				So(r.Squad, ShouldNotBeNil)
				So(r.Squad.MemberIds, ShouldHaveLength, 1)
				So(r.Squad.MemberIds[0], ShouldEqual, ownerID)
			})

			Convey("Fetching the squad by the removed member should return no squad", func() {
				r, err := client.GetSquadByChannelID(ctx, &meepo.GetSquadByChannelIDRequest{
					ChannelId: memberID,
				})

				So(err, ShouldBeNil)
				So(r, ShouldNotBeNil)
				So(r.Squad, ShouldBeNil)
			})

			Convey("Ending the squad by the removed member should return no squad", func() {
				_, e := client.LeaveSquad(ctx, &meepo.LeaveSquadRequest{
					MemberId: ownerID,
					CallerId: ownerID,
					SquadId:  squadID,
				})
				So(e, ShouldBeNil)

				r, err := client.GetSquadByID(ctx, &meepo.GetSquadByIDRequest{
					Id: squadID,
				})

				So(err, ShouldBeNil)
				So(r, ShouldNotBeNil)
				So(r.Squad, ShouldNotBeNil)
				So(r.Squad.MemberIds, ShouldHaveLength, 0)
				So(r.Squad.Status, ShouldEqual, meepo.Squad_ENDED)
			})
		})

		Convey("Ending a squad should update the cache with new values", func() {
			ownerID := util.NewUserID()
			memberID := util.NewUserID()
			squad := CreateTestSquadWithMemberAndInvitation(ctx, t, client, injectables, ownerID, memberID, util.NewUserID())
			So(squad, ShouldNotBeNil)
			squadID := squad.Id
			So(squadID, ShouldNotBeEmpty)

			// Remove all members, and end the squad.  This is expected to update the cache.
			for _, mid := range []string{ownerID, memberID} {
				_, e := client.LeaveSquad(ctx, &meepo.LeaveSquadRequest{
					MemberId: mid,
					CallerId: mid,
					SquadId:  squadID,
				})
				So(e, ShouldBeNil)
			}

			Convey("Fetching the squad by ID should return the updated squad", func() {
				r, err := client.GetSquadByID(ctx, &meepo.GetSquadByIDRequest{
					Id: squadID,
				})

				So(err, ShouldBeNil)
				So(r, ShouldNotBeNil)
				So(r.Squad, ShouldNotBeNil)
				So(r.Squad.MemberIds, ShouldHaveLength, 0)
				So(r.Squad.Status, ShouldEqual, meepo.Squad_ENDED)
			})

			Convey("Fetching the squad by the removed member should return no squad", func() {
				r, err := client.GetSquadByChannelID(ctx, &meepo.GetSquadByChannelIDRequest{
					ChannelId: memberID,
				})

				So(err, ShouldBeNil)
				So(r, ShouldNotBeNil)
				So(r.Squad, ShouldBeNil)
			})
		})

		Convey("RemoveMember should update the cache with new values", func() {
			ownerID := util.NewUserID()
			memberID := util.NewUserID()
			squad := CreateTestSquadWithMemberAndInvitation(ctx, t, client, injectables, ownerID, memberID, util.NewUserID())
			So(squad, ShouldNotBeNil)
			squadID := squad.Id
			So(squadID, ShouldNotBeEmpty)

			// Remove a member.  This is expected to update the cache.
			_, e := client.RemoveMember(ctx, &meepo.RemoveMemberRequest{
				MemberId: memberID,
				CallerId: ownerID,
				SquadId:  squadID,
			})
			So(e, ShouldBeNil)

			Convey("Fetching the squad by ID should return the updated squad", func() {
				r, err := client.GetSquadByID(ctx, &meepo.GetSquadByIDRequest{
					Id: squadID,
				})

				So(err, ShouldBeNil)
				So(r, ShouldNotBeNil)
				So(r.Squad, ShouldNotBeNil)
				So(r.Squad.MemberIds, ShouldHaveLength, 1)
				So(r.Squad.MemberIds[0], ShouldEqual, ownerID)
			})

			Convey("Fetching the squad by the remaining member should return the updated squad", func() {
				r, err := client.GetSquadByChannelID(ctx, &meepo.GetSquadByChannelIDRequest{
					ChannelId: ownerID,
				})

				So(err, ShouldBeNil)
				So(r, ShouldNotBeNil)
				So(r.Squad, ShouldNotBeNil)
				So(r.Squad.MemberIds, ShouldHaveLength, 1)
				So(r.Squad.MemberIds[0], ShouldEqual, ownerID)
			})

			Convey("Fetching the squad by the removed member should return no squad", func() {
				r, err := client.GetSquadByChannelID(ctx, &meepo.GetSquadByChannelIDRequest{
					ChannelId: memberID,
				})

				So(err, ShouldBeNil)
				So(r, ShouldNotBeNil)
				So(r.Squad, ShouldBeNil)
			})
		})
	})

	ts.onFinish(serverShutdownWaitTime)
}

func TestCreateSquad(t *testing.T) {

	injectables := newDefaultInjectables()
	ts := startServer(t, injectables)
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	authorizer := (injectables.authorizer).(*mocks.Authorizer)
	pubsubClient := (injectables.pubsubClient).(*testutil.PubsubMockClient)

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		client := ts.meepoClient
		ctx := ts.ctx

		authorizer.On("CanCreateSquad", mock.Anything, mock.Anything, mock.Anything).Return(true)

		Convey("No owner ID param should throw error", func() {
			req := &meepo.CreateSquadRequest{CallerId: util.NewUserID()}
			_, err := client.CreateSquad(ctx, req)
			So(err, ShouldNotBeNil)
		})

		Convey("No squad ID param should throw error", func() {
			req := &meepo.CreateSquadRequest{OwnerId: util.NewUserID()}
			_, err := client.CreateSquad(ctx, req)
			So(err, ShouldNotBeNil)
		})

		Convey("Should create a squad with the owner for valid input", func() {
			ownerID := util.NewUserID()
			memberIDs := []string{ownerID}

			req := &meepo.CreateSquadRequest{OwnerId: ownerID, CallerId: ownerID}
			r, err := client.CreateSquad(ctx, req)
			So(err, ShouldBeNil)
			So(r, ShouldNotBeNil)
			So(r.Squad, ShouldNotBeNil)
			So(r.Squad.OwnerId, ShouldEqual, ownerID)
			So(r.Squad.Status, ShouldEqual, meepo.Squad_PENDING)
			So(r.Squad.MemberIds, ShouldResemble, memberIDs)

			pubsubClient.AssertSquadUpdatePublished(t, r.Squad.Id, models.SquadStatusPending, memberIDs)

			Convey("Should throw an error if the owner is already in a squad", func() {
				_, err = client.CreateSquad(ctx, req)
				So(err, ShouldNotBeNil)
			})
		})
	})

	ts.onFinish(serverShutdownWaitTime)
}

func TestCreateMembership(t *testing.T) {

	injectables := newDefaultInjectables()
	ts := startServer(t, injectables)
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}
	pubsubClient := (injectables.pubsubClient).(*testutil.PubsubMockClient)

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		client := ts.meepoClient
		ctx := ts.ctx

		Convey("No target user ID param should throw error", func() {
			req := &meepo.CreateMembershipRequest{SquadId: util.NewUserID(), CallerId: util.NewUserID()}
			_, err := client.CreateMembership(ctx, req)
			So(err, ShouldNotBeNil)
		})

		Convey("No caller ID param should throw error", func() {
			req := &meepo.CreateMembershipRequest{TargetUserId: util.NewUserID(), SquadId: util.NewUserID()}
			_, err := client.CreateMembership(ctx, req)
			So(err, ShouldNotBeNil)
		})

		Convey("No squad ID param should throw error", func() {
			req := &meepo.CreateMembershipRequest{TargetUserId: util.NewUserID(), CallerId: util.NewUserID()}
			_, err := client.CreateMembership(ctx, req)
			So(err, ShouldNotBeNil)
		})

		Convey("Should add the user to the squad for valid input", func() {
			ownerID := util.NewUserID()
			memberID := util.NewUserID()
			newMemberID := util.NewUserID()
			squad := CreateTestSquadWithMemberAndInvitation(ctx, t, client, injectables, ownerID, memberID, util.NewUserID())
			squadID := squad.Id
			memberIDs := []string{ownerID, memberID, newMemberID}

			req := &meepo.CreateMembershipRequest{TargetUserId: newMemberID, SquadId: squadID, CallerId: ownerID}
			r, err := client.CreateMembership(ctx, req)
			So(err, ShouldBeNil)
			So(r, ShouldNotBeNil)
			So(r.Membership, ShouldNotBeNil)
			So(r.Membership.MemberId, ShouldEqual, newMemberID)
			So(r.Membership.Status, ShouldEqual, meepo.Member_ACTIVE)
			So(r.Membership.SquadId, ShouldNotBeBlank)

			pubsubClient.AssertSquadUpdatePublished(t, squadID, models.SquadStatusPending, memberIDs)

			getSquadByIDReq := &meepo.GetSquadByIDRequest{
				Id: squadID,
			}
			s, err := client.GetSquadByID(ctx, getSquadByIDReq)
			So(err, ShouldBeNil)
			So(s, ShouldNotBeNil)
			So(s.Squad, ShouldNotBeNil)
			So(s.Squad.Id, ShouldEqual, squadID)
			So(s.Squad.OwnerId, ShouldEqual, ownerID)
			So(s.Squad.Status, ShouldEqual, meepo.Squad_PENDING)
			So(s.Squad.MemberIds, ShouldResemble, memberIDs)

			Convey("Should throw an error if the squad is full", func() {
				userID := util.NewUserID()
				req := &meepo.CreateMembershipRequest{
					SquadId:      squadID,
					TargetUserId: userID,
					CallerId:     ownerID,
				}
				_, err := client.CreateMembership(ctx, req)
				So(err, ShouldNotBeNil)
			})
		})

		Convey("Should throw an error if the recipient is already in the squad", func() {
			ownerID := util.NewUserID()
			memberID := util.NewUserID()
			squad := CreateTestSquadWithMemberAndInvitation(ctx, t, client, injectables, ownerID, memberID, util.NewUserID())
			squadID := squad.Id
			createMembershipReq := &meepo.CreateMembershipRequest{
				SquadId:      squadID,
				TargetUserId: memberID,
				CallerId:     ownerID,
			}
			_, err := client.CreateMembership(ctx, createMembershipReq)
			So(err, ShouldNotBeNil)
		})
	})

	ts.onFinish(serverShutdownWaitTime)
}

func TestCreateInvitation(t *testing.T) {

	adminUserID := util.NewUserID()
	dartAllowlistedUserID := util.NewUserID()

	injectables := newDefaultInjectables()
	ts := startServer(t, injectables, map[string][]byte{
		"meepo.squad_enabled_for_affiliates":  []byte("false"),
		"meepo.admin_users":                   []byte(adminUserID),
		"meepo.invite_notification_whitelist": []byte(dartAllowlistedUserID),
	})
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	authorizer := (injectables.authorizer).(*mocks.Authorizer)
	pubsubClient := (injectables.pubsubClient).(*testutil.PubsubMockClient)
	ripleyClient := (injectables.ripleyClient).(*mocks.RipleyClient)
	usersClient := (injectables.usersClient).(*mocks.UsersClient)
	dartClient := (injectables.dartClient).(*mocks.NotifierClient)

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		client := ts.meepoClient
		ctx := ts.ctx

		authorizer.On("CanCreateInvitation", mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(true)
		authorizer.On("CanRejectInvitation", mock.Anything, mock.Anything, mock.Anything).Return(true)
		authorizer.On("CanGetPendingInvitationsByRecipientID", mock.Anything, mock.Anything, mock.Anything).Return(true)
		authorizer.On("CanGetInvitationsBySquadID", mock.Anything, mock.Anything, mock.Anything).Return(true)

		usersClient.On("ValidateUser", mock.Anything, mock.Anything).Return(true, nil, nil)
		isSubadmin := false
		usersClient.On("GetUserByID", mock.Anything, mock.Anything).Return(&usermodels.Properties{
			Subadmin: &isSubadmin,
		}, nil)

		Convey("No sender ID param should throw error", func() {
			req := &meepo.CreateInvitationRequest{RecipientId: util.NewUserID(), CallerId: util.NewUserID()}
			_, err := client.CreateInvitation(ctx, req)
			So(err, ShouldNotBeNil)
		})

		Convey("No recipient ID param should throw error", func() {
			req := &meepo.CreateInvitationRequest{SenderId: util.NewUserID(), CallerId: util.NewUserID()}
			_, err := client.CreateInvitation(ctx, req)
			So(err, ShouldNotBeNil)
		})

		Convey("No caller ID param should throw error", func() {
			req := &meepo.CreateInvitationRequest{SenderId: util.NewUserID(), RecipientId: util.NewUserID()}
			_, err := client.CreateInvitation(ctx, req)
			So(err, ShouldNotBeNil)
		})

		Convey("Should send a notification to dart if the user is whitelisted", func() {
			senderID := util.NewUserID()
			notWhitelistedUserID := util.NewUserID()

			dartClient.On("SendInviteNotification", mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(nil, nil)
			ripleyClient.On("GetUserPayoutType", mock.Anything, senderID).Return(
				&clients.UserPayoutType{
					IsPartner: true,
				}, nil)
			ripleyClient.On("GetUserPayoutType", mock.Anything, dartAllowlistedUserID).Once().Return(
				&clients.UserPayoutType{
					IsPartner: true,
				}, nil)
			ripleyClient.On("GetUserPayoutType", mock.Anything, notWhitelistedUserID).Once().Return(
				&clients.UserPayoutType{
					IsPartner: true,
				}, nil)

			// Should send a dart notification if the user is in the invite_notification_whitelist
			pendingInvitation, err := client.CreateInvitation(
				ctx,
				&meepo.CreateInvitationRequest{
					SenderId:    senderID,
					CallerId:    senderID,
					RecipientId: dartAllowlistedUserID,
				},
			)
			So(err, ShouldBeNil)
			So(pendingInvitation, ShouldNotBeNil)

			dartClient.AssertCalled(t, "SendInviteNotification", mock.Anything, senderID, dartAllowlistedUserID, pendingInvitation.Invitation.Id)

			// Should NOT send a dart notification if the user is not in the invite_notification_whitelist
			pendingInvitation, err = client.CreateInvitation(
				ctx,
				&meepo.CreateInvitationRequest{
					SenderId:    senderID,
					CallerId:    senderID,
					RecipientId: notWhitelistedUserID,
				},
			)
			So(err, ShouldBeNil)
			So(pendingInvitation, ShouldNotBeNil)

			dartClient.AssertNotCalled(t, "SendInviteNotification", mock.Anything, senderID, notWhitelistedUserID, pendingInvitation.Invitation.Id)
		})

		Convey("All users are partners", func() {
			ripleyClient.On("GetUserPayoutType", mock.Anything, mock.Anything).Return(
				&clients.UserPayoutType{
					IsPartner: true,
				}, nil)

			Convey("Should throw an error if the recipient is already in the squad", func() {
				ownerID := util.NewUserID()
				memberID := util.NewUserID()
				CreateTestSquadWithMemberAndInvitation(ctx, t, client, injectables, ownerID, memberID, util.NewUserID())

				req := &meepo.CreateInvitationRequest{
					SenderId:    ownerID,
					RecipientId: memberID,
					CallerId:    ownerID,
				}
				_, err := client.CreateInvitation(ctx, req)
				So(err, ShouldNotBeNil)
			})

			Convey("Should throw an error if the sender is not the squad owner", func() {
				ownerID := util.NewUserID()
				memberID := util.NewUserID()
				recipientID := util.NewUserID()
				CreateTestSquadWithMemberAndInvitation(ctx, t, client, injectables, ownerID, memberID, util.NewUserID())

				newCreateInvitationReq := &meepo.CreateInvitationRequest{
					SenderId:    memberID,
					RecipientId: recipientID,
					CallerId:    memberID,
				}
				_, err := client.CreateInvitation(ctx, newCreateInvitationReq)
				So(err, ShouldNotBeNil)
			})

			Convey("Should create a squad if the sender doesn't have a squad", func() {
				senderID := util.NewUserID()
				recipientID := util.NewUserID()
				memberIDs := []string{senderID}

				createInvitationReq := &meepo.CreateInvitationRequest{
					SenderId:    senderID,
					RecipientId: recipientID,
					CallerId:    senderID,
				}
				r, err := client.CreateInvitation(ctx, createInvitationReq)
				So(err, ShouldBeNil)
				So(r, ShouldNotBeNil)
				So(r.Invitation, ShouldNotBeNil)
				So(r.Invitation.SenderId, ShouldEqual, senderID)
				So(r.Invitation.RecipientId, ShouldEqual, recipientID)
				So(r.Invitation.Status, ShouldEqual, meepo.Invitation_PENDING)
				So(r.Invitation.SquadId, ShouldNotBeBlank)
				squadID := r.Invitation.SquadId

				// Check that pubsub messages are sent to the recipient and the sender with the new invitation
				pubsubClient.AssertIncomingSquadInvitesPublished(t, recipientID, []testutil.ExpectedPubsubInvitations{
					{SenderID: senderID, RecipientID: recipientID, Status: models.InvitationStatusPending},
				})
				pubsubClient.AssertSquadUpdatePublished(t, squadID, models.SquadStatusPending, []string{senderID})

				getSquadByIDReq := &meepo.GetSquadByIDRequest{
					Id: squadID,
				}
				s, err := client.GetSquadByID(ctx, getSquadByIDReq)
				So(err, ShouldBeNil)
				So(s, ShouldNotBeNil)
				So(s.Squad, ShouldNotBeNil)
				So(s.Squad.Id, ShouldEqual, squadID)
				So(s.Squad.OwnerId, ShouldEqual, senderID)
				So(s.Squad.Status, ShouldEqual, meepo.Squad_PENDING)
				So(s.Squad.MemberIds, ShouldResemble, memberIDs)

				getInvitationBySquadIDReq := &meepo.GetInvitationsBySquadIDRequest{
					SquadId:  squadID,
					CallerId: senderID,
					Status:   meepo.Invitation_PENDING,
				}
				i, err := client.GetInvitationsBySquadID(ctx, getInvitationBySquadIDReq)
				So(err, ShouldBeNil)
				So(i, ShouldNotBeNil)
				So(i.Invitations, ShouldHaveLength, 1)
				So(i.Invitations[0], ShouldNotBeNil)
				So(i.Invitations[0], ShouldResemble, r.Invitation)

				Convey("Should throw an error if an invitation to the recipient already exists", func() {
					_, err := client.CreateInvitation(ctx, createInvitationReq)
					So(err, ShouldNotBeNil)
				})
			})

			Convey("Should send the recipient an invitation to the squad if the sender's squad is not full", func() {
				ownerID := util.NewUserID()
				memberID := util.NewUserID()
				memberIDs := []string{ownerID, memberID}
				recipientID := util.NewUserID()
				squad := CreateTestSquadWithMember(ctx, t, client, injectables, ownerID, memberID)
				squadID := squad.Id

				createInvitationReq := &meepo.CreateInvitationRequest{
					SenderId:    ownerID,
					RecipientId: recipientID,
					CallerId:    ownerID,
				}
				r, err := client.CreateInvitation(ctx, createInvitationReq)
				So(err, ShouldBeNil)
				So(r, ShouldNotBeNil)
				So(r.Invitation, ShouldNotBeNil)
				So(r.Invitation.SenderId, ShouldEqual, ownerID)
				So(r.Invitation.RecipientId, ShouldEqual, recipientID)
				So(r.Invitation.Status, ShouldEqual, meepo.Invitation_PENDING)
				So(r.Invitation.SquadId, ShouldEqual, squadID)

				// Check that pubsub messages are sent to the recipient and the squad owner with the new invitation
				pubsubClient.AssertIncomingSquadInvitesPublished(t, recipientID, []testutil.ExpectedPubsubInvitations{
					{SenderID: ownerID, RecipientID: recipientID, Status: models.InvitationStatusPending},
				})
				pubsubClient.AssertSquadUpdatePublished(t, squadID, models.SquadStatusPending, memberIDs)

				getSquadByIDReq := &meepo.GetSquadByIDRequest{
					Id: squadID,
				}
				s, err := client.GetSquadByID(ctx, getSquadByIDReq)
				So(err, ShouldBeNil)
				So(s, ShouldNotBeNil)
				So(s.Squad, ShouldNotBeNil)
				So(s.Squad.Id, ShouldEqual, squadID)
				So(s.Squad.OwnerId, ShouldEqual, ownerID)
				So(s.Squad.Status, ShouldEqual, meepo.Squad_PENDING)
				So(s.Squad.MemberIds, ShouldResemble, memberIDs)

				getInvitationBySquadIDReq := &meepo.GetInvitationsBySquadIDRequest{
					SquadId:  squadID,
					CallerId: ownerID,
					Status:   meepo.Invitation_PENDING,
				}
				i, err := client.GetInvitationsBySquadID(ctx, getInvitationBySquadIDReq)
				So(err, ShouldBeNil)
				So(i, ShouldNotBeNil)
				So(i.Invitations, ShouldHaveLength, 1)
				So(i.Invitations[0], ShouldNotBeNil)
				So(i.Invitations[0], ShouldResemble, r.Invitation)

				Convey("Should throw an error if the sender's squad is full", func() {
					recipientID := util.NewUserID()

					// add one more invitation so that the squad is full
					req := &meepo.CreateInvitationRequest{
						SenderId:    ownerID,
						RecipientId: recipientID,
						CallerId:    ownerID,
					}
					r, err := client.CreateInvitation(ctx, req)
					So(err, ShouldBeNil)
					So(r, ShouldNotBeNil)
					So(r.Invitation, ShouldNotBeNil)

					recipientID2 := util.NewUserID()

					req2 := &meepo.CreateInvitationRequest{
						SenderId:    ownerID,
						RecipientId: recipientID2,
						CallerId:    ownerID,
					}
					_, err = client.CreateInvitation(ctx, req2)
					So(err, ShouldNotBeNil)
				})
			})

			Convey("Should delete the existing rejected invitation from the squad to the recipient", func() {
				ownerID := util.NewUserID()
				recipientID := util.NewUserID()
				squad := CreateTestSquad(ctx, t, client, injectables, ownerID)
				squadID := squad.Id

				createInvitationReq := &meepo.CreateInvitationRequest{
					SenderId:    ownerID,
					RecipientId: recipientID,
					CallerId:    ownerID,
				}
				createInvitationRes, err := client.CreateInvitation(ctx, createInvitationReq)
				So(err, ShouldBeNil)
				So(createInvitationRes, ShouldNotBeNil)
				So(createInvitationRes.Invitation, ShouldNotBeNil)

				rejectInvitationReq := &meepo.RejectInvitationRequest{
					InvitationId: createInvitationRes.Invitation.GetId(),
					CallerId:     recipientID,
				}
				rejectInvitationRes, err := client.RejectInvitation(ctx, rejectInvitationReq)
				So(err, ShouldBeNil)
				So(rejectInvitationRes, ShouldNotBeNil)
				So(rejectInvitationRes.Invitation, ShouldNotBeNil)

				getInvitationBySquadIDReq := &meepo.GetInvitationsBySquadIDRequest{
					SquadId:  squadID,
					CallerId: ownerID,
					Status:   meepo.Invitation_REJECTED,
				}
				getInvitationsRes, err := client.GetInvitationsBySquadID(ctx, getInvitationBySquadIDReq)
				So(err, ShouldBeNil)
				So(getInvitationsRes, ShouldNotBeNil)
				So(getInvitationsRes.Invitations, ShouldHaveLength, 1)
				So(getInvitationsRes.Invitations[0], ShouldNotBeNil)
				So(getInvitationsRes.Invitations[0], ShouldResemble, rejectInvitationRes.Invitation)

				createInvitationRes, err = client.CreateInvitation(ctx, createInvitationReq)
				So(err, ShouldBeNil)
				So(createInvitationRes, ShouldNotBeNil)
				So(createInvitationRes.Invitation, ShouldNotBeNil)

				getInvitationsRes, err = client.GetInvitationsBySquadID(ctx, getInvitationBySquadIDReq)
				So(err, ShouldBeNil)
				So(getInvitationsRes, ShouldNotBeNil)
				So(getInvitationsRes.Invitations, ShouldHaveLength, 0)

				getInvitationBySquadIDReq2 := &meepo.GetInvitationsBySquadIDRequest{
					SquadId:  squadID,
					CallerId: ownerID,
					Status:   meepo.Invitation_PENDING,
				}
				getInvitationsRes, err = client.GetInvitationsBySquadID(ctx, getInvitationBySquadIDReq2)
				So(err, ShouldBeNil)
				So(getInvitationsRes, ShouldNotBeNil)
				So(getInvitationsRes.Invitations, ShouldHaveLength, 1)
				So(getInvitationsRes.Invitations[0], ShouldNotBeNil)
				So(getInvitationsRes.Invitations[0], ShouldResemble, createInvitationRes.Invitation)
			})

			Convey("Should not reject pending invitations for recipient if recipient has <= 10 incoming, pending invitations", func() {
				recipientID := util.NewUserID()
				pendingInvitationIDs := make([]string, 0, 10)

				// Verify recipient initially has 0 invitations
				response, err := client.GetIncomingInvitationsByChannelID(ctx, &meepo.GetIncomingInvitationsByChannelIDRequest{
					ChannelId: recipientID,
					CallerId:  recipientID,
				})
				So(err, ShouldBeNil)
				So(response.Invitations, ShouldBeNil)

				// Create max pending invitations
				for i := 0; i < backend.MaxPendingInvitationsPerChannel; i++ {
					ownerID := util.NewUserID()
					pendingInvitation, err := client.CreateInvitation(
						ctx,
						&meepo.CreateInvitationRequest{
							SenderId:    ownerID,
							CallerId:    ownerID,
							RecipientId: recipientID,
						},
					)
					So(err, ShouldBeNil)
					So(pendingInvitation, ShouldNotBeNil)
					pendingInvitationIDs = append(pendingInvitationIDs, pendingInvitation.Invitation.Id)
				}

				// Verify recipient has 10 invitations
				response, err = client.GetIncomingInvitationsByChannelID(ctx, &meepo.GetIncomingInvitationsByChannelIDRequest{
					ChannelId: recipientID,
					CallerId:  recipientID,
				})
				So(err, ShouldBeNil)
				So(response.Invitations, ShouldHaveLength, backend.MaxPendingInvitationsPerChannel)

				// Verify that all 10 incoming invitations are ones created here
				for _, invitation := range response.Invitations {
					So(invitation.Id, ShouldBeIn, pendingInvitationIDs)
				}
			})

			Convey("Should reject oldest pending invitation for recipient if recipient has > 10 incoming, pending invitations", func() {
				recipientID := util.NewUserID()
				var oldestPendingInvitation *meepo.Invitation

				// Verify recipient initially has 0 invitations
				response, err := client.GetIncomingInvitationsByChannelID(ctx, &meepo.GetIncomingInvitationsByChannelIDRequest{
					ChannelId: recipientID,
					CallerId:  recipientID,
				})
				So(err, ShouldBeNil)
				So(response.Invitations, ShouldBeNil)

				// Create max pending invitations, and save the oldest pending invitation
				for i := 0; i < backend.MaxPendingInvitationsPerChannel; i++ {
					ownerID := util.NewUserID()
					pendingInvitation, err := client.CreateInvitation(
						ctx,
						&meepo.CreateInvitationRequest{
							SenderId:    ownerID,
							CallerId:    ownerID,
							RecipientId: recipientID,
						},
					)
					So(err, ShouldBeNil)
					So(pendingInvitation, ShouldNotBeNil)

					// save the oldest pending invitation
					if i == 0 {
						oldestPendingInvitation = pendingInvitation.Invitation
					}
				}

				// Verify recipient initially has 10 invitations
				response, err = client.GetIncomingInvitationsByChannelID(ctx, &meepo.GetIncomingInvitationsByChannelIDRequest{
					ChannelId: recipientID,
					CallerId:  recipientID,
				})
				So(err, ShouldBeNil)
				So(response.Invitations, ShouldHaveLength, backend.MaxPendingInvitationsPerChannel)
				pubsubClient.AssertSquadUpdatePublishedTimes(t, oldestPendingInvitation.SquadId, models.SquadStatusPending, []string{oldestPendingInvitation.SenderId}, 1)

				// Create 11th invitation, which should trigger older pending to be rejected
				newestOwnerID := util.NewUserID()
				resp, err := client.CreateInvitation(ctx, &meepo.CreateInvitationRequest{
					SenderId:    newestOwnerID,
					CallerId:    newestOwnerID,
					RecipientId: recipientID,
				})
				So(err, ShouldBeNil)
				So(resp, ShouldNotBeNil)
				pubsubClient.AssertSquadUpdatePublishedTimes(t, oldestPendingInvitation.SquadId, models.SquadStatusPending, []string{oldestPendingInvitation.SenderId}, 2)

				newestPendingInvitation := resp.Invitation

				// Verify recipient still has 10 invitations
				pendingInvitations, err := client.GetIncomingInvitationsByChannelID(ctx, &meepo.GetIncomingInvitationsByChannelIDRequest{
					ChannelId: recipientID,
					CallerId:  recipientID,
				})
				So(err, ShouldBeNil)
				So(pendingInvitations.Invitations, ShouldHaveLength, backend.MaxPendingInvitationsPerChannel)

				// Verify oldest pending invitation has been rejected
				pendingInvitationIDs := make([]string, 0, 10)
				for _, invitation := range pendingInvitations.Invitations {
					pendingInvitationIDs = append(pendingInvitationIDs, invitation.Id)
				}
				So(oldestPendingInvitation.Id, ShouldNotBeIn, pendingInvitationIDs)
				So(newestPendingInvitation.Id, ShouldBeIn, pendingInvitationIDs)

				// TODO: Validate that oldestPendingInvitation now has status rejected
			})
		})
	})

	ts.onFinish(serverShutdownWaitTime)
}

func TestUpdateSquad(t *testing.T) {

	injectables := newDefaultInjectables()
	ts := startServer(t, injectables)
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	pubsubClient := (injectables.pubsubClient).(*testutil.PubsubMockClient)
	channelStatePublisher := (injectables.channelStatePublisherClient).(*mocks.ChannelStatePublisher)
	authorizer := (injectables.authorizer).(*mocks.Authorizer)

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		client := ts.meepoClient
		ctx := ts.ctx

		authorizer.On("CanLeaveSquad", mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(true)
		authorizer.On("CanUpdateSquad", mock.Anything, mock.Anything, mock.Anything).Return(true)

		Convey("No ID param should throw error", func() {
			req := &meepo.UpdateSquadRequest{CallerId: util.NewUserID(), Status: meepo.Squad_LIVE}
			_, err := client.UpdateSquad(ctx, req)
			So(err, ShouldNotBeNil)
		})

		Convey("No status param should throw error", func() {
			req := &meepo.UpdateSquadRequest{Id: testutil.MustNewID(), CallerId: util.NewUserID()}
			_, err := client.UpdateSquad(ctx, req)
			So(err, ShouldNotBeNil)
		})

		Convey("No caller ID param should throw error", func() {
			req := &meepo.UpdateSquadRequest{Id: testutil.MustNewID(), Status: meepo.Squad_LIVE}
			_, err := client.UpdateSquad(ctx, req)
			So(err, ShouldNotBeNil)
		})

		Convey("Should throw an error for ID that doesn't belong to a squad", func() {
			req := &meepo.UpdateSquadRequest{Id: testutil.MustNewID(), Status: meepo.Squad_ENDED, CallerId: util.NewUserID()}
			_, err := client.UpdateSquad(ctx, req)
			So(err, ShouldNotBeNil)
		})

		Convey("Should throw an error for an unknown status", func() {
			ownerID := util.NewUserID()
			memberID := util.NewUserID()
			squad := CreateTestSquadWithMember(ctx, t, client, injectables, ownerID, memberID)
			req := &meepo.UpdateSquadRequest{Id: squad.Id, Status: meepo.Squad_UNKNOWN, CallerId: ownerID}
			_, err := client.UpdateSquad(ctx, req)
			So(err, ShouldNotBeNil)
		})

		Convey("Should throw an error for ENDED status", func() {
			ownerID := util.NewUserID()
			memberID := util.NewUserID()
			squad := CreateTestSquadWithMember(ctx, t, client, injectables, ownerID, memberID)
			req := &meepo.UpdateSquadRequest{Id: squad.Id, Status: meepo.Squad_ENDED, CallerId: ownerID}
			_, err := client.UpdateSquad(ctx, req)
			So(err, ShouldNotBeNil)
		})

		Convey("Should throw an error for PENDING status", func() {
			ownerID := util.NewUserID()
			memberID := util.NewUserID()
			squad := CreateTestSquadWithMember(ctx, t, client, injectables, ownerID, memberID)
			req := &meepo.UpdateSquadRequest{Id: squad.Id, Status: meepo.Squad_PENDING, CallerId: ownerID}
			_, err := client.UpdateSquad(ctx, req)
			So(err, ShouldNotBeNil)
		})

		Convey("Should update a squad for a valid squad and input", func() {
			ownerID := util.NewUserID()
			memberID := util.NewUserID()
			squad := CreateTestSquadWithMemberAndInvitation(ctx, t, client, injectables, ownerID, memberID, util.NewUserID())
			memberIDs := []string{ownerID, memberID}
			squadID := squad.Id

			status := meepo.Squad_LIVE
			req := &meepo.UpdateSquadRequest{Id: squadID, Status: status, CallerId: ownerID}
			r, err := client.UpdateSquad(ctx, req)

			So(err, ShouldBeNil)
			So(r, ShouldNotBeNil)
			So(r.Squad, ShouldNotBeNil)
			So(r.Squad.Id, ShouldEqual, squadID)
			So(r.Squad.MemberIds, ShouldResemble, memberIDs)
			So(r.Squad.OwnerId, ShouldEqual, squad.OwnerId)
			So(r.Squad.Status, ShouldEqual, status)

			pubsubClient.AssertSquadUpdatePublished(t, r.Squad.Id, models.SquadStatusLive, memberIDs)

			// Verify that we publish that the members are part of a live squad.
			channelStatePublisher.AssertCalled(t, "PublishChannelsInLiveSquad", mock.Anything, squadID, memberIDs)

			Convey("Should throw an error for squad in LIVE status", func() {
				req := &meepo.UpdateSquadRequest{Id: squad.Id, Status: meepo.Squad_LIVE, CallerId: ownerID}
				_, err := client.UpdateSquad(ctx, req)
				So(err, ShouldNotBeNil)
			})

			Convey("Should throw an error for squad in ENDED status", func() {
				leaveReq := &meepo.LeaveSquadRequest{
					MemberId: memberID,
					CallerId: memberID,
					SquadId:  squadID,
				}
				_, err := client.LeaveSquad(ctx, leaveReq)
				So(err, ShouldBeNil)

				leaveReq2 := &meepo.LeaveSquadRequest{
					MemberId: ownerID,
					CallerId: ownerID,
					SquadId:  squadID,
				}
				_, err = client.LeaveSquad(ctx, leaveReq2)
				So(err, ShouldBeNil)

				getReq := &meepo.GetSquadByIDRequest{Id: squadID}
				r, err := client.GetSquadByID(ctx, getReq)
				So(err, ShouldBeNil)
				So(r, ShouldNotBeNil)
				So(r.Squad, ShouldNotBeNil)
				So(r.Squad.Status, ShouldEqual, meepo.Squad_ENDED)

				req := &meepo.UpdateSquadRequest{Id: squad.Id, Status: meepo.Squad_LIVE, CallerId: ownerID}
				_, err = client.UpdateSquad(ctx, req)
				So(err, ShouldNotBeNil)
			})
		})

		Convey("Should throw an error if there are less than two members in the squad", func() {
			ownerID := util.NewUserID()
			memberID := util.NewUserID()
			squad := CreateTestSquadWithMember(ctx, t, client, injectables, ownerID, memberID)
			squadID := squad.Id

			leaveReq := &meepo.LeaveSquadRequest{
				MemberId: memberID,
				CallerId: memberID,
				SquadId:  squadID,
			}
			_, err := client.LeaveSquad(ctx, leaveReq)
			So(err, ShouldBeNil)

			status := meepo.Squad_LIVE
			req := &meepo.UpdateSquadRequest{Id: squadID, Status: status, CallerId: ownerID}
			_, err = client.UpdateSquad(ctx, req)

			So(err, ShouldNotBeNil)
		})
	})

	ts.onFinish(serverShutdownWaitTime)
}

func TestLeaveSquad(t *testing.T) {

	adminUser := util.NewUserID()
	injectables := newDefaultInjectables()
	ts := startServer(t, injectables, map[string][]byte{
		"meepo.admin_users": []byte(adminUser),
	})
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	pubsubClient := (injectables.pubsubClient).(*testutil.PubsubMockClient)
	channelStatePublisher := (injectables.channelStatePublisherClient).(*mocks.ChannelStatePublisher)
	authorizer := (injectables.authorizer).(*mocks.Authorizer)

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		client := ts.meepoClient
		ctx := ts.ctx

		authorizer.On("CanLeaveSquad", mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(true)
		authorizer.On("CanGetInvitationsBySquadID", mock.Anything, mock.Anything, mock.Anything).Return(true)

		Convey("No member ID param should throw error", func() {
			req := &meepo.LeaveSquadRequest{SquadId: testutil.MustNewID(), CallerId: util.NewUserID()}
			_, err := client.LeaveSquad(ctx, req)
			So(err, ShouldNotBeNil)
		})

		Convey("No squad ID param should throw error", func() {
			memberID := util.NewUserID()
			req := &meepo.LeaveSquadRequest{MemberId: memberID, CallerId: memberID}
			_, err := client.LeaveSquad(ctx, req)
			So(err, ShouldNotBeNil)
		})

		Convey("No caller ID param should throw error", func() {
			req := &meepo.LeaveSquadRequest{MemberId: util.NewUserID(), SquadId: testutil.MustNewID()}
			_, err := client.LeaveSquad(ctx, req)
			So(err, ShouldNotBeNil)
		})

		Convey("Should throw an error if the recipient is not in the squad", func() {
			ownerID := util.NewUserID()
			memberID := util.NewUserID()
			squad := CreateTestSquadWithMemberAndInvitation(ctx, t, client, injectables, ownerID, memberID, util.NewUserID())
			squadID := squad.Id

			channelID := util.NewUserID()
			req := &meepo.LeaveSquadRequest{
				MemberId: channelID,
				CallerId: channelID,
				SquadId:  squadID,
			}
			_, err := client.LeaveSquad(ctx, req)
			So(err, ShouldNotBeNil)
		})

		Convey("Should remove the member if the member was not the owner of the squad", func() {
			ownerID := util.NewUserID()
			memberID := util.NewUserID()
			recipientID := util.NewUserID()
			squad := CreateTestSquadWithMemberAndInvitation(ctx, t, client, injectables, ownerID, memberID, recipientID)
			squadID := squad.Id
			memberIDs := []string{ownerID}
			pubsubClient.AssertIncomingSquadInvitesPublished(t, recipientID, []testutil.ExpectedPubsubInvitations{
				{SenderID: ownerID, RecipientID: recipientID, Status: models.InvitationStatusPending},
			})

			pubsubClient.ClearCalls()
			req := &meepo.LeaveSquadRequest{
				MemberId: memberID,
				CallerId: memberID,
				SquadId:  squadID,
			}
			r, err := client.LeaveSquad(ctx, req)
			So(err, ShouldBeNil)
			So(r, ShouldNotBeNil)
			So(r.Squad, ShouldNotBeNil)
			So(r.Squad.Id, ShouldEqual, squadID)
			So(r.Squad.OwnerId, ShouldEqual, ownerID)
			So(r.Squad.Status, ShouldEqual, meepo.Squad_PENDING)
			So(r.Squad.MemberIds, ShouldResemble, memberIDs)

			pubsubClient.AssertChannelNotInSquadPublished(t, memberID)
			pubsubClient.AssertSquadUpdatePublished(t, squadID, models.SquadStatusPending, memberIDs)
			pubsubClient.AssertIncomingSquadInvitesNotPublished(t, recipientID, []testutil.ExpectedPubsubInvitations{})

			Convey("Should remove the member, delete all invitations, and end the squad if there are no more members left", func() {

				req := &meepo.LeaveSquadRequest{
					MemberId: ownerID,
					CallerId: ownerID,
					SquadId:  squadID,
				}
				r, err := client.LeaveSquad(ctx, req)
				So(err, ShouldBeNil)
				So(r, ShouldNotBeNil)
				So(r.Squad, ShouldNotBeNil)
				So(r.Squad.Id, ShouldEqual, squadID)
				So(r.Squad.OwnerId, ShouldBeBlank)
				So(r.Squad.Status, ShouldEqual, meepo.Squad_ENDED)
				So(r.Squad.MemberIds, ShouldBeEmpty)

				pubsubClient.AssertChannelNotInSquadPublished(t, ownerID)
				pubsubClient.AssertSquadUpdatePublished(t, squadID, models.SquadStatusEnded, []string{})

				getInvitationsReq := &meepo.GetInvitationsBySquadIDRequest{
					SquadId:  squadID,
					CallerId: adminUser,
					Status:   meepo.Invitation_PENDING,
				}
				i, err := client.GetInvitationsBySquadID(ctx, getInvitationsReq)
				So(err, ShouldBeNil)
				So(i, ShouldNotBeNil)
				So(i.Invitations, ShouldBeEmpty)
			})

		})

		Convey("Should remove the member and update the owner if the member was the owner of the squad", func() {
			ownerID := util.NewUserID()
			memberID := util.NewUserID()
			recipientID := util.NewUserID()
			squad := CreateTestSquadWithMemberAndInvitation(ctx, t, client, injectables, ownerID, memberID, recipientID)
			squadID := squad.Id
			memberIDs := []string{memberID}
			pubsubClient.AssertIncomingSquadInvitesPublished(t, recipientID, []testutil.ExpectedPubsubInvitations{
				{SenderID: ownerID, RecipientID: recipientID, Status: models.InvitationStatusPending},
			})

			req := &meepo.LeaveSquadRequest{
				MemberId: ownerID,
				CallerId: ownerID,
				SquadId:  squadID,
			}
			r, err := client.LeaveSquad(ctx, req)
			So(err, ShouldBeNil)
			So(r, ShouldNotBeNil)
			So(r.Squad, ShouldNotBeNil)
			So(r.Squad.Id, ShouldEqual, squadID)
			So(r.Squad.OwnerId, ShouldEqual, memberID)
			So(r.Squad.Status, ShouldEqual, meepo.Squad_PENDING)
			So(r.Squad.MemberIds, ShouldResemble, memberIDs)

			pubsubClient.AssertChannelNotInSquadPublished(t, ownerID)
			pubsubClient.AssertSquadUpdatePublished(t, squadID, models.SquadStatusPending, memberIDs)
			pubsubClient.AssertIncomingSquadInvitesPublished(t, recipientID, []testutil.ExpectedPubsubInvitations{})

			getInvitationsReq := &meepo.GetInvitationsBySquadIDRequest{
				SquadId:  squadID,
				CallerId: adminUser,
				Status:   meepo.Invitation_PENDING,
			}
			i, err := client.GetInvitationsBySquadID(ctx, getInvitationsReq)
			So(err, ShouldBeNil)
			So(i, ShouldNotBeNil)
			So(i.Invitations, ShouldBeEmpty)
		})

		Convey("When a member leaves, we should publish a channel update if the squad was LIVE", func() {
			ownerID := util.NewUserID()
			memberID := util.NewUserID()
			squad := CreateTestLiveSquadWithMember(ctx, t, client, injectables, ownerID, memberID)
			So(squad, ShouldNotBeNil)
			So(squad.Status, ShouldEqual, meepo.Squad_LIVE)
			squadID := squad.Id

			req := &meepo.LeaveSquadRequest{
				MemberId: memberID,
				CallerId: memberID,
				SquadId:  squadID,
			}
			_, err := client.LeaveSquad(ctx, req)
			So(err, ShouldBeNil)

			channelStatePublisher.AssertCalled(t, "PublishChannelOutOfLiveSquad", mock.Anything, memberID)
			channelStatePublisher.AssertNotCalled(t, "PublishChannelOutOfLiveSquad", mock.Anything, ownerID)

			req = &meepo.LeaveSquadRequest{
				MemberId: ownerID,
				CallerId: ownerID,
				SquadId:  squadID,
			}
			_, err = client.LeaveSquad(ctx, req)
			So(err, ShouldBeNil)
			channelStatePublisher.AssertCalled(t, "PublishChannelOutOfLiveSquad", mock.Anything, ownerID)

			res, err := client.GetSquadByID(ctx, &meepo.GetSquadByIDRequest{
				Id: squadID,
			})
			So(err, ShouldBeNil)
			So(res.GetSquad().Status, ShouldEqual, meepo.Squad_ENDED)
		})

		Convey("When a member leaves, we should NOT publish a channel update if the squad was NOT LIVE", func() {
			ownerID := util.NewUserID()
			memberID := util.NewUserID()
			squad := CreateTestSquadWithMember(ctx, t, client, injectables, ownerID, memberID)
			So(squad, ShouldNotBeNil)
			So(squad.Status, ShouldEqual, meepo.Squad_PENDING)
			squadID := squad.Id

			req := &meepo.LeaveSquadRequest{
				MemberId: memberID,
				CallerId: memberID,
				SquadId:  squadID,
			}
			_, err := client.LeaveSquad(ctx, req)
			So(err, ShouldBeNil)

			channelStatePublisher.AssertNotCalled(t, "PublishChannelOutOfLiveSquad", mock.Anything, memberID)
			channelStatePublisher.AssertNotCalled(t, "PublishChannelOutOfLiveSquad", mock.Anything, ownerID)
		})
	})

	ts.onFinish(serverShutdownWaitTime)
}

func TestRemoveMember(t *testing.T) {

	adminUser := util.NewUserID()
	injectables := newDefaultInjectables()
	ts := startServer(t, injectables, map[string][]byte{
		"meepo.admin_users": []byte(adminUser),
	})
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	authorizer := (injectables.authorizer).(*mocks.Authorizer)
	pubsubClient := (injectables.pubsubClient).(*testutil.PubsubMockClient)
	ripleyClient := (injectables.ripleyClient).(*mocks.RipleyClient)
	usersClient := (injectables.usersClient).(*mocks.UsersClient)
	channelStatePublisher := (injectables.channelStatePublisherClient).(*mocks.ChannelStatePublisher)

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		client := ts.meepoClient
		ctx := ts.ctx

		authorizer.On("CanGetInvitationsBySquadID", mock.Anything, mock.Anything, mock.Anything).Return(true)
		authorizer.On("CanRemoveMember", mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(true)
		ripleyClient.On("GetUserPayoutType", mock.Anything, mock.Anything).Return(
			&clients.UserPayoutType{
				IsPartner: true,
			}, nil)
		usersClient.On("ValidateUser", mock.Anything, mock.Anything).Return(true, nil, nil)
		authorizer.On("CanLeaveSquad", mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(true)

		Convey("No member ID param should throw error", func() {
			req := &meepo.RemoveMemberRequest{SquadId: testutil.MustNewID(), CallerId: util.NewUserID()}
			_, err := client.RemoveMember(ctx, req)
			So(err, ShouldNotBeNil)
		})

		Convey("No caller ID param should throw error", func() {
			req := &meepo.RemoveMemberRequest{MemberId: util.NewUserID(), SquadId: testutil.MustNewID()}
			_, err := client.RemoveMember(ctx, req)
			So(err, ShouldNotBeNil)
		})

		Convey("No squad ID param should throw error", func() {
			req := &meepo.RemoveMemberRequest{MemberId: util.NewUserID(), CallerId: util.NewUserID()}
			_, err := client.RemoveMember(ctx, req)
			So(err, ShouldNotBeNil)
		})

		Convey("Should throw an error if the recipient is not in the squad", func() {
			ownerID := util.NewUserID()
			memberID := util.NewUserID()
			squad := CreateTestSquadWithMemberAndInvitation(ctx, t, client, injectables, ownerID, memberID, util.NewUserID())
			squadID := squad.Id

			channelID := util.NewUserID()
			req := &meepo.RemoveMemberRequest{
				MemberId: channelID,
				SquadId:  squadID,
				CallerId: ownerID,
			}
			_, err := client.RemoveMember(ctx, req)
			So(err, ShouldNotBeNil)
		})

		Convey("Should remove the member if the member was not the owner of the squad", func() {
			ownerID := util.NewUserID()
			memberID := util.NewUserID()
			recipientID := util.NewUserID()
			squad := CreateTestSquadWithMemberAndInvitation(ctx, t, client, injectables, ownerID, memberID, recipientID)
			squadID := squad.Id
			memberIDs := []string{ownerID}

			req := &meepo.RemoveMemberRequest{
				MemberId: memberID,
				SquadId:  squadID,
				CallerId: ownerID,
			}
			r, err := client.RemoveMember(ctx, req)
			So(err, ShouldBeNil)
			So(r, ShouldNotBeNil)
			So(r.Squad, ShouldNotBeNil)
			So(r.Squad.Id, ShouldEqual, squadID)
			So(r.Squad.OwnerId, ShouldEqual, ownerID)
			So(r.Squad.Status, ShouldEqual, meepo.Squad_PENDING)
			So(r.Squad.MemberIds, ShouldResemble, memberIDs)

			pubsubClient.AssertChannelNotInSquadPublished(t, memberID)
			pubsubClient.AssertSquadUpdatePublished(t, squadID, models.SquadStatusPending, memberIDs)
			pubsubClient.AssertIncomingSquadInvitesNotPublished(t, recipientID, []testutil.ExpectedPubsubInvitations{})
		})

		Convey("Should remove the member and update the owner if the member was the owner of the squad", func() {
			ownerID := util.NewUserID()
			memberID := util.NewUserID()
			recipientID := util.NewUserID()
			squad := CreateTestSquadWithMemberAndInvitation(ctx, t, client, injectables, ownerID, memberID, recipientID)
			squadID := squad.Id
			memberIDs := []string{memberID}

			pubsubClient.AssertIncomingSquadInvitesPublished(t, recipientID, []testutil.ExpectedPubsubInvitations{
				{SenderID: ownerID, RecipientID: recipientID, Status: models.InvitationStatusPending},
			})
			req := &meepo.RemoveMemberRequest{
				MemberId: ownerID,
				SquadId:  squadID,
				CallerId: ownerID,
			}
			r, err := client.RemoveMember(ctx, req)
			So(err, ShouldBeNil)
			So(r, ShouldNotBeNil)
			So(r.Squad, ShouldNotBeNil)
			So(r.Squad.Id, ShouldEqual, squadID)
			So(r.Squad.OwnerId, ShouldEqual, memberID)
			So(r.Squad.Status, ShouldEqual, meepo.Squad_PENDING)
			So(r.Squad.MemberIds, ShouldResemble, memberIDs)

			pubsubClient.AssertChannelNotInSquadPublished(t, ownerID)
			pubsubClient.AssertSquadUpdatePublished(t, squadID, models.SquadStatusPending, memberIDs)
			pubsubClient.AssertIncomingSquadInvitesPublished(t, recipientID, []testutil.ExpectedPubsubInvitations{})

			getInvitationsReq := &meepo.GetInvitationsBySquadIDRequest{
				SquadId:  squadID,
				CallerId: adminUser,
				Status:   meepo.Invitation_PENDING,
			}
			i, err := client.GetInvitationsBySquadID(ctx, getInvitationsReq)
			So(err, ShouldBeNil)
			So(i, ShouldNotBeNil)
			So(i.Invitations, ShouldBeEmpty)

			Convey("Should remove the member, delete all invitations, and end the squad if there are no more members left", func() {
				recipientID := util.NewUserID()

				createInvitationReq := &meepo.CreateInvitationRequest{
					SenderId:    memberID,
					RecipientId: recipientID,
					CallerId:    memberID,
				}
				_, err := client.CreateInvitation(ctx, createInvitationReq)
				So(err, ShouldBeNil)

				req := &meepo.RemoveMemberRequest{
					MemberId: memberID,
					SquadId:  squadID,
					CallerId: memberID,
				}
				r, err := client.RemoveMember(ctx, req)
				So(err, ShouldBeNil)
				So(r, ShouldNotBeNil)
				So(r.Squad, ShouldNotBeNil)
				So(r.Squad.Id, ShouldEqual, squadID)
				So(r.Squad.OwnerId, ShouldBeBlank)
				So(r.Squad.Status, ShouldEqual, meepo.Squad_ENDED)
				So(r.Squad.MemberIds, ShouldBeEmpty)

				pubsubClient.AssertChannelNotInSquadPublished(t, memberID)
				pubsubClient.AssertSquadUpdatePublished(t, squadID, models.SquadStatusEnded, []string{})

				getInvitationsReq := &meepo.GetInvitationsBySquadIDRequest{
					SquadId:  squadID,
					CallerId: adminUser,
					Status:   meepo.Invitation_PENDING,
				}
				i, err := client.GetInvitationsBySquadID(ctx, getInvitationsReq)
				So(err, ShouldBeNil)
				So(i, ShouldNotBeNil)
				So(i.Invitations, ShouldBeEmpty)
			})
		})

		Convey("Removing a member should publish a channel update if the squad was LIVE", func() {
			ownerID := util.NewUserID()
			memberID := util.NewUserID()
			squad := CreateTestLiveSquadWithMember(ctx, t, client, injectables, ownerID, memberID)
			So(squad, ShouldNotBeNil)
			So(squad.Status, ShouldEqual, meepo.Squad_LIVE)
			squadID := squad.Id

			req := &meepo.LeaveSquadRequest{
				MemberId: memberID,
				CallerId: memberID,
				SquadId:  squadID,
			}
			_, err := client.LeaveSquad(ctx, req)
			So(err, ShouldBeNil)

			channelStatePublisher.AssertCalled(t, "PublishChannelOutOfLiveSquad", mock.Anything, memberID)
			channelStatePublisher.AssertNotCalled(t, "PublishChannelOutOfLiveSquad", mock.Anything, ownerID)
		})

		Convey("Removing a member  should NOT publish a channel update if the squad was NOT LIVE", func() {
			ownerID := util.NewUserID()
			memberID := util.NewUserID()
			squad := CreateTestSquadWithMember(ctx, t, client, injectables, ownerID, memberID)
			So(squad, ShouldNotBeNil)
			So(squad.Status, ShouldEqual, meepo.Squad_PENDING)
			squadID := squad.Id

			req := &meepo.LeaveSquadRequest{
				MemberId: memberID,
				CallerId: memberID,
				SquadId:  squadID,
			}
			_, err := client.LeaveSquad(ctx, req)
			So(err, ShouldBeNil)

			channelStatePublisher.AssertNotCalled(t, "PublishChannelOutOfLiveSquad", mock.Anything, memberID)
			channelStatePublisher.AssertNotCalled(t, "PublishChannelOutOfLiveSquad", mock.Anything, ownerID)
		})
	})

	ts.onFinish(serverShutdownWaitTime)
}

func TestGetInvitationsBySquadID(t *testing.T) {

	injectables := newDefaultInjectables()
	ts := startServer(t, injectables)
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	authorizer := (injectables.authorizer).(*mocks.Authorizer)
	ripleyClient := (injectables.ripleyClient).(*mocks.RipleyClient)
	usersClient := (injectables.usersClient).(*mocks.UsersClient)

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		client := ts.meepoClient
		ctx := ts.ctx

		authorizer.On("CanCreateInvitation", mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(true)
		authorizer.On("CanGetInvitationsBySquadID", mock.Anything, mock.Anything, mock.Anything).Return(true)
		authorizer.On("CanAcceptInvitation", mock.Anything, mock.Anything, mock.Anything).Return(true)
		ripleyClient.On("GetUserPayoutType", mock.Anything, mock.Anything).Return(
			&clients.UserPayoutType{
				IsPartner: true,
			}, nil)
		usersClient.On("ValidateUser", mock.Anything, mock.Anything).Return(true, nil, nil)

		Convey("No squad ID param should throw error", func() {
			req := &meepo.GetInvitationsBySquadIDRequest{CallerId: util.NewUserID(), Status: meepo.Invitation_PENDING}
			_, err := client.GetInvitationsBySquadID(ctx, req)
			So(err, ShouldNotBeNil)
		})

		Convey("No user ID param should throw error", func() {
			req := &meepo.GetInvitationsBySquadIDRequest{SquadId: testutil.MustNewID(), Status: meepo.Invitation_PENDING}
			_, err := client.GetInvitationsBySquadID(ctx, req)
			So(err, ShouldNotBeNil)
		})

		Convey("No status param should throw error", func() {
			req := &meepo.GetInvitationsBySquadIDRequest{SquadId: testutil.MustNewID(), CallerId: util.NewUserID()}
			_, err := client.GetInvitationsBySquadID(ctx, req)
			So(err, ShouldNotBeNil)
		})

		Convey("Should return no invitations for a squad that doesn't have any invitations", func() {
			ownerID := util.NewUserID()
			memberID := util.NewUserID()
			squad := CreateTestSquadWithMember(ctx, t, client, injectables, ownerID, memberID)

			req := &meepo.GetInvitationsBySquadIDRequest{SquadId: squad.Id, CallerId: ownerID, Status: meepo.Invitation_PENDING}
			r, err := client.GetInvitationsBySquadID(ctx, req)
			So(err, ShouldBeNil)
			So(r, ShouldNotBeNil)
			So(r.Invitations, ShouldBeNil)
		})

		Convey("Should return invitations of the correct status for a valid squad with invitations", func() {
			ownerID := util.NewUserID()
			memberID := util.NewUserID()
			squad := CreateTestSquadWithMember(ctx, t, client, injectables, ownerID, memberID)
			squadID := squad.Id

			recipientID := util.NewUserID()
			recipientID2 := util.NewUserID()

			createInvitationReq := &meepo.CreateInvitationRequest{
				SenderId:    ownerID,
				RecipientId: recipientID,
				CallerId:    ownerID,
			}
			_, err := client.CreateInvitation(ctx, createInvitationReq)
			So(err, ShouldBeNil)

			createInvitationReq2 := &meepo.CreateInvitationRequest{
				SenderId:    ownerID,
				RecipientId: recipientID2,
				CallerId:    ownerID,
			}
			_, err = client.CreateInvitation(ctx, createInvitationReq2)
			So(err, ShouldBeNil)

			req := &meepo.GetInvitationsBySquadIDRequest{SquadId: squadID, CallerId: memberID, Status: meepo.Invitation_PENDING}
			r, err := client.GetInvitationsBySquadID(ctx, req)
			recipientIDs := []string{recipientID, recipientID2}

			So(err, ShouldBeNil)
			So(r, ShouldNotBeNil)
			So(r.Invitations, ShouldHaveLength, 2)
			So(r.Invitations[0], ShouldNotBeNil)
			So(r.Invitations[0].SenderId, ShouldEqual, ownerID)
			So(r.Invitations[0].RecipientId, ShouldBeIn, recipientIDs)
			So(r.Invitations[0].SquadId, ShouldEqual, squadID)
			So(r.Invitations[0].Status, ShouldEqual, meepo.Invitation_PENDING)
			So(r.Invitations[1].SenderId, ShouldEqual, ownerID)
			So(r.Invitations[1].RecipientId, ShouldBeIn, recipientIDs)
			So(r.Invitations[1].SquadId, ShouldEqual, squadID)
			So(r.Invitations[1].Status, ShouldEqual, meepo.Invitation_PENDING)
		})

		Convey("Should return invitations sorted in ascending order by updated_at", func() {
			ownerID := util.NewUserID()
			squad := CreateTestSquad(ctx, t, client, injectables, ownerID)
			squadID := squad.Id

			recipientID := util.NewUserID()
			recipientID2 := util.NewUserID()
			recipientID3 := util.NewUserID()

			createInvitationReq := &meepo.CreateInvitationRequest{
				SenderId:    ownerID,
				RecipientId: recipientID,
				CallerId:    ownerID,
			}
			_, err := client.CreateInvitation(ctx, createInvitationReq)
			So(err, ShouldBeNil)

			createInvitationReq2 := &meepo.CreateInvitationRequest{
				SenderId:    ownerID,
				RecipientId: recipientID2,
				CallerId:    ownerID,
			}
			_, err = client.CreateInvitation(ctx, createInvitationReq2)
			So(err, ShouldBeNil)

			createInvitationReq3 := &meepo.CreateInvitationRequest{
				SenderId:    ownerID,
				RecipientId: recipientID3,
				CallerId:    ownerID,
			}
			_, err = client.CreateInvitation(ctx, createInvitationReq3)
			So(err, ShouldBeNil)

			req := &meepo.GetInvitationsBySquadIDRequest{SquadId: squadID, CallerId: ownerID, Status: meepo.Invitation_PENDING}
			r, err := client.GetInvitationsBySquadID(ctx, req)

			So(err, ShouldBeNil)
			So(r, ShouldNotBeNil)
			So(r.Invitations, ShouldHaveLength, 3)
			So(
				time.Unix(r.Invitations[0].UpdatedAt.GetSeconds(), int64(r.Invitations[0].UpdatedAt.GetNanos())),
				ShouldHappenBefore,
				time.Unix(r.Invitations[1].UpdatedAt.GetSeconds(), int64(r.Invitations[1].UpdatedAt.GetNanos())),
			)
			So(
				time.Unix(r.Invitations[1].UpdatedAt.GetSeconds(), int64(r.Invitations[1].UpdatedAt.GetNanos())),
				ShouldHappenBefore,
				time.Unix(r.Invitations[2].UpdatedAt.GetSeconds(), int64(r.Invitations[2].UpdatedAt.GetNanos())),
			)
		})

		Convey("Should not return invitations of another status for a valid squad", func() {
			ownerID := util.NewUserID()
			memberID := util.NewUserID()
			squad := CreateTestSquadWithMember(ctx, t, client, injectables, ownerID, memberID)
			squadID := squad.Id
			recipientID := util.NewUserID()

			createInvitationReq := &meepo.CreateInvitationRequest{
				SenderId:    ownerID,
				RecipientId: recipientID,
				CallerId:    ownerID,
			}
			i, err := client.CreateInvitation(ctx, createInvitationReq)
			So(err, ShouldBeNil)
			So(i, ShouldNotBeNil)
			So(i.Invitation, ShouldNotBeNil)

			acceptReq := &meepo.AcceptInvitationRequest{
				InvitationId: i.Invitation.Id,
				CallerId:     recipientID,
			}
			_, err = client.AcceptInvitation(ctx, acceptReq)
			So(err, ShouldBeNil)

			req := &meepo.GetInvitationsBySquadIDRequest{SquadId: squadID, CallerId: memberID, Status: meepo.Invitation_PENDING}
			r, err := client.GetInvitationsBySquadID(ctx, req)
			So(err, ShouldBeNil)
			So(r, ShouldNotBeNil)
			So(r.Invitations, ShouldHaveLength, 0)
		})
	})

	ts.onFinish(serverShutdownWaitTime)
}

func TestGetIncomingInvitationsByChannelID(t *testing.T) {

	injectables := newDefaultInjectables()
	ts := startServer(t, injectables)
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	authorizer := (injectables.authorizer).(*mocks.Authorizer)
	ripleyClient := (injectables.ripleyClient).(*mocks.RipleyClient)
	usersClient := (injectables.usersClient).(*mocks.UsersClient)

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		client := ts.meepoClient
		ctx := ts.ctx

		authorizer.On("CanAcceptInvitation", mock.Anything, mock.Anything, mock.Anything).Return(true)
		authorizer.On("CanGetPendingInvitationsByRecipientID", mock.Anything, mock.Anything, mock.Anything).Return(true)
		ripleyClient.On("GetUserPayoutType", mock.Anything, mock.Anything).Return(
			&clients.UserPayoutType{
				IsPartner: true,
			}, nil)
		usersClient.On("ValidateUser", mock.Anything, mock.Anything).Return(true, nil, nil)

		Convey("No channel ID param should throw error", func() {
			req := &meepo.GetIncomingInvitationsByChannelIDRequest{CallerId: util.NewUserID()}
			_, err := client.GetIncomingInvitationsByChannelID(ctx, req)
			So(err, ShouldNotBeNil)
		})

		Convey("No caller ID param should throw error", func() {
			req := &meepo.GetIncomingInvitationsByChannelIDRequest{ChannelId: util.NewUserID()}
			_, err := client.GetIncomingInvitationsByChannelID(ctx, req)
			So(err, ShouldNotBeNil)
		})

		Convey("Should return no invitations for a channel that doesn't have any invitations", func() {
			channelID := util.NewUserID()
			req := &meepo.GetIncomingInvitationsByChannelIDRequest{ChannelId: channelID, CallerId: channelID}
			r, err := client.GetIncomingInvitationsByChannelID(ctx, req)
			So(err, ShouldBeNil)
			So(r, ShouldNotBeNil)
			So(r.Invitations, ShouldBeEmpty)
		})

		Convey("Should return only pending invitations for a channel with invitations", func() {
			recipientID := util.NewUserID()

			ownerID := util.NewUserID()
			CreateTestSquad(ctx, t, client, injectables, ownerID)

			ownerID2 := util.NewUserID()
			CreateTestSquad(ctx, t, client, injectables, ownerID2)

			invitations := CreatePendingInvitations(
				ctx, t, client, injectables,
				&meepo.CreateInvitationRequest{
					SenderId:    ownerID,
					RecipientId: recipientID,
					CallerId:    ownerID,
				},
				&meepo.CreateInvitationRequest{
					SenderId:    ownerID2,
					RecipientId: recipientID,
					CallerId:    ownerID2,
				})

			So(invitations, ShouldHaveLength, 2)
			i1 := invitations[0]
			i2 := invitations[1]

			acceptReq := &meepo.AcceptInvitationRequest{
				InvitationId: i2.Id,
				CallerId:     recipientID,
			}
			_, err := client.AcceptInvitation(ctx, acceptReq)
			So(err, ShouldBeNil)

			req := &meepo.GetIncomingInvitationsByChannelIDRequest{ChannelId: recipientID, CallerId: recipientID}
			r, err := client.GetIncomingInvitationsByChannelID(ctx, req)
			So(err, ShouldBeNil)
			So(r, ShouldNotBeNil)
			So(r.Invitations, ShouldHaveLength, 1)
			So(r.Invitations[0], ShouldNotBeNil)
			i1.NetworkType = meepo.Invitation_IN_NETWORK
			So(r.Invitations[0], ShouldResemble, i1)
			So(r.Invitations[0].Status, ShouldEqual, meepo.Invitation_PENDING)
		})
	})

	ts.onFinish(serverShutdownWaitTime)
}

func TestGetIncomingInvitationsByChannelIDWithNetworkType(t *testing.T) {

	injectables := newDefaultInjectables()

	authorizer := (injectables.authorizer).(*mocks.Authorizer)
	followsClient := &mocks.FollowsClient{}
	friendshipClient := &mocks.FriendshipClient{}
	rosterClient := &mocks.RosterClient{}

	injectables.followsClient = followsClient
	injectables.friendshipClient = friendshipClient
	injectables.rosterClient = rosterClient

	ts := startServer(t, injectables)
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		client := ts.meepoClient
		ctx := ts.ctx

		authorizer.On("CanGetPendingInvitationsByRecipientID", mock.Anything, mock.Anything, mock.Anything).Return(true)

		Convey("Should hydrate with out of network if clients return error", func() {
			recipientID := util.NewUserID()
			ownerID := util.NewUserID()

			client.UpdateInvitePolicyByChannelID(ctx, &meepo.UpdateInvitePolicyByChannelIDRequest{
				ChannelId:    recipientID,
				CallerId:     recipientID,
				InvitePolicy: meepo.InvitePolicy_ALL,
			})

			// recipient is following ownerID
			followsClient.On("IsFollowing", mock.Anything, recipientID, ownerID).Maybe().Return(false, errors.New("new error"))
			friendshipClient.On("IsFriend", mock.Anything, recipientID, ownerID).Maybe().Return(false, errors.New("new error"))
			rosterClient.On("IsTeammate", mock.Anything, recipientID, ownerID).Maybe().Return(false, errors.New("new error"))

			CreateTestSquad(ctx, t, client, injectables, ownerID)
			invitations := CreatePendingInvitations(
				ctx, t, client, injectables,
				&meepo.CreateInvitationRequest{
					SenderId:    ownerID,
					RecipientId: recipientID,
					CallerId:    ownerID,
				})
			So(invitations, ShouldHaveLength, 1)
			invitation := invitations[0]
			invitation.NetworkType = meepo.Invitation_OUT_OF_NETWORK

			req := &meepo.GetIncomingInvitationsByChannelIDRequest{ChannelId: recipientID, CallerId: recipientID}
			r, err := client.GetIncomingInvitationsByChannelID(ctx, req)
			So(err, ShouldBeNil)
			So(r, ShouldNotBeNil)
			So(r.Invitations, ShouldHaveLength, 1)
			So(r.Invitations[0], ShouldNotBeNil)
			So(r.Invitations[0], ShouldResemble, invitation)
			So(r.Invitations[0].Status, ShouldEqual, meepo.Invitation_PENDING)
		})

		Convey("Should hydrate with correct network information for pending invitations", func() {
			recipientID := util.NewUserID()

			client.UpdateInvitePolicyByChannelID(ctx, &meepo.UpdateInvitePolicyByChannelIDRequest{
				ChannelId:    recipientID,
				CallerId:     recipientID,
				InvitePolicy: meepo.InvitePolicy_ALL,
			})

			ownerID := util.NewUserID()
			CreateTestSquad(ctx, t, client, injectables, ownerID)

			ownerID2 := util.NewUserID()
			CreateTestSquad(ctx, t, client, injectables, ownerID2)

			ownerID3 := util.NewUserID()
			CreateTestSquad(ctx, t, client, injectables, ownerID3)

			ownerID4 := util.NewUserID()
			CreateTestSquad(ctx, t, client, injectables, ownerID4)

			// recipient is following ownerID
			followsClient.On("IsFollowing", mock.Anything, recipientID, ownerID).Maybe().Return(true, nil)
			friendshipClient.On("IsFriend", mock.Anything, recipientID, ownerID).Maybe().Return(false, nil)
			rosterClient.On("IsTeammate", mock.Anything, recipientID, ownerID).Maybe().Return(false, nil)

			// recipient is friend with ownerID2
			followsClient.On("IsFollowing", mock.Anything, recipientID, ownerID2).Maybe().Return(false, nil)
			friendshipClient.On("IsFriend", mock.Anything, recipientID, ownerID2).Maybe().Return(true, nil)
			rosterClient.On("IsTeammate", mock.Anything, recipientID, ownerID2).Maybe().Return(false, nil)

			// recipient is teammate with ownerID3
			followsClient.On("IsFollowing", mock.Anything, recipientID, ownerID3).Maybe().Return(false, nil)
			friendshipClient.On("IsFriend", mock.Anything, recipientID, ownerID3).Maybe().Return(false, nil)
			rosterClient.On("IsTeammate", mock.Anything, recipientID, ownerID3).Maybe().Return(true, nil)

			// ownerID4 is not in the network of recipient
			followsClient.On("IsFollowing", mock.Anything, recipientID, ownerID4).Maybe().Return(false, nil)
			friendshipClient.On("IsFriend", mock.Anything, recipientID, ownerID4).Maybe().Return(false, nil)
			rosterClient.On("IsTeammate", mock.Anything, recipientID, ownerID4).Maybe().Return(false, nil)

			invitations := CreatePendingInvitations(
				ctx, t, client, injectables,
				&meepo.CreateInvitationRequest{
					SenderId:    ownerID,
					RecipientId: recipientID,
					CallerId:    ownerID,
				},
				&meepo.CreateInvitationRequest{
					SenderId:    ownerID2,
					RecipientId: recipientID,
					CallerId:    ownerID2,
				},
				&meepo.CreateInvitationRequest{
					SenderId:    ownerID3,
					RecipientId: recipientID,
					CallerId:    ownerID3,
				},
				&meepo.CreateInvitationRequest{
					SenderId:    ownerID4,
					RecipientId: recipientID,
					CallerId:    ownerID4,
				})
			So(invitations, ShouldHaveLength, 4)

			invitations[0].NetworkType = meepo.Invitation_IN_NETWORK
			invitations[1].NetworkType = meepo.Invitation_IN_NETWORK
			invitations[2].NetworkType = meepo.Invitation_IN_NETWORK
			invitations[3].NetworkType = meepo.Invitation_OUT_OF_NETWORK

			req := &meepo.GetIncomingInvitationsByChannelIDRequest{ChannelId: recipientID, CallerId: recipientID}
			r, err := client.GetIncomingInvitationsByChannelID(ctx, req)
			So(err, ShouldBeNil)
			So(r, ShouldNotBeNil)
			So(r.Invitations, ShouldHaveLength, 4)
			So(r.Invitations[0], ShouldNotBeNil)
			So(r.Invitations[0], ShouldResemble, invitations[3])
			So(r.Invitations[0].Status, ShouldEqual, meepo.Invitation_PENDING)
			So(r.Invitations[1], ShouldNotBeNil)
			So(r.Invitations[1], ShouldResemble, invitations[2])
			So(r.Invitations[1].Status, ShouldEqual, meepo.Invitation_PENDING)
			So(r.Invitations[2], ShouldNotBeNil)
			So(r.Invitations[2], ShouldResemble, invitations[1])
			So(r.Invitations[2].Status, ShouldEqual, meepo.Invitation_PENDING)
			So(r.Invitations[3], ShouldNotBeNil)
			So(r.Invitations[3], ShouldResemble, invitations[0])
			So(r.Invitations[3].Status, ShouldEqual, meepo.Invitation_PENDING)
		})
	})

	ts.onFinish(serverShutdownWaitTime)
}

func TestAcceptInvitation(t *testing.T) {

	adminUser := util.NewUserID()
	injectables := newDefaultInjectables()
	ts := startServer(t, injectables, map[string][]byte{
		"meepo.admin_users": []byte(adminUser),
	})
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	authorizer := (injectables.authorizer).(*mocks.Authorizer)
	pubsubClient := (injectables.pubsubClient).(*testutil.PubsubMockClient)
	ripleyClient := (injectables.ripleyClient).(*mocks.RipleyClient)
	usersClient := (injectables.usersClient).(*mocks.UsersClient)
	channelStatePublisher := (injectables.channelStatePublisherClient).(*mocks.ChannelStatePublisher)

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		client := ts.meepoClient
		ctx := ts.ctx

		authorizer.On("CanAcceptInvitation", mock.Anything, mock.Anything, mock.Anything).Return(true)
		authorizer.On("CanCreateInvitation", mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(true)
		authorizer.On("CanGetInvitationsBySquadID", mock.Anything, mock.Anything, mock.Anything).Return(true)
		ripleyClient.On("GetUserPayoutType", mock.Anything, mock.Anything).Return(
			&clients.UserPayoutType{
				IsPartner: true,
			}, nil)
		usersClient.On("ValidateUser", mock.Anything, mock.Anything).Return(true, nil, nil)

		Convey("No invitation ID param should throw error", func() {
			req := &meepo.AcceptInvitationRequest{CallerId: util.NewUserID()}
			_, err := client.AcceptInvitation(ctx, req)
			So(err, ShouldNotBeNil)
		})

		Convey("No caller ID param should throw error", func() {
			req := &meepo.AcceptInvitationRequest{InvitationId: testutil.MustNewID()}
			_, err := client.AcceptInvitation(ctx, req)
			So(err, ShouldNotBeNil)
		})

		Convey("Invitation ID that does not belong to an invitation should throw error", func() {
			req := &meepo.AcceptInvitationRequest{InvitationId: testutil.MustNewID(), CallerId: util.NewUserID()}
			_, err := client.AcceptInvitation(ctx, req)
			So(err, ShouldNotBeNil)
		})

		Convey("Should accept the invitation if the recipient does not belong to a squad", func() {
			senderID := util.NewUserID()
			recipientID := util.NewUserID()
			memberIDs := []string{senderID, recipientID}

			createInvitationReq := &meepo.CreateInvitationRequest{
				SenderId:    senderID,
				RecipientId: recipientID,
				CallerId:    senderID,
			}
			// A squad is created here if sender does not belong to a squad
			r, err := client.CreateInvitation(ctx, createInvitationReq)
			So(err, ShouldBeNil)
			So(r, ShouldNotBeNil)
			So(r.Invitation, ShouldNotBeNil)
			squadID := r.Invitation.SquadId

			acceptReq := &meepo.AcceptInvitationRequest{
				InvitationId: r.Invitation.Id,
				CallerId:     recipientID,
			}
			r2, err := client.AcceptInvitation(ctx, acceptReq)
			So(err, ShouldBeNil)
			So(r2, ShouldNotBeNil)
			So(r2.Invitation, ShouldNotBeNil)
			So(r2.Invitation.Id, ShouldEqual, r.Invitation.Id)
			So(r2.Invitation.SquadId, ShouldEqual, squadID)
			So(r2.Invitation.SenderId, ShouldEqual, senderID)
			So(r2.Invitation.RecipientId, ShouldEqual, recipientID)
			So(r2.Invitation.Status, ShouldEqual, meepo.Invitation_ACCEPTED)

			pubsubClient.AssertSquadUpdatePublished(t, squadID, models.SquadStatusPending, memberIDs)

			// Check if the recipient has been added as a member.
			getSquadByIDReq := &meepo.GetSquadByIDRequest{
				Id: squadID,
			}
			s, err := client.GetSquadByID(ctx, getSquadByIDReq)
			So(err, ShouldBeNil)
			So(s, ShouldNotBeNil)
			So(s.Squad, ShouldNotBeNil)
			So(s.Squad.Id, ShouldEqual, squadID)
			So(s.Squad.OwnerId, ShouldEqual, senderID)
			So(s.Squad.Status, ShouldEqual, meepo.Squad_PENDING)
			So(s.Squad.MemberIds, ShouldResemble, memberIDs)

			getInvitationBySquadIDReq := &meepo.GetInvitationsBySquadIDRequest{
				SquadId:  squadID,
				CallerId: senderID,
				Status:   meepo.Invitation_PENDING,
			}
			i, err := client.GetInvitationsBySquadID(ctx, getInvitationBySquadIDReq)
			So(err, ShouldBeNil)
			So(i, ShouldNotBeNil)
			So(i.Invitations, ShouldHaveLength, 0)
		})

		Convey("Should accept the invitation if the recipient already belongs to a squad", func() {
			recipientID := util.NewUserID()
			oldOwnerID := util.NewUserID()
			oldSquad := CreateTestSquadWithMember(ctx, t, client, injectables, oldOwnerID, recipientID)
			senderID := util.NewUserID()
			memberIDs := []string{senderID, recipientID}

			createInvitationReq := &meepo.CreateInvitationRequest{
				SenderId:    senderID,
				RecipientId: recipientID,
				CallerId:    senderID,
			}
			r, err := client.CreateInvitation(ctx, createInvitationReq)
			So(err, ShouldBeNil)
			So(r, ShouldNotBeNil)
			So(r.Invitation, ShouldNotBeNil)
			squadID := r.Invitation.SquadId

			acceptReq := &meepo.AcceptInvitationRequest{
				InvitationId: r.Invitation.Id,
				CallerId:     recipientID,
			}
			r2, err := client.AcceptInvitation(ctx, acceptReq)
			So(err, ShouldBeNil)
			So(r2, ShouldNotBeNil)
			So(r2.Invitation, ShouldNotBeNil)
			So(r2.Invitation.Id, ShouldEqual, r.Invitation.Id)
			So(r2.Invitation.SquadId, ShouldEqual, squadID)
			So(r2.Invitation.SenderId, ShouldEqual, senderID)
			So(r2.Invitation.RecipientId, ShouldEqual, recipientID)
			So(r2.Invitation.Status, ShouldEqual, meepo.Invitation_ACCEPTED)

			pubsubClient.AssertSquadUpdatePublished(t, squadID, models.SquadStatusPending, memberIDs)
			pubsubClient.AssertSquadUpdatePublished(t, oldSquad.Id, models.SquadStatusPending, []string{oldOwnerID})

			getSquadByIDReq := &meepo.GetSquadByIDRequest{
				Id: squadID,
			}
			s, err := client.GetSquadByID(ctx, getSquadByIDReq)
			So(err, ShouldBeNil)
			So(s, ShouldNotBeNil)
			So(s.Squad, ShouldNotBeNil)
			So(s.Squad.Id, ShouldEqual, squadID)
			So(s.Squad.OwnerId, ShouldEqual, senderID)
			So(s.Squad.Status, ShouldEqual, meepo.Squad_PENDING)
			So(s.Squad.MemberIds, ShouldResemble, memberIDs)

			getInvitationBySquadIDReq := &meepo.GetInvitationsBySquadIDRequest{
				SquadId:  squadID,
				CallerId: senderID,
				Status:   meepo.Invitation_PENDING,
			}
			i, err := client.GetInvitationsBySquadID(ctx, getInvitationBySquadIDReq)
			So(err, ShouldBeNil)
			So(i, ShouldNotBeNil)
			So(i.Invitations, ShouldHaveLength, 0)

			getSquadByIDReq2 := &meepo.GetSquadByIDRequest{
				Id: oldSquad.Id,
			}
			s2, err := client.GetSquadByID(ctx, getSquadByIDReq2)
			So(err, ShouldBeNil)
			So(s2, ShouldNotBeNil)
			So(s2.Squad, ShouldNotBeNil)
			So(s2.Squad.Id, ShouldEqual, oldSquad.Id)
			So(s2.Squad.OwnerId, ShouldEqual, oldSquad.OwnerId)
			So(s2.Squad.MemberIds, ShouldResemble, []string{oldSquad.OwnerId})

			getInvitationBySquadIDReq2 := &meepo.GetInvitationsBySquadIDRequest{
				SquadId:  oldSquad.Id,
				CallerId: oldSquad.OwnerId,
				Status:   meepo.Invitation_PENDING,
			}
			i2, err := client.GetInvitationsBySquadID(ctx, getInvitationBySquadIDReq2)
			So(err, ShouldBeNil)
			So(i2, ShouldNotBeNil)
			So(i2.Invitations, ShouldHaveLength, 0)
		})

		Convey("Should publish a channel update when a recipient accepts an invite to a LIVE squad", func() {
			// Create a LIVE squad
			ownerID := util.NewUserID()
			squad := CreateTestLiveSquadWithMember(ctx, t, client, injectables, ownerID, util.NewUserID())
			So(squad, ShouldNotBeNil)
			So(squad.Status, ShouldEqual, meepo.Squad_LIVE)

			squadID := squad.Id
			recipientID := util.NewUserID()

			// Create an invitation to join the live squad.
			createInvitationReq := &meepo.CreateInvitationRequest{
				SenderId:    ownerID,
				RecipientId: recipientID,
				CallerId:    ownerID,
			}
			r, err := client.CreateInvitation(ctx, createInvitationReq)
			So(err, ShouldBeNil)
			So(r, ShouldNotBeNil)
			So(r.Invitation, ShouldNotBeNil)
			pubsubClient.AssertIncomingSquadInvitesPublished(t, recipientID, []testutil.ExpectedPubsubInvitations{
				{SenderID: ownerID, RecipientID: recipientID, Status: models.InvitationStatusPending},
			})

			// Accept the invitation, and verify that a channel update was published.
			acceptReq := &meepo.AcceptInvitationRequest{
				InvitationId: r.Invitation.Id,
				CallerId:     recipientID,
			}
			_, err = client.AcceptInvitation(ctx, acceptReq)
			So(err, ShouldBeNil)

			channelStatePublisher.AssertCalled(t, "PublishChannelIsInLiveSquad", mock.Anything, squadID, recipientID)
			pubsubClient.AssertIncomingSquadInvitesPublished(t, recipientID, []testutil.ExpectedPubsubInvitations{})
		})

		Convey("Should NOT publish a channel update when a recipient accepts an invite to a PENDING squad", func() {
			// Create a PENDING squad
			ownerID := util.NewUserID()
			squad := CreateTestSquadWithMember(ctx, t, client, injectables, ownerID, util.NewUserID())
			So(squad, ShouldNotBeNil)
			So(squad.Status, ShouldEqual, meepo.Squad_PENDING)

			squadID := squad.Id
			recipientID := util.NewUserID()

			// Create an invitation to join the live squad.
			createInvitationReq := &meepo.CreateInvitationRequest{
				SenderId:    ownerID,
				RecipientId: recipientID,
				CallerId:    ownerID,
			}
			r, err := client.CreateInvitation(ctx, createInvitationReq)
			So(err, ShouldBeNil)
			So(r, ShouldNotBeNil)
			So(r.Invitation, ShouldNotBeNil)

			// Accept the invitation, and verify that NO channel update was published.
			acceptReq := &meepo.AcceptInvitationRequest{
				InvitationId: r.Invitation.Id,
				CallerId:     recipientID,
			}
			_, err = client.AcceptInvitation(ctx, acceptReq)
			So(err, ShouldBeNil)

			channelStatePublisher.AssertNotCalled(t, "PublishChannelIsInLiveSquad", mock.Anything, squadID, recipientID)
		})
	})

	ts.onFinish(serverShutdownWaitTime)
}

func TestDeleteInvitation(t *testing.T) {

	injectables := newDefaultInjectables()
	pubsubClient := (injectables.pubsubClient).(*testutil.PubsubMockClient)
	ts := startServer(t, injectables)
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	authorizer := (injectables.authorizer).(*mocks.Authorizer)
	ripleyClient := (injectables.ripleyClient).(*mocks.RipleyClient)
	usersClient := (injectables.usersClient).(*mocks.UsersClient)

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		client := ts.meepoClient
		ctx := ts.ctx

		authorizer.On("CanCreateInvitation", mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(true)
		authorizer.On("CanDeleteInvitation", mock.Anything, mock.Anything, mock.Anything).Return(true)
		authorizer.On("CanGetInvitationsBySquadID", mock.Anything, mock.Anything, mock.Anything).Return(true)
		ripleyClient.On("GetUserPayoutType", mock.Anything, mock.Anything).Return(
			&clients.UserPayoutType{
				IsPartner: true,
			}, nil)
		usersClient.On("ValidateUser", mock.Anything, mock.Anything).Return(true, nil, nil)

		Convey("No invitation ID param should throw error", func() {
			req := &meepo.DeleteInvitationRequest{CallerId: util.NewUserID()}
			_, err := client.DeleteInvitation(ctx, req)
			So(err, ShouldNotBeNil)
		})

		Convey("No caller ID param should throw error", func() {
			req := &meepo.DeleteInvitationRequest{InvitationId: testutil.MustNewID()}
			_, err := client.DeleteInvitation(ctx, req)
			So(err, ShouldNotBeNil)
		})

		Convey("Invitation ID that does not exist should throw error", func() {
			req := &meepo.DeleteInvitationRequest{InvitationId: testutil.MustNewID(), CallerId: util.NewUserID()}
			_, err := client.DeleteInvitation(ctx, req)
			So(err, ShouldNotBeNil)
		})

		Convey("Should cancel pending invitation", func() {
			ownerID := util.NewUserID()
			memberID := util.NewUserID()
			recipientID := util.NewUserID()
			squad := CreateTestSquadWithMemberAndInvitation(ctx, t, client, injectables, ownerID, memberID, recipientID)

			// Check that squad owner ID is ownerID
			So(squad.OwnerId, ShouldEqual, ownerID)

			// Check that squad has pending invitation and exists
			invitations, err := client.GetInvitationsBySquadID(ctx, &meepo.GetInvitationsBySquadIDRequest{
				SquadId:  squad.Id,
				CallerId: ownerID,
				Status:   meepo.Invitation_PENDING,
			})
			So(err, ShouldBeNil)
			So(invitations.Invitations, ShouldHaveLength, 1)
			So(invitations.Invitations[0].Status, ShouldEqual, meepo.Invitation_PENDING)
			So(invitations.Invitations[0].RecipientId, ShouldEqual, recipientID)
			pubsubClient.AssertIncomingSquadInvitesPublished(t, recipientID, []testutil.ExpectedPubsubInvitations{
				{SenderID: ownerID, RecipientID: recipientID, Status: models.InvitationStatusPending},
			})

			// Delete invitation with caller as ownerID
			req := &meepo.DeleteInvitationRequest{
				InvitationId: invitations.Invitations[0].Id,
				CallerId:     ownerID,
			}
			res, err := client.DeleteInvitation(ctx, req)
			So(err, ShouldBeNil)

			// Invitation deleted
			So(res.Invitation.Id, ShouldEqual, invitations.Invitations[0].Id)
			So(res.Invitation.Status, ShouldEqual, meepo.Invitation_DELETED)
			pubsubClient.AssertIncomingSquadInvitesPublished(t, recipientID, []testutil.ExpectedPubsubInvitations{})

			// No more pending invitations
			reInvitations, err := client.GetInvitationsBySquadID(ctx, &meepo.GetInvitationsBySquadIDRequest{
				SquadId:  squad.Id,
				CallerId: ownerID,
				Status:   meepo.Invitation_PENDING,
			})
			So(err, ShouldBeNil)
			So(reInvitations.Invitations, ShouldHaveLength, 0)
		})

		Convey("Twitch editors for invitation sender can cancel pending invitation", func() {
			ownerID := util.NewUserID()
			memberID := util.NewUserID()
			recipientID := util.NewUserID()
			editorID := util.NewUserID()

			squad := CreateTestSquadWithMemberAndInvitation(ctx, t, client, injectables, ownerID, memberID, recipientID)

			// Check that squad has pending invitation and exists
			invitations, err := client.GetInvitationsBySquadID(ctx, &meepo.GetInvitationsBySquadIDRequest{
				SquadId:  squad.Id,
				CallerId: ownerID,
				Status:   meepo.Invitation_PENDING,
			})
			So(err, ShouldBeNil)
			So(invitations.Invitations, ShouldHaveLength, 1)
			So(invitations.Invitations[0].Status, ShouldEqual, meepo.Invitation_PENDING)
			So(invitations.Invitations[0].RecipientId, ShouldEqual, recipientID)
			pubsubClient.AssertIncomingSquadInvitesPublished(t, recipientID, []testutil.ExpectedPubsubInvitations{
				{SenderID: ownerID, RecipientID: recipientID, Status: models.InvitationStatusPending},
			})

			req := &meepo.DeleteInvitationRequest{
				InvitationId: invitations.Invitations[0].Id,
				CallerId:     editorID,
			}
			res, err := client.DeleteInvitation(ctx, req)
			So(err, ShouldBeNil)

			// Invitation deleted
			So(res.Invitation.Id, ShouldEqual, invitations.Invitations[0].Id)
			So(res.Invitation.Status, ShouldEqual, meepo.Invitation_DELETED)
			pubsubClient.AssertIncomingSquadInvitesPublished(t, recipientID, []testutil.ExpectedPubsubInvitations{})

			// No more pending invitations
			reInvitations, err := client.GetInvitationsBySquadID(ctx, &meepo.GetInvitationsBySquadIDRequest{
				SquadId:  squad.Id,
				CallerId: ownerID,
				Status:   meepo.Invitation_PENDING,
			})
			So(err, ShouldBeNil)
			So(reInvitations.Invitations, ShouldHaveLength, 0)
		})

		// TODO: Test to check that rejected invitation can also be cancelled
	})

	ts.onFinish(serverShutdownWaitTime)
}

func TestAuthorizeTopicSubscription(t *testing.T) {

	injectables := newDefaultInjectables()
	ts := startServer(t, injectables)
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	authorizer := (injectables.authorizer).(*mocks.Authorizer)

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		client := ts.meepoClient
		ctx := ts.ctx

		authorizer.On("CanSubscribe", mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(true)

		Convey("No topic param should throw error", func() {
			req := &meepo.AuthorizeTopicSubscriptionRequest{CallerId: util.NewUserID()}
			_, err := client.AuthorizeTopicSubscription(ctx, req)
			So(err, ShouldNotBeNil)
		})

		Convey("No caller ID param should throw error", func() {
			req := &meepo.AuthorizeTopicSubscriptionRequest{Topic: "channel-squad-invites.<channel_id>"}
			_, err := client.AuthorizeTopicSubscription(ctx, req)
			So(err, ShouldNotBeNil)
		})

		Convey("channel-squad-invites.<channel_id> topic", func() {
			Convey("Authorize channel owner to subscribe to topic", func() {
				channelID := util.NewUserID()
				req := &meepo.AuthorizeTopicSubscriptionRequest{
					Topic:    fmt.Sprintf("channel-squad-invites.%s", channelID),
					CallerId: channelID,
				}

				isAuthorized, err := client.AuthorizeTopicSubscription(ctx, req)
				So(err, ShouldBeNil)
				So(isAuthorized.IsAuthorized, ShouldBeTrue)
			})
		})
	})

	ts.onFinish(serverShutdownWaitTime)
}

func TestRejectInvitation(t *testing.T) {

	injectables := newDefaultInjectables()
	ts := startServer(t, injectables)
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	authorizer := (injectables.authorizer).(*mocks.Authorizer)
	pubsubClient := (injectables.pubsubClient).(*testutil.PubsubMockClient)

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		client := ts.meepoClient
		ctx := ts.ctx

		authorizer.On("CanRejectInvitation", mock.Anything, mock.Anything, mock.Anything).Return(true)
		authorizer.On("CanGetInvitationsBySquadID", mock.Anything, mock.Anything, mock.Anything).Return(true)

		Convey("No invitation ID param should throw error", func() {
			req := &meepo.RejectInvitationRequest{CallerId: util.NewUserID()}
			_, err := client.RejectInvitation(ctx, req)
			So(err, ShouldNotBeNil)
		})

		Convey("No caller ID param should throw error", func() {
			req := &meepo.RejectInvitationRequest{InvitationId: testutil.MustNewID()}
			_, err := client.RejectInvitation(ctx, req)
			So(err, ShouldNotBeNil)
		})

		Convey("Invitation ID that does not exist should throw error", func() {
			req := &meepo.RejectInvitationRequest{InvitationId: testutil.MustNewID(), CallerId: util.NewUserID()}
			_, err := client.RejectInvitation(ctx, req)
			So(err, ShouldNotBeNil)
		})

		Convey("Should reject pending invitations with reason recipient rejected", func() {
			ownerID := util.NewUserID()
			memberID := util.NewUserID()
			recipientID := util.NewUserID()
			squad := CreateTestSquadWithMemberAndInvitation(ctx, t, client, injectables, ownerID, memberID, recipientID)

			// Check that squad has pending invitation and exists
			invitations, err := client.GetInvitationsBySquadID(ctx, &meepo.GetInvitationsBySquadIDRequest{
				SquadId:  squad.Id,
				CallerId: ownerID,
				Status:   meepo.Invitation_PENDING,
			})
			So(err, ShouldBeNil)
			So(invitations.Invitations, ShouldHaveLength, 1)
			So(invitations.Invitations[0].Status, ShouldEqual, meepo.Invitation_PENDING)
			So(invitations.Invitations[0].RecipientId, ShouldEqual, recipientID)

			// Reject invitation with caller as recipientID
			req := &meepo.RejectInvitationRequest{
				InvitationId: invitations.Invitations[0].Id,
				CallerId:     recipientID,
			}
			rejectedInvitation, err := client.RejectInvitation(ctx, req)

			So(err, ShouldBeNil)
			So(rejectedInvitation.Invitation.Status, ShouldEqual, meepo.Invitation_REJECTED)
			So(rejectedInvitation.Invitation.Id, ShouldEqual, invitations.Invitations[0].Id)
			So(rejectedInvitation.Invitation.RecipientId, ShouldEqual, recipientID)
			So(rejectedInvitation.Invitation.ReasonRejected, ShouldEqual, meepo.Invitation_RECIPIENT_REJECTED)

			pubsubClient.AssertSquadUpdatePublished(t, squad.Id, models.SquadStatusPending, []string{ownerID, memberID})
			pubsubClient.AssertIncomingSquadInvitesPublished(t, recipientID, []testutil.ExpectedPubsubInvitations{})
		})

		Convey("Should delete older invitation when there are more than 3 rejected per squad", func() {
			ownerID := util.NewUserID()
			recipientID1 := util.NewUserID()
			recipientID2 := util.NewUserID()
			recipientID3 := util.NewUserID()
			recipientID4 := util.NewUserID()
			squad := CreateTestSquad(ctx, t, client, injectables, ownerID)

			// Create 3 pending invitations to hit max.
			CreatePendingInvitations(
				ctx, t, client, injectables,
				&meepo.CreateInvitationRequest{
					SenderId:    ownerID,
					CallerId:    ownerID,
					RecipientId: recipientID1,
				},
				&meepo.CreateInvitationRequest{
					SenderId:    ownerID,
					CallerId:    ownerID,
					RecipientId: recipientID2,
				},
				&meepo.CreateInvitationRequest{
					SenderId:    ownerID,
					CallerId:    ownerID,
					RecipientId: recipientID3,
				},
			)

			// Check that squad has 3 pending invitations
			invitations, err := client.GetInvitationsBySquadID(ctx, &meepo.GetInvitationsBySquadIDRequest{
				SquadId:  squad.Id,
				CallerId: ownerID,
				Status:   meepo.Invitation_PENDING,
			})
			So(err, ShouldBeNil)
			So(invitations.Invitations, ShouldHaveLength, 3)

			// Reject invitation all 3 pending invitations
			rejectedInvitation1, err := client.RejectInvitation(ctx, &meepo.RejectInvitationRequest{
				InvitationId: invitations.Invitations[0].Id,
				CallerId:     invitations.Invitations[0].RecipientId,
			})
			So(err, ShouldBeNil)
			So(rejectedInvitation1, ShouldNotBeNil)
			rejectedInvitation2, err := client.RejectInvitation(ctx, &meepo.RejectInvitationRequest{
				InvitationId: invitations.Invitations[1].Id,
				CallerId:     invitations.Invitations[1].RecipientId,
			})
			So(err, ShouldBeNil)
			So(rejectedInvitation2, ShouldNotBeNil)
			rejectedInvitation3, err := client.RejectInvitation(ctx, &meepo.RejectInvitationRequest{
				InvitationId: invitations.Invitations[2].Id,
				CallerId:     invitations.Invitations[2].RecipientId,
			})
			So(err, ShouldBeNil)
			So(rejectedInvitation3, ShouldNotBeNil)

			// Verify no pending invitations
			invitations, err = client.GetInvitationsBySquadID(ctx, &meepo.GetInvitationsBySquadIDRequest{
				SquadId:  squad.Id,
				CallerId: ownerID,
				Status:   meepo.Invitation_PENDING,
			})
			So(err, ShouldBeNil)
			So(invitations.Invitations, ShouldBeNil)

			// Create 1 more pending invitation
			pendingInvitation := CreatePendingInvitations(
				ctx, t, client, injectables,
				&meepo.CreateInvitationRequest{
					SenderId:    ownerID,
					CallerId:    ownerID,
					RecipientId: recipientID4,
				},
			)

			// Verify that there are 3 rejected invitations, hitting max
			currRejectedInvitations, err := client.GetInvitationsBySquadID(ctx, &meepo.GetInvitationsBySquadIDRequest{
				SquadId:  squad.Id,
				CallerId: ownerID,
				Status:   meepo.Invitation_REJECTED,
			})
			So(err, ShouldBeNil)
			So(currRejectedInvitations.Invitations, ShouldHaveLength, 3)

			// Verify that there are currently no deleted invitations
			currDeletedInvitations, err := client.GetInvitationsBySquadID(ctx, &meepo.GetInvitationsBySquadIDRequest{
				SquadId:  squad.Id,
				CallerId: ownerID,
				Status:   meepo.Invitation_DELETED,
			})
			So(err, ShouldBeNil)
			So(currDeletedInvitations.Invitations, ShouldBeNil)

			// Reject another invitation, oldest rejected inv should be deleted here
			rejectedInvitation4, err := client.RejectInvitation(ctx, &meepo.RejectInvitationRequest{
				InvitationId: pendingInvitation[0].Id,
				CallerId:     recipientID4,
			})
			So(err, ShouldBeNil)
			So(rejectedInvitation4, ShouldNotBeNil)

			// Verify that oldest rejected inv has been deleted
			currDeletedInvitations, err = client.GetInvitationsBySquadID(ctx, &meepo.GetInvitationsBySquadIDRequest{
				SquadId:  squad.Id,
				CallerId: ownerID,
				Status:   meepo.Invitation_DELETED,
			})
			So(err, ShouldBeNil)
			So(currDeletedInvitations.Invitations, ShouldHaveLength, 1)
			So(currDeletedInvitations.Invitations[0].Id, ShouldEqual, rejectedInvitation1.Invitation.Id)

			// Verify that squad has 3 rejected invitations
			currRejectedInvitations, err = client.GetInvitationsBySquadID(ctx, &meepo.GetInvitationsBySquadIDRequest{
				SquadId:  squad.Id,
				CallerId: ownerID,
				Status:   meepo.Invitation_REJECTED,
			})
			So(err, ShouldBeNil)
			So(currRejectedInvitations.Invitations, ShouldHaveLength, 3)
		})
	})

	ts.onFinish(serverShutdownWaitTime)
}

func TestRejectOutOfNetworkInvitations(t *testing.T) {

	followsClient := &mocks.FollowsClient{}
	friendshipClient := &mocks.FriendshipClient{}
	rosterClient := &mocks.RosterClient{}
	spadeClient := &mocks.SpadeClient{}

	injectables := newDefaultInjectables()
	injectables.followsClient = followsClient
	injectables.friendshipClient = friendshipClient
	injectables.rosterClient = rosterClient
	injectables.spadeClient = spadeClient

	pubsubClient := (injectables.pubsubClient).(*testutil.PubsubMockClient)

	ts := startServer(t, injectables)
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	authorizer := (injectables.authorizer).(*mocks.Authorizer)
	ripleyClient := (injectables.ripleyClient).(*mocks.RipleyClient)
	usersClient := (injectables.usersClient).(*mocks.UsersClient)

	authorizer.On("CanGetInvitationsBySquadID", mock.Anything, mock.Anything, mock.Anything).Return(true)
	authorizer.On("CanRejectInvitation", mock.Anything, mock.Anything, mock.Anything).Return(true)
	authorizer.On("CanRejectOutOfNetworkInvitations", mock.Anything, mock.Anything, mock.Anything).Return(true)
	authorizer.On("CanGetPendingInvitationsByRecipientID", mock.Anything, mock.Anything, mock.Anything).Return(true)
	ripleyClient.On("GetUserPayoutType", mock.Anything, mock.Anything).Return(
		&clients.UserPayoutType{
			IsPartner: true,
		}, nil)
	usersClient.On("ValidateUser", mock.Anything, mock.Anything).Return(true, nil, nil)
	spadeClient.On("TrackEvent", mock.Anything, mock.Anything, mock.Anything).Return(nil)

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		client := ts.meepoClient
		ctx := ts.ctx

		Convey("No channel ID param should throw error", func() {
			req := &meepo.RejectOutOfNetworkInvitationsRequest{CallerId: util.NewUserID()}
			_, err := client.RejectOutOfNetworkInvitations(ctx, req)
			So(err, ShouldNotBeNil)
		})

		Convey("No caller ID param should throw error", func() {
			req := &meepo.RejectOutOfNetworkInvitationsRequest{ChannelId: util.NewUserID()}
			_, err := client.RejectOutOfNetworkInvitations(ctx, req)
			So(err, ShouldNotBeNil)
		})

		Convey("Should reject pending, out-of-network invitations with reason recipient rejected", func() {
			ownerID := util.NewUserID()
			memberID := util.NewUserID()
			recipientID := util.NewUserID()

			client.UpdateInvitePolicyByChannelID(ctx, &meepo.UpdateInvitePolicyByChannelIDRequest{
				ChannelId:    recipientID,
				CallerId:     recipientID,
				InvitePolicy: meepo.InvitePolicy_ALL,
			})

			// owner is not in the network of the recipient
			followsClient.On("IsFollowing", mock.Anything, recipientID, ownerID).Maybe().Return(false, nil)
			friendshipClient.On("IsFriend", mock.Anything, recipientID, ownerID).Maybe().Return(false, nil)
			rosterClient.On("IsTeammate", mock.Anything, recipientID, ownerID).Maybe().Return(false, nil)

			squad := CreateTestSquadWithMemberAndInvitation(ctx, t, client, injectables, ownerID, memberID, recipientID)

			// Check that squad has pending invitation and exists
			squadInvitations, err := client.GetInvitationsBySquadID(ctx, &meepo.GetInvitationsBySquadIDRequest{
				SquadId:  squad.Id,
				CallerId: ownerID,
				Status:   meepo.Invitation_PENDING,
			})
			So(err, ShouldBeNil)
			So(squadInvitations.Invitations, ShouldHaveLength, 1)
			So(squadInvitations.Invitations[0].Status, ShouldEqual, meepo.Invitation_PENDING)
			So(squadInvitations.Invitations[0].RecipientId, ShouldEqual, recipientID)

			// Check that recipient has a pending invitation
			channelInvitations, err := client.GetIncomingInvitationsByChannelID(ctx, &meepo.GetIncomingInvitationsByChannelIDRequest{
				ChannelId: recipientID,
				CallerId:  recipientID,
			})
			So(err, ShouldBeNil)
			So(channelInvitations.Invitations, ShouldHaveLength, 1)
			So(channelInvitations.Invitations[0].Status, ShouldEqual, meepo.Invitation_PENDING)
			So(channelInvitations.Invitations[0].SenderId, ShouldEqual, ownerID)

			pubsubClient.AssertIncomingSquadInvitesPublished(t, recipientID, []testutil.ExpectedPubsubInvitations{
				{SenderID: ownerID, RecipientID: recipientID, Status: models.InvitationStatusPending},
			})

			// Reject out-of-network invitations for recipient
			req := &meepo.RejectOutOfNetworkInvitationsRequest{
				ChannelId: recipientID,
				CallerId:  recipientID,
			}
			_, err = client.RejectOutOfNetworkInvitations(ctx, req)
			So(err, ShouldBeNil)

			// Check that squad has no pending invitation
			squadInvitations, err = client.GetInvitationsBySquadID(ctx, &meepo.GetInvitationsBySquadIDRequest{
				SquadId:  squad.Id,
				CallerId: ownerID,
				Status:   meepo.Invitation_PENDING,
			})
			So(err, ShouldBeNil)
			So(squadInvitations.Invitations, ShouldHaveLength, 0)

			// Check that recipient has no pending invitation
			channelInvitations, err = client.GetIncomingInvitationsByChannelID(ctx, &meepo.GetIncomingInvitationsByChannelIDRequest{
				ChannelId: recipientID,
				CallerId:  recipientID,
			})
			So(err, ShouldBeNil)
			So(channelInvitations.Invitations, ShouldHaveLength, 0)

			pubsubClient.AssertIncomingSquadInvitesPublished(t, recipientID, []testutil.ExpectedPubsubInvitations{})
		})

		Convey("Should not reject pending, in-network invitations", func() {
			ownerID := util.NewUserID()
			memberID := util.NewUserID()
			recipientID := util.NewUserID()

			// recipient is following owner
			followsClient.On("IsFollowing", mock.Anything, recipientID, ownerID).Maybe().Return(true, nil)
			friendshipClient.On("IsFriend", mock.Anything, recipientID, ownerID).Maybe().Return(false, nil)
			rosterClient.On("IsTeammate", mock.Anything, recipientID, ownerID).Maybe().Return(false, nil)

			squad := CreateTestSquadWithMemberAndInvitation(ctx, t, client, injectables, ownerID, memberID, recipientID)

			// Check that squad has pending invitation and exists
			squadInvitations, err := client.GetInvitationsBySquadID(ctx, &meepo.GetInvitationsBySquadIDRequest{
				SquadId:  squad.Id,
				CallerId: ownerID,
				Status:   meepo.Invitation_PENDING,
			})
			So(err, ShouldBeNil)
			So(squadInvitations.Invitations, ShouldHaveLength, 1)
			So(squadInvitations.Invitations[0].Status, ShouldEqual, meepo.Invitation_PENDING)
			So(squadInvitations.Invitations[0].RecipientId, ShouldEqual, recipientID)

			// Check that recipient has a pending invitation
			channelInvitations, err := client.GetIncomingInvitationsByChannelID(ctx, &meepo.GetIncomingInvitationsByChannelIDRequest{
				ChannelId: recipientID,
				CallerId:  recipientID,
			})
			So(err, ShouldBeNil)
			So(channelInvitations.Invitations, ShouldHaveLength, 1)
			So(channelInvitations.Invitations[0].Status, ShouldEqual, meepo.Invitation_PENDING)
			So(channelInvitations.Invitations[0].SenderId, ShouldEqual, ownerID)

			// Reject out-of-network invitations for recipient
			req := &meepo.RejectOutOfNetworkInvitationsRequest{
				ChannelId: recipientID,
				CallerId:  recipientID,
			}
			_, err = client.RejectOutOfNetworkInvitations(ctx, req)

			So(err, ShouldBeNil)

			// Check that squad has pending invitation
			squadInvitations, err = client.GetInvitationsBySquadID(ctx, &meepo.GetInvitationsBySquadIDRequest{
				SquadId:  squad.Id,
				CallerId: ownerID,
				Status:   meepo.Invitation_PENDING,
			})
			So(err, ShouldBeNil)
			So(squadInvitations.Invitations, ShouldHaveLength, 1)
			So(squadInvitations.Invitations[0].Status, ShouldEqual, meepo.Invitation_PENDING)
			So(squadInvitations.Invitations[0].RecipientId, ShouldEqual, recipientID)

			// Check that recipient has pending invitation
			channelInvitations, err = client.GetIncomingInvitationsByChannelID(ctx, &meepo.GetIncomingInvitationsByChannelIDRequest{
				ChannelId: recipientID,
				CallerId:  recipientID,
			})
			So(err, ShouldBeNil)
			So(channelInvitations.Invitations, ShouldHaveLength, 1)
			So(channelInvitations.Invitations[0].Status, ShouldEqual, meepo.Invitation_PENDING)
			So(channelInvitations.Invitations[0].SenderId, ShouldEqual, ownerID)
		})

		Convey("Should delete older invitation when there are more than 3 rejected per squad", func() {
			ownerID := util.NewUserID()
			recipientID1 := util.NewUserID()
			recipientID2 := util.NewUserID()
			recipientID3 := util.NewUserID()
			recipientID4 := util.NewUserID()
			squad := CreateTestSquad(ctx, t, client, injectables, ownerID)

			client.UpdateInvitePolicyByChannelID(ctx, &meepo.UpdateInvitePolicyByChannelIDRequest{
				ChannelId:    recipientID1,
				CallerId:     recipientID1,
				InvitePolicy: meepo.InvitePolicy_ALL,
			})

			client.UpdateInvitePolicyByChannelID(ctx, &meepo.UpdateInvitePolicyByChannelIDRequest{
				ChannelId:    recipientID2,
				CallerId:     recipientID2,
				InvitePolicy: meepo.InvitePolicy_ALL,
			})

			client.UpdateInvitePolicyByChannelID(ctx, &meepo.UpdateInvitePolicyByChannelIDRequest{
				ChannelId:    recipientID3,
				CallerId:     recipientID3,
				InvitePolicy: meepo.InvitePolicy_ALL,
			})

			client.UpdateInvitePolicyByChannelID(ctx, &meepo.UpdateInvitePolicyByChannelIDRequest{
				ChannelId:    recipientID4,
				CallerId:     recipientID4,
				InvitePolicy: meepo.InvitePolicy_ALL,
			})

			followsClient.On("IsFollowing", mock.Anything, recipientID1, ownerID).Maybe().Return(false, nil)
			friendshipClient.On("IsFriend", mock.Anything, recipientID1, ownerID).Maybe().Return(true, nil)
			rosterClient.On("IsTeammate", mock.Anything, recipientID1, ownerID).Maybe().Return(false, nil)

			followsClient.On("IsFollowing", mock.Anything, recipientID2, ownerID).Maybe().Return(false, nil)
			friendshipClient.On("IsFriend", mock.Anything, recipientID2, ownerID).Maybe().Return(true, nil)
			rosterClient.On("IsTeammate", mock.Anything, recipientID2, ownerID).Maybe().Return(false, nil)

			followsClient.On("IsFollowing", mock.Anything, recipientID3, ownerID).Maybe().Return(false, nil)
			friendshipClient.On("IsFriend", mock.Anything, recipientID3, ownerID).Maybe().Return(true, nil)
			rosterClient.On("IsTeammate", mock.Anything, recipientID3, ownerID).Maybe().Return(false, nil)

			// Create 3 pending invitations to hit max.
			CreatePendingInvitations(
				ctx, t, client, injectables,
				&meepo.CreateInvitationRequest{
					SenderId:    ownerID,
					CallerId:    ownerID,
					RecipientId: recipientID1,
				},
				&meepo.CreateInvitationRequest{
					SenderId:    ownerID,
					CallerId:    ownerID,
					RecipientId: recipientID2,
				},
				&meepo.CreateInvitationRequest{
					SenderId:    ownerID,
					CallerId:    ownerID,
					RecipientId: recipientID3,
				},
			)

			// Check that squad has 3 pending invitations
			invitations, err := client.GetInvitationsBySquadID(ctx, &meepo.GetInvitationsBySquadIDRequest{
				SquadId:  squad.Id,
				CallerId: ownerID,
				Status:   meepo.Invitation_PENDING,
			})
			So(err, ShouldBeNil)
			So(invitations.Invitations, ShouldHaveLength, 3)

			// Reject invitation all 3 pending invitations
			rejectedInvitation1, err := client.RejectInvitation(ctx, &meepo.RejectInvitationRequest{
				InvitationId: invitations.Invitations[0].Id,
				CallerId:     invitations.Invitations[0].RecipientId,
			})
			So(err, ShouldBeNil)
			So(rejectedInvitation1, ShouldNotBeNil)
			rejectedInvitation2, err := client.RejectInvitation(ctx, &meepo.RejectInvitationRequest{
				InvitationId: invitations.Invitations[1].Id,
				CallerId:     invitations.Invitations[1].RecipientId,
			})
			So(err, ShouldBeNil)
			So(rejectedInvitation2, ShouldNotBeNil)
			rejectedInvitation3, err := client.RejectInvitation(ctx, &meepo.RejectInvitationRequest{
				InvitationId: invitations.Invitations[2].Id,
				CallerId:     invitations.Invitations[2].RecipientId,
			})
			So(err, ShouldBeNil)
			So(rejectedInvitation3, ShouldNotBeNil)

			// Verify no pending invitations
			invitations, err = client.GetInvitationsBySquadID(ctx, &meepo.GetInvitationsBySquadIDRequest{
				SquadId:  squad.Id,
				CallerId: ownerID,
				Status:   meepo.Invitation_PENDING,
			})
			So(err, ShouldBeNil)
			So(invitations.Invitations, ShouldBeNil)

			// owner is not in the network of recipient4
			followsClient.On("IsFollowing", mock.Anything, recipientID4, ownerID).Maybe().Return(false, nil)
			friendshipClient.On("IsFriend", mock.Anything, recipientID4, ownerID).Maybe().Return(false, nil)
			rosterClient.On("IsTeammate", mock.Anything, recipientID4, ownerID).Maybe().Return(false, nil)

			// Create 1 more pending invitation
			CreatePendingInvitations(
				ctx, t, client, injectables,
				&meepo.CreateInvitationRequest{
					SenderId:    ownerID,
					CallerId:    ownerID,
					RecipientId: recipientID4,
				},
			)

			// Verify that there are 3 rejected invitations, hitting max
			currRejectedInvitations, err := client.GetInvitationsBySquadID(ctx, &meepo.GetInvitationsBySquadIDRequest{
				SquadId:  squad.Id,
				CallerId: ownerID,
				Status:   meepo.Invitation_REJECTED,
			})
			So(err, ShouldBeNil)
			So(currRejectedInvitations.Invitations, ShouldHaveLength, 3)

			// Verify that there are currently no deleted invitations
			currDeletedInvitations, err := client.GetInvitationsBySquadID(ctx, &meepo.GetInvitationsBySquadIDRequest{
				SquadId:  squad.Id,
				CallerId: ownerID,
				Status:   meepo.Invitation_DELETED,
			})
			So(err, ShouldBeNil)
			So(currDeletedInvitations.Invitations, ShouldBeNil)

			// Reject out-of-network invitations for recipient4
			_, err = client.RejectOutOfNetworkInvitations(ctx, &meepo.RejectOutOfNetworkInvitationsRequest{
				ChannelId: recipientID4,
				CallerId:  recipientID4,
			})
			So(err, ShouldBeNil)

			// Verify that oldest rejected inv has been deleted
			currDeletedInvitations, err = client.GetInvitationsBySquadID(ctx, &meepo.GetInvitationsBySquadIDRequest{
				SquadId:  squad.Id,
				CallerId: ownerID,
				Status:   meepo.Invitation_DELETED,
			})
			So(err, ShouldBeNil)
			So(currDeletedInvitations.Invitations, ShouldHaveLength, 1)
			So(currDeletedInvitations.Invitations[0].Id, ShouldEqual, rejectedInvitation1.Invitation.Id)

			// Verify that squad has 3 rejected invitations
			currRejectedInvitations, err = client.GetInvitationsBySquadID(ctx, &meepo.GetInvitationsBySquadIDRequest{
				SquadId:  squad.Id,
				CallerId: ownerID,
				Status:   meepo.Invitation_REJECTED,
			})
			So(err, ShouldBeNil)
			So(currRejectedInvitations.Invitations, ShouldHaveLength, 3)
		})

		Convey("Should publish the squads of rejected invitations to pubsub", func() {
			recipientID := util.NewUserID()

			// Set recipient's invite policy to ALL, so that we can create invitations that are out of network.
			_, err := client.UpdateInvitePolicyByChannelID(ctx, &meepo.UpdateInvitePolicyByChannelIDRequest{
				ChannelId:    recipientID,
				CallerId:     recipientID,
				InvitePolicy: meepo.InvitePolicy_ALL,
			})
			So(err, ShouldBeNil)
			// Create two out of network invitations to squads.
			ownerID1 := util.NewUserID()
			ownerID2 := util.NewUserID()
			for _, ownerID := range []string{ownerID1, ownerID2} {
				// owner is not in the network of the recipient
				followsClient.On("IsFollowing", mock.Anything, recipientID, ownerID).Maybe().Return(false, nil)
				friendshipClient.On("IsFriend", mock.Anything, recipientID, ownerID).Maybe().Return(false, nil)
				rosterClient.On("IsTeammate", mock.Anything, recipientID, ownerID).Maybe().Return(false, nil)
			}

			memberID1 := util.NewUserID()
			squad1 := CreateTestSquadWithMemberAndInvitation(ctx, t, client, injectables, ownerID1, memberID1, recipientID)
			members1 := []string{ownerID1, memberID1}

			memberID2 := util.NewUserID()
			squad2 := CreateTestSquadWithMemberAndInvitation(ctx, t, client, injectables, ownerID2, memberID2, recipientID)
			members2 := []string{ownerID2, memberID2}

			// Create an in-network invitation to a squad.
			ownerID3 := util.NewUserID()
			followsClient.On("IsFollowing", mock.Anything, recipientID, ownerID3).Maybe().Return(true, nil)
			friendshipClient.On("IsFriend", mock.Anything, recipientID, ownerID3).Maybe().Return(true, nil)
			rosterClient.On("IsTeammate", mock.Anything, recipientID, ownerID3).Maybe().Return(true, nil)

			memberID3 := util.NewUserID()
			squad3 := CreateTestSquadWithMemberAndInvitation(ctx, t, client, injectables, ownerID3, memberID3, recipientID)
			members3 := []string{ownerID3, memberID3}

			// Clear recorded calls, so they don't interfere with our assertions.
			pubsubClient.ClearCalls()

			// Reject out-of-network invitations for recipient
			req := &meepo.RejectOutOfNetworkInvitationsRequest{
				ChannelId: recipientID,
				CallerId:  recipientID,
			}
			_, err = client.RejectOutOfNetworkInvitations(ctx, req)
			So(err, ShouldBeNil)

			// We should have published updates for the squads that are out of network.
			pubsubClient.AssertSquadUpdatePublished(t, squad1.Id, models.SquadStatusPending, members1)
			pubsubClient.AssertSquadUpdatePublished(t, squad2.Id, models.SquadStatusPending, members2)

			// We should NOT have published updates for the squad that is in network.
			pubsubClient.AssertSquadUpdateNotPublished(t, squad3.Id, models.SquadStatusPending, members3)
		})
	})

	ts.onFinish(serverShutdownWaitTime)
}

func TestGetIncomingInvitationsCountByChannelID(t *testing.T) {

	injectables := newDefaultInjectables()
	ts := startServer(t, injectables, map[string][]byte{
		"meepo.squad_enabled_for_partners": []byte("true"),
	})
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	authorizer := (injectables.authorizer).(*mocks.Authorizer)
	ripleyClient := (injectables.ripleyClient).(*mocks.RipleyClient)
	usersClient := (injectables.usersClient).(*mocks.UsersClient)

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		client := ts.meepoClient
		ctx := ts.ctx

		authorizer.On("CanCreateInvitation", mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(true)
		authorizer.On("CanGetPendingInvitationsCountByRecipientID", mock.Anything, mock.Anything, mock.Anything).Return(true)
		ripleyClient.On("GetUserPayoutType", mock.Anything, mock.Anything).Return(
			&clients.UserPayoutType{
				IsPartner: true,
			}, nil)
		usersClient.On("ValidateUser", mock.Anything, mock.Anything).Return(true, nil, nil)

		Convey("No channel ID param should throw error", func() {
			req := &meepo.GetIncomingInvitationsCountByChannelIDRequest{CallerId: util.NewUserID()}
			_, err := client.GetIncomingInvitationsCountByChannelID(ctx, req)
			So(err, ShouldNotBeNil)
		})

		Convey("No caller ID param should throw error", func() {
			req := &meepo.GetIncomingInvitationsCountByChannelIDRequest{ChannelId: util.NewUserID()}
			_, err := client.GetIncomingInvitationsCountByChannelID(ctx, req)
			So(err, ShouldNotBeNil)
		})

		Convey("Should return 0 for channel with no incoming, pending invitations", func() {
			channelID := util.NewUserID()
			req := &meepo.GetIncomingInvitationsCountByChannelIDRequest{
				ChannelId: channelID,
				CallerId:  channelID,
			}

			resp, err := client.GetIncomingInvitationsCountByChannelID(ctx, req)
			So(err, ShouldBeNil)
			So(resp.Count, ShouldEqual, 0)
		})

		Convey("Should return count of incoming, pending invitations for channel", func() {
			channelID := util.NewUserID()

			numberOfIncomingInvitations := 5
			for i := 0; i < numberOfIncomingInvitations; i++ {
				senderID := util.NewUserID()
				res, err := client.CreateInvitation(
					ctx,
					&meepo.CreateInvitationRequest{
						SenderId:    senderID,
						CallerId:    senderID,
						RecipientId: channelID,
					},
				)
				So(err, ShouldBeNil)
				So(res, ShouldNotBeNil)
			}

			resp, err := client.GetIncomingInvitationsCountByChannelID(
				ctx,
				&meepo.GetIncomingInvitationsCountByChannelIDRequest{
					ChannelId: channelID,
					CallerId:  channelID,
				},
			)
			So(err, ShouldBeNil)
			So(resp.Count, ShouldEqual, numberOfIncomingInvitations)
		})
	})

	ts.onFinish(serverShutdownWaitTime)
}

func TestAdDeclines(t *testing.T) {

	injectables := newDefaultInjectables()
	ts := startServer(t, injectables)
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		client := ts.meepoClient
		ctx := ts.ctx

		deviceID := NewDeviceID()
		userID := util.NewUserID()

		Convey("SetPrimaryPlayer endpoint", func() {
			Convey("Missing deviceID should throw an error", func() {
				req := &meepo.SetPrimaryPlayerRequest{
					SquadId:                "test-squad",
					PrimaryPlayerChannelId: "primary-channel-id",
				}
				r, err := client.SetPrimaryPlayer(ctx, req)
				So(err, ShouldNotBeNil)
				So(r, ShouldBeNil)
			})

			Convey("Missing squadID should throw an error", func() {
				req := &meepo.SetPrimaryPlayerRequest{
					DeviceId:               deviceID,
					PrimaryPlayerChannelId: "primary-channel-id",
				}
				r, err := client.SetPrimaryPlayer(ctx, req)
				So(err, ShouldNotBeNil)
				So(r, ShouldBeNil)
			})

			Convey("Missing primaryPlayerChannelId should throw an error", func() {
				req := &meepo.SetPrimaryPlayerRequest{
					SquadId:  "test-squad",
					DeviceId: deviceID,
				}
				r, err := client.SetPrimaryPlayer(ctx, req)
				So(err, ShouldNotBeNil)
				So(r, ShouldBeNil)
			})

			Convey("Setting a primary player ID should not throw an error", func() {
				req := &meepo.SetPrimaryPlayerRequest{
					DeviceId:               deviceID,
					SquadId:                "test-squad",
					PrimaryPlayerChannelId: "primary-channel-id",
				}
				r, err := client.SetPrimaryPlayer(ctx, req)
				So(err, ShouldBeNil)
				So(r, ShouldNotBeNil)
				So(r.PrimaryPlayerChannelId, ShouldEqual, req.PrimaryPlayerChannelId)
			})
		})

		Convey("ShouldShowAd endpoint", func() {
			Convey("Missing channelID should throw an error", func() {
				req := &meepo.ShouldShowAdRequest{
					AdType:   meepo.ShouldShowAdRequest_PRE_ROLL,
					DeviceId: deviceID,
					UserId:   userID,
				}
				req.PlayerType = meepo.ShouldShowAdRequest_PRIMARY
				r, err := client.ShouldShowAd(ctx, req)
				So(err, ShouldNotBeNil)
				So(r, ShouldBeNil)
			})

			Convey("Missing deviceID should show a mid-roll ad", func() {
				req := &meepo.ShouldShowAdRequest{
					AdType:    meepo.ShouldShowAdRequest_MID_ROLL,
					ChannelId: userID,
					UserId:    userID,
				}
				req.PlayerType = meepo.ShouldShowAdRequest_SECONDARY
				r, err := client.ShouldShowAd(ctx, req)
				So(err, ShouldBeNil)
				So(r, ShouldNotBeNil)
				So(r.ShowAd, ShouldBeTrue)
			})

			Convey("Missing deviceID should show a post-roll ad", func() {
				req := &meepo.ShouldShowAdRequest{
					AdType:    meepo.ShouldShowAdRequest_POST_ROLL,
					ChannelId: userID,
					UserId:    userID,
				}
				req.PlayerType = meepo.ShouldShowAdRequest_SECONDARY
				r, err := client.ShouldShowAd(ctx, req)
				So(err, ShouldBeNil)
				So(r, ShouldNotBeNil)
				So(r.ShowAd, ShouldBeTrue)
			})

			Convey("A pre-roll ad on a non-live squad ID", func() {
				req := &meepo.ShouldShowAdRequest{
					AdType:    meepo.ShouldShowAdRequest_PRE_ROLL,
					ChannelId: userID,
					DeviceId:  deviceID,
					UserId:    userID,
				}
				Convey("Should show ads for primary player", func() {
					req.PlayerType = meepo.ShouldShowAdRequest_PRIMARY
					r, err := client.ShouldShowAd(ctx, req)
					So(err, ShouldBeNil)
					So(r, ShouldNotBeNil)
					So(r.ShowAd, ShouldBeTrue)
				})
				Convey("Should show ads for secondary player", func() {
					req.PlayerType = meepo.ShouldShowAdRequest_SECONDARY
					r, err := client.ShouldShowAd(ctx, req)
					So(err, ShouldBeNil)
					So(r, ShouldNotBeNil)
					So(r.ShowAd, ShouldBeTrue)

				})
			})

			Convey("A pre-roll ad on a live squad ID", func() {
				squadOwnerID := util.NewUserID()
				squadMemberID := util.NewUserID()

				squad := CreateTestLiveSquadWithMember(ctx, t, client, injectables, squadOwnerID, squadMemberID)
				So(squad, ShouldNotBeNil)

				req := &meepo.ShouldShowAdRequest{
					AdType:    meepo.ShouldShowAdRequest_PRE_ROLL,
					ChannelId: squadOwnerID,
					DeviceId:  deviceID,
					UserId:    userID,
				}
				Convey("Should show ad for primary player", func() {
					req.PlayerType = meepo.ShouldShowAdRequest_PRIMARY
					r, err := client.ShouldShowAd(ctx, req)
					So(err, ShouldBeNil)
					So(r, ShouldNotBeNil)
					So(r.ShowAd, ShouldBeTrue)
				})
				Convey("Should not show ad for secondary player", func() {
					req.PlayerType = meepo.ShouldShowAdRequest_SECONDARY
					r, err := client.ShouldShowAd(ctx, req)
					So(err, ShouldBeNil)
					So(r, ShouldNotBeNil)
					So(r.ShowAd, ShouldBeFalse)
				})

				Convey("Should not show ad for secondary player with missing deviceID", func() {
					req.PlayerType = meepo.ShouldShowAdRequest_SECONDARY
					req.DeviceId = ""
					r, err := client.ShouldShowAd(ctx, req)
					So(err, ShouldBeNil)
					So(r, ShouldNotBeNil)
					So(r.ShowAd, ShouldBeFalse)
				})
			})

			Convey("A mid-roll ad on a non-live squad ID", func() {
				req := &meepo.ShouldShowAdRequest{
					AdType:    meepo.ShouldShowAdRequest_MID_ROLL,
					ChannelId: userID,
					DeviceId:  deviceID,
					UserId:    userID,
				}
				Convey("Should show ads for primary player", func() {
					req.PlayerType = meepo.ShouldShowAdRequest_PRIMARY
					r, err := client.ShouldShowAd(ctx, req)
					So(err, ShouldBeNil)
					So(r, ShouldNotBeNil)
					So(r.ShowAd, ShouldBeTrue)
				})
				Convey("Should show ads for secondary player", func() {
					req.PlayerType = meepo.ShouldShowAdRequest_SECONDARY
					r, err := client.ShouldShowAd(ctx, req)
					So(err, ShouldBeNil)
					So(r, ShouldNotBeNil)
					So(r.ShowAd, ShouldBeTrue)

				})
			})

			Convey("A mid-roll ad on a live squad ID", func() {
				squadOwnerID := util.NewUserID()
				squadMemberID := util.NewUserID()

				squad := CreateTestLiveSquadWithMember(ctx, t, client, injectables, squadOwnerID, squadMemberID)
				So(squad, ShouldNotBeNil)

				res, err := client.SetPrimaryPlayer(ctx, &meepo.SetPrimaryPlayerRequest{
					DeviceId:               deviceID,
					SquadId:                squad.Id,
					PrimaryPlayerChannelId: squadOwnerID,
				})
				So(err, ShouldBeNil)
				So(res, ShouldNotBeNil)
				So(res.PrimaryPlayerChannelId, ShouldEqual, squadOwnerID)

				req := &meepo.ShouldShowAdRequest{
					AdType:   meepo.ShouldShowAdRequest_MID_ROLL,
					DeviceId: deviceID,
					UserId:   userID,
				}

				Convey("Should show ad for primary player", func() {
					req.ChannelId = squadOwnerID
					r, err := client.ShouldShowAd(ctx, req)
					So(err, ShouldBeNil)
					So(r, ShouldNotBeNil)
					So(r.ShowAd, ShouldBeTrue)

					Convey("Should not show ad when user switches primary player", func() {
						res, err := client.SetPrimaryPlayer(ctx, &meepo.SetPrimaryPlayerRequest{
							DeviceId:               deviceID,
							SquadId:                squad.Id,
							PrimaryPlayerChannelId: squadMemberID,
						})
						So(err, ShouldBeNil)
						So(res, ShouldNotBeNil)
						So(res.PrimaryPlayerChannelId, ShouldEqual, squadMemberID)

						req.ChannelId = squadOwnerID
						r, err := client.ShouldShowAd(ctx, req)
						So(err, ShouldBeNil)
						So(r, ShouldNotBeNil)
						So(r.ShowAd, ShouldBeFalse)
					})
				})
				Convey("Should not show ad for secondary player", func() {
					req.ChannelId = squadMemberID
					r, err := client.ShouldShowAd(ctx, req)
					So(err, ShouldBeNil)
					So(r, ShouldNotBeNil)
					So(r.ShowAd, ShouldBeFalse)

					Convey("Should show ad when user switches secondary player to primary player", func() {
						res, err := client.SetPrimaryPlayer(ctx, &meepo.SetPrimaryPlayerRequest{
							DeviceId:               deviceID,
							SquadId:                squad.Id,
							PrimaryPlayerChannelId: squadMemberID,
						})
						So(err, ShouldBeNil)
						So(res, ShouldNotBeNil)
						So(res.PrimaryPlayerChannelId, ShouldEqual, squadMemberID)

						req.ChannelId = squadMemberID
						r, err := client.ShouldShowAd(ctx, req)
						So(err, ShouldBeNil)
						So(r, ShouldNotBeNil)
						So(r.ShowAd, ShouldBeTrue)
					})
				})
			})

			Convey("Should show mid-roll when there is no entry in cache for deviceID & squadID", func() {
				squadOwnerID := util.NewUserID()
				squadMemberID := util.NewUserID()

				squad := CreateTestLiveSquadWithMember(ctx, t, client, injectables, squadOwnerID, squadMemberID)
				So(squad, ShouldNotBeNil)

				req := &meepo.ShouldShowAdRequest{
					AdType:    meepo.ShouldShowAdRequest_MID_ROLL,
					DeviceId:  deviceID,
					UserId:    userID,
					ChannelId: squadOwnerID,
				}

				r, err := client.ShouldShowAd(ctx, req)
				So(err, ShouldBeNil)
				So(r, ShouldNotBeNil)
				So(r.ShowAd, ShouldBeTrue)
			})

			Convey("A post-roll ad on a non-live squad ID", func() {
				req := &meepo.ShouldShowAdRequest{
					AdType:    meepo.ShouldShowAdRequest_POST_ROLL,
					ChannelId: userID,
					DeviceId:  deviceID,
					UserId:    userID,
				}
				Convey("Should show ads for primary player", func() {
					req.PlayerType = meepo.ShouldShowAdRequest_PRIMARY
					r, err := client.ShouldShowAd(ctx, req)
					So(err, ShouldBeNil)
					So(r, ShouldNotBeNil)
					So(r.ShowAd, ShouldBeTrue)
				})
				Convey("Should show ads for secondary player", func() {
					req.PlayerType = meepo.ShouldShowAdRequest_SECONDARY
					r, err := client.ShouldShowAd(ctx, req)
					So(err, ShouldBeNil)
					So(r, ShouldNotBeNil)
					So(r.ShowAd, ShouldBeTrue)

				})
			})

			Convey("A post-roll ad on a live squad ID", func() {
				squadOwnerID := util.NewUserID()
				squadMemberID := util.NewUserID()

				squad := CreateTestLiveSquadWithMember(ctx, t, client, injectables, squadOwnerID, squadMemberID)
				So(squad, ShouldNotBeNil)

				res, err := client.SetPrimaryPlayer(ctx, &meepo.SetPrimaryPlayerRequest{
					DeviceId:               deviceID,
					SquadId:                squad.Id,
					PrimaryPlayerChannelId: squadOwnerID,
				})
				So(err, ShouldBeNil)
				So(res, ShouldNotBeNil)
				So(res.PrimaryPlayerChannelId, ShouldEqual, squadOwnerID)

				req := &meepo.ShouldShowAdRequest{
					AdType:   meepo.ShouldShowAdRequest_POST_ROLL,
					DeviceId: deviceID,
					UserId:   userID,
				}

				Convey("Should show ad for primary player", func() {
					req.ChannelId = squadOwnerID
					r, err := client.ShouldShowAd(ctx, req)
					So(err, ShouldBeNil)
					So(r, ShouldNotBeNil)
					So(r.ShowAd, ShouldBeTrue)

					Convey("Should not show ad when user switches primary player", func() {
						res, err := client.SetPrimaryPlayer(ctx, &meepo.SetPrimaryPlayerRequest{
							DeviceId:               deviceID,
							SquadId:                squad.Id,
							PrimaryPlayerChannelId: squadMemberID,
						})
						So(err, ShouldBeNil)
						So(res, ShouldNotBeNil)
						So(res.PrimaryPlayerChannelId, ShouldEqual, squadMemberID)

						req.ChannelId = squadOwnerID
						r, err := client.ShouldShowAd(ctx, req)
						So(err, ShouldBeNil)
						So(r, ShouldNotBeNil)
						So(r.ShowAd, ShouldBeFalse)
					})
				})
				Convey("Should not show ad for secondary player", func() {
					req.ChannelId = squadMemberID
					r, err := client.ShouldShowAd(ctx, req)
					So(err, ShouldBeNil)
					So(r, ShouldNotBeNil)
					So(r.ShowAd, ShouldBeFalse)

					Convey("Should show ad when user switches secondary player to primary player", func() {
						res, err := client.SetPrimaryPlayer(ctx, &meepo.SetPrimaryPlayerRequest{
							DeviceId:               deviceID,
							SquadId:                squad.Id,
							PrimaryPlayerChannelId: squadMemberID,
						})
						So(err, ShouldBeNil)
						So(res, ShouldNotBeNil)
						So(res.PrimaryPlayerChannelId, ShouldEqual, squadMemberID)

						req.ChannelId = squadMemberID
						r, err := client.ShouldShowAd(ctx, req)
						So(err, ShouldBeNil)
						So(r, ShouldNotBeNil)
						So(r.ShowAd, ShouldBeTrue)
					})
				})
			})

			Convey("Should show post-roll when there is no entry in cache for deviceID & squadID", func() {
				squadOwnerID := util.NewUserID()
				squadMemberID := util.NewUserID()

				squad := CreateTestLiveSquadWithMember(ctx, t, client, injectables, squadOwnerID, squadMemberID)
				So(squad, ShouldNotBeNil)

				req := &meepo.ShouldShowAdRequest{
					AdType:    meepo.ShouldShowAdRequest_POST_ROLL,
					DeviceId:  deviceID,
					UserId:    userID,
					ChannelId: squadOwnerID,
				}

				r, err := client.ShouldShowAd(ctx, req)
				So(err, ShouldBeNil)
				So(r, ShouldNotBeNil)
				So(r.ShowAd, ShouldBeTrue)
			})
		})

	})

	ts.onFinish(serverShutdownWaitTime)
}

func TestSquadAccess(t *testing.T) {
	unwhitelistedUserID := util.NewUserID()
	whitelistedUserID := util.NewUserID()

	injectables := newDefaultInjectables()
	ts := startServer(t, injectables, map[string][]byte{
		"meepo.squad_enabled_whitelist":      []byte(whitelistedUserID),
		"meepo.squad_enabled_for_affiliates": []byte("true"),
		"meepo.squad_enabled_for_partners":   []byte("true"),
	})
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	ripleyClient := (injectables.ripleyClient).(*mocks.RipleyClient)
	usersClient := (injectables.usersClient).(*mocks.UsersClient)

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		client := ts.meepoClient
		ctx := ts.ctx

		usersClient.On("ValidateUser", mock.Anything, mock.Anything).Return(true, nil, nil)

		Convey("CanAccessSquads endpoint", func() {
			Convey("Should allow access for a user on the whitelist", func() {
				ripleyClient.On("GetUserPayoutType", mock.Anything, mock.Anything).Once().Return(
					&clients.UserPayoutType{}, nil)

				canAccessSquadsReq := &meepo.CanAccessSquadsRequest{
					ChannelId: unwhitelistedUserID,
				}
				canAccessSquadsResp, err := client.CanAccessSquads(ctx, canAccessSquadsReq)
				So(err, ShouldBeNil)
				So(canAccessSquadsResp.CanAccessSquads, ShouldBeFalse)

				canAccessSquadsReq = &meepo.CanAccessSquadsRequest{
					ChannelId: whitelistedUserID,
				}
				canAccessSquadsResp, err = client.CanAccessSquads(ctx, canAccessSquadsReq)
				So(err, ShouldBeNil)
				So(canAccessSquadsResp.CanAccessSquads, ShouldBeTrue)
			})

			Convey("Should allow access for a partner, if partner access is enabled", func() {
				ripleyClient.On("GetUserPayoutType", mock.Anything, mock.Anything).Once().Return(
					&clients.UserPayoutType{
						IsPartner: true,
					}, nil)

				canAccessSquadsReq := &meepo.CanAccessSquadsRequest{
					ChannelId: unwhitelistedUserID,
				}
				canAccessSquadsResp, err := client.CanAccessSquads(ctx, canAccessSquadsReq)
				So(err, ShouldBeNil)
				So(canAccessSquadsResp.CanAccessSquads, ShouldBeTrue)
			})

			Convey("Should allow access for an affiliate, if affiliate access is enabled", func() {
				ripleyClient.On("GetUserPayoutType", mock.Anything, mock.Anything).Once().Return(
					&clients.UserPayoutType{
						IsAffiliate: true,
					}, nil)

				canAccessSquadsReq := &meepo.CanAccessSquadsRequest{
					ChannelId: unwhitelistedUserID,
				}
				canAccessSquadsResp, err := client.CanAccessSquads(ctx, canAccessSquadsReq)
				So(err, ShouldBeNil)
				So(canAccessSquadsResp.CanAccessSquads, ShouldBeTrue)
			})
		})
	})

	injectables = newDefaultInjectables()
	ts = startServer(t, injectables, map[string][]byte{
		"meepo.squad_enabled_for_affiliates": []byte("false"),
		"meepo.squad_enabled_for_partners":   []byte("false"),
	})
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	ripleyClient = (injectables.ripleyClient).(*mocks.RipleyClient)
	usersClient = (injectables.usersClient).(*mocks.UsersClient)

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		client := ts.meepoClient
		ctx := ts.ctx

		usersClient.On("ValidateUser", mock.Anything, mock.Anything).Return(true, nil, nil)

		Convey("CanAccessSquads endpoint", func() {
			Convey("Should deny access for a partner, if partner access is disabled", func() {
				ripleyClient.On("GetUserPayoutType", mock.Anything, mock.Anything).Once().Return(
					&clients.UserPayoutType{
						IsPartner: true,
					}, nil)

				canAccessSquadsReq := &meepo.CanAccessSquadsRequest{
					ChannelId: unwhitelistedUserID,
				}
				canAccessSquadsResp, err := client.CanAccessSquads(ctx, canAccessSquadsReq)
				So(err, ShouldBeNil)
				So(canAccessSquadsResp.CanAccessSquads, ShouldBeFalse)
			})

			Convey("Should deny access for an affiliate, if affiliate access is disabled", func() {
				ripleyClient.On("GetUserPayoutType", mock.Anything, mock.Anything).Once().Return(
					&clients.UserPayoutType{
						IsAffiliate: true,
					}, nil)

				canAccessSquadsReq := &meepo.CanAccessSquadsRequest{
					ChannelId: unwhitelistedUserID,
				}
				canAccessSquadsResp, err := client.CanAccessSquads(ctx, canAccessSquadsReq)
				So(err, ShouldBeNil)
				So(canAccessSquadsResp.CanAccessSquads, ShouldBeFalse)
			})
		})
	})

	ts.onFinish(serverShutdownWaitTime)
}

// TestChannelStatePublisherClient verifies that we wired up channelStatePublisherClient correctly.
func TestChannelStatePublisherClient(t *testing.T) {

	injectables := newDefaultInjectables()
	// Use a real ChannelStatePublisherClient
	injectables.channelStatePublisherClient = nil

	ts := startServer(t, injectables)
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		ctx := ts.ctx

		Convey("channelStatePublisherClient.PublishChannelIsInLiveSquad should not panic", func() {
			ts.thisInstance.channelStatePublisherClient.PublishChannelIsInLiveSquad(ctx, "squadid", util.NewUserID())
		})

		Convey("channelStatePublisherClient.PublishChannelOutOfLiveSquad should not panic", func() {
			ts.thisInstance.channelStatePublisherClient.PublishChannelOutOfLiveSquad(ctx, util.NewUserID())
		})

		Convey("channelStatePublisherClient.PublishChannelsInLiveSquad should not panic", func() {
			ts.thisInstance.channelStatePublisherClient.PublishChannelsInLiveSquad(ctx, "squadid", []string{util.NewUserID(), util.NewUserID()})
		})
	})

	ts.onFinish(serverShutdownWaitTime)
}

func TestInvitePolicy(t *testing.T) {

	injectables := newDefaultInjectables()

	followsClient := &mocks.FollowsClient{}
	friendshipClient := &mocks.FriendshipClient{}
	rosterClient := &mocks.RosterClient{}

	injectables.followsClient = followsClient
	injectables.friendshipClient = friendshipClient
	injectables.rosterClient = rosterClient

	adminID := util.NewUserID()

	ts := startServer(t, injectables, map[string][]byte{
		"meepo.admin_users": []byte(adminID),
	})
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	authorizer := (injectables.authorizer).(*mocks.Authorizer)
	ripleyClient := (injectables.ripleyClient).(*mocks.RipleyClient)
	usersClient := (injectables.usersClient).(*mocks.UsersClient)

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		client := ts.meepoClient
		ctx := ts.ctx

		usersClient.On("ValidateUser", mock.Anything, mock.Anything).Return(true, nil, nil)
		ripleyClient.On("GetUserPayoutType", mock.Anything, mock.Anything).Return(
			&clients.UserPayoutType{
				IsPartner: true,
			}, nil)
		authorizer.On("CanCreateInvitation", mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(true)

		Convey("GetInvitePolicyByChannelID endpoint", func() {
			Convey("Should fetch a policy of NETWORK for a user whose policy has not been set", func() {
				channelID := util.NewUserID()
				res, err := client.GetInvitePolicyByChannelID(ctx, &meepo.GetInvitePolicyByChannelIDRequest{
					ChannelId: channelID,
					CallerId:  channelID,
				})
				So(err, ShouldBeNil)
				So(res, ShouldNotBeNil)
				So(res.InvitePolicy, ShouldEqual, meepo.InvitePolicy_NETWORK)
			})
			Convey("Should fetch if the callerID is an admin", func() {
				channelID := util.NewUserID()
				res, err := client.GetInvitePolicyByChannelID(ctx, &meepo.GetInvitePolicyByChannelIDRequest{
					ChannelId: channelID,
					CallerId:  adminID,
				})
				So(err, ShouldBeNil)
				So(res, ShouldNotBeNil)
			})
			Convey("Should NOT fetch if the callerID differs from the channelID and is not an admin", func() {
				channelID := util.NewUserID()
				differentID := util.NewUserID()
				res, err := client.GetInvitePolicyByChannelID(ctx, &meepo.GetInvitePolicyByChannelIDRequest{
					ChannelId: channelID,
					CallerId:  differentID,
				})
				So(err, ShouldNotBeNil)
				So(res, ShouldBeNil)
			})
		})

		Convey("UpdateInvitePolicyByChannelID endpoint", func() {
			Convey("Should be able to update the invite policy for a user", func() {
				channelID := util.NewUserID()
				res, err := client.UpdateInvitePolicyByChannelID(ctx, &meepo.UpdateInvitePolicyByChannelIDRequest{
					ChannelId:    channelID,
					CallerId:     channelID,
					InvitePolicy: meepo.InvitePolicy_ALL,
				})
				So(err, ShouldBeNil)
				So(res, ShouldNotBeNil)
				So(res.InvitePolicy, ShouldEqual, meepo.InvitePolicy_ALL)

				res2, err := client.GetInvitePolicyByChannelID(ctx, &meepo.GetInvitePolicyByChannelIDRequest{
					ChannelId: channelID,
					CallerId:  channelID,
				})
				So(err, ShouldBeNil)
				So(res2, ShouldNotBeNil)
				So(res2.InvitePolicy, ShouldEqual, meepo.InvitePolicy_ALL)
			})
			Convey("Should update if the callerID is an admin", func() {
				channelID := util.NewUserID()
				res, err := client.UpdateInvitePolicyByChannelID(ctx, &meepo.UpdateInvitePolicyByChannelIDRequest{
					ChannelId:    channelID,
					CallerId:     adminID,
					InvitePolicy: meepo.InvitePolicy_NONE,
				})
				So(err, ShouldBeNil)
				So(res, ShouldNotBeNil)
			})
			Convey("Should NOT update if the callerID differs from the channelID and is not an admin", func() {
				channelID := util.NewUserID()
				differentID := util.NewUserID()
				res, err := client.UpdateInvitePolicyByChannelID(ctx, &meepo.UpdateInvitePolicyByChannelIDRequest{
					ChannelId:    channelID,
					CallerId:     differentID,
					InvitePolicy: meepo.InvitePolicy_NONE,
				})
				So(err, ShouldNotBeNil)
				So(res, ShouldBeNil)
			})
		})

		Convey("ChannelCanInviteUser function", func() {
			Convey("Should not be able to invite a recipient with a policy of NONE", func() {
				ownerID := util.NewUserID()
				recipientID := util.NewUserID()

				client.UpdateInvitePolicyByChannelID(ctx, &meepo.UpdateInvitePolicyByChannelIDRequest{
					ChannelId:    recipientID,
					CallerId:     recipientID,
					InvitePolicy: meepo.InvitePolicy_NONE,
				})

				req := &meepo.CreateInvitationRequest{
					SenderId:    ownerID,
					CallerId:    ownerID,
					RecipientId: recipientID,
				}

				res, err := client.CreateInvitation(ctx, req)
				So(err, ShouldNotBeNil)
				So(res, ShouldBeNil)
			})
			Convey("Should be able to invite a recipient with a policy of ALL", func() {
				ownerID := util.NewUserID()
				recipientID := util.NewUserID()

				followsClient.On("IsFollowing", mock.Anything, mock.Anything, mock.Anything).Once().Return(false, nil)
				friendshipClient.On("IsFriend", mock.Anything, mock.Anything, mock.Anything).Once().Return(false, nil)
				rosterClient.On("IsTeammate", mock.Anything, mock.Anything, mock.Anything).Once().Return(false, nil)

				client.UpdateInvitePolicyByChannelID(ctx, &meepo.UpdateInvitePolicyByChannelIDRequest{
					ChannelId:    recipientID,
					CallerId:     recipientID,
					InvitePolicy: meepo.InvitePolicy_ALL,
				})

				req := &meepo.CreateInvitationRequest{
					SenderId:    ownerID,
					CallerId:    ownerID,
					RecipientId: recipientID,
				}

				res, err := client.CreateInvitation(ctx, req)
				So(err, ShouldBeNil)
				So(res, ShouldNotBeNil)
			})
			Convey("Should be able to invite a recipient with a policy of NETWORK if recipient follows sender", func() {
				ownerID := util.NewUserID()
				recipientID := util.NewUserID()

				followsClient.On("IsFollowing", mock.Anything, recipientID, ownerID).Maybe().Return(true, nil)
				friendshipClient.On("IsFriend", mock.Anything, recipientID, ownerID).Maybe().Return(false, nil)
				rosterClient.On("IsTeammate", mock.Anything, recipientID, ownerID).Maybe().Return(false, nil)

				res, err := client.CreateInvitation(
					ctx,
					&meepo.CreateInvitationRequest{
						SenderId:    ownerID,
						CallerId:    ownerID,
						RecipientId: recipientID,
					},
				)
				So(err, ShouldBeNil)
				So(res, ShouldNotBeNil)
			})
			Convey("Should be able to invite a recipient with a policy of NETWORK if recipient is friends with sender", func() {
				ownerID := util.NewUserID()
				recipientID := util.NewUserID()

				followsClient.On("IsFollowing", mock.Anything, recipientID, ownerID).Maybe().Return(false, nil)
				friendshipClient.On("IsFriend", mock.Anything, recipientID, ownerID).Maybe().Return(true, nil)
				rosterClient.On("IsTeammate", mock.Anything, recipientID, ownerID).Maybe().Return(false, nil)

				res, err := client.CreateInvitation(
					ctx,
					&meepo.CreateInvitationRequest{
						SenderId:    ownerID,
						CallerId:    ownerID,
						RecipientId: recipientID,
					},
				)
				So(err, ShouldBeNil)
				So(res, ShouldNotBeNil)
			})
			Convey("Should be able to invite a recipient with a policy of NETWORK if recipient is teammates with sender", func() {
				ownerID := util.NewUserID()
				recipientID := util.NewUserID()

				followsClient.On("IsFollowing", mock.Anything, recipientID, ownerID).Maybe().Return(false, nil)
				friendshipClient.On("IsFriend", mock.Anything, recipientID, ownerID).Maybe().Return(false, nil)
				rosterClient.On("IsTeammate", mock.Anything, recipientID, ownerID).Maybe().Return(true, nil)

				res, err := client.CreateInvitation(
					ctx,
					&meepo.CreateInvitationRequest{
						SenderId:    ownerID,
						CallerId:    ownerID,
						RecipientId: recipientID,
					},
				)
				So(err, ShouldBeNil)
				So(res, ShouldNotBeNil)
			})
			Convey("Should not be able to invite a recipient with a policy of NETWORK if recipient is neither follower, friend, nor teammate", func() {
				ownerID := util.NewUserID()
				recipientID := util.NewUserID()

				followsClient.On("IsFollowing", mock.Anything, recipientID, ownerID).Maybe().Return(false, nil)
				friendshipClient.On("IsFriend", mock.Anything, recipientID, ownerID).Maybe().Return(false, nil)
				rosterClient.On("IsTeammate", mock.Anything, recipientID, ownerID).Maybe().Return(false, nil)

				res, err := client.CreateInvitation(
					ctx,
					&meepo.CreateInvitationRequest{
						SenderId:    ownerID,
						CallerId:    ownerID,
						RecipientId: recipientID,
					},
				)
				So(err, ShouldNotBeNil)
				So(res, ShouldBeNil)
			})
		})
	})

	ts.onFinish(serverShutdownWaitTime)
}

func TestSpadeTracking(t *testing.T) {

	controlledTime := time.Date(2017, 01, 01, 01, 02, 03, 00, time.UTC)

	adminUser := util.NewUserID()
	injectables := newDefaultInjectables()

	spadeClient := &mocks.SpadeClient{}
	livelineClient := &stubs.LivelineStub{}
	clock := &stubs.ClockStub{}
	clock.SetNow(controlledTime)

	injectables.spadeClient = spadeClient
	injectables.livelineClient = livelineClient
	injectables.clock = clock

	ts := startServer(t, injectables, map[string][]byte{
		"meepo.admin_users":                      []byte(adminUser),
		"meepo.livecheck_new_squad_grace_period": []byte("0m"),
	})
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	authorizer := (injectables.authorizer).(*mocks.Authorizer)
	ripleyClient := (injectables.ripleyClient).(*mocks.RipleyClient)
	usersClient := (injectables.usersClient).(*mocks.UsersClient)

	ripleyClient.On("GetUserPayoutType", mock.Anything, mock.Anything).Return(
		&clients.UserPayoutType{
			IsPartner: true,
		}, nil)
	usersClient.On("ValidateUser", mock.Anything, mock.Anything).Return(true, nil, nil)
	spadeClient.On("TrackEvent", mock.Anything, mock.Anything, mock.Anything).Return(nil)
	authorizer.On("CanAcceptInvitation", mock.Anything, mock.Anything, mock.Anything).Return(true)
	authorizer.On("CanCreateInvitation", mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(true)
	authorizer.On("CanDeleteInvitation", mock.Anything, mock.Anything, mock.Anything).Return(true)
	authorizer.On("CanLeaveSquad", mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(true)
	authorizer.On("CanRemoveMember", mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(true)
	authorizer.On("CanUpdateSquad", mock.Anything, mock.Anything, mock.Anything).Return(true)
	authorizer.On("CanRejectInvitation", mock.Anything, mock.Anything, mock.Anything).Return(true)

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		internalClient := ts.internalMeepoClient
		client := ts.meepoClient
		ctx := context.Background()

		// reset liveline stub
		livelineClient.ClearLiveChannels()

		Convey("Spade event should fire on invitation methods", func() {
			Convey("Should fire on accept_invitation if a user is live and the squad has not yet started", func() {
				senderID := util.NewUserID()
				recipientID := util.NewUserID()
				intRecipientID, err := strconv.Atoi(recipientID)
				So(err, ShouldBeNil)

				livelineClient.SetLiveChannels([]string{recipientID})
				squad := CreateTestSquad(ctx, t, client, injectables, senderID)

				createInvitationReq := &meepo.CreateInvitationRequest{
					SenderId:    senderID,
					RecipientId: recipientID,
					CallerId:    senderID,
				}
				i, err := client.CreateInvitation(ctx, createInvitationReq)
				So(err, ShouldBeNil)
				So(i, ShouldNotBeNil)
				So(i.Invitation, ShouldNotBeNil)

				acceptReq := &meepo.AcceptInvitationRequest{
					InvitationId: i.Invitation.Id,
					CallerId:     recipientID,
				}
				_, err = client.AcceptInvitation(ctx, acceptReq)
				So(err, ShouldBeNil)

				spadeClient.AssertCalled(t, "TrackEvent", mock.Anything, models.CreatorActionTrackingEvent, models.CreatorActionTracking{
					ChannelID:     intRecipientID,
					Channel:       "user" + recipientID,
					NumMembers:    2,
					SquadStreamID: squad.Id,
					IsLive:        true,
					IsOwner:       false,
					SquadViewable: false,
					InviteID:      i.Invitation.Id,
					Action:        models.CreatorActionTypeAcceptInvite,
					Method:        models.CreatorMethodTypeClickAccept,
				})
			})

			Convey("Should fire on create_invitation", func() {
				senderID := util.NewUserID()
				intSenderID, err := strconv.Atoi(senderID)
				So(err, ShouldBeNil)

				recipientID := util.NewUserID()
				squad := CreateTestSquad(ctx, t, client, injectables, senderID)

				createInvitationReq := &meepo.CreateInvitationRequest{
					SenderId:    senderID,
					RecipientId: recipientID,
					CallerId:    senderID,
				}
				i, err := client.CreateInvitation(ctx, createInvitationReq)
				So(err, ShouldBeNil)
				So(i, ShouldNotBeNil)
				So(i.Invitation, ShouldNotBeNil)

				spadeClient.AssertCalled(t, "TrackEvent", mock.Anything, models.CreatorActionTrackingEvent, models.CreatorActionTracking{
					ChannelID:     intSenderID,
					Channel:       "user" + senderID,
					NumMembers:    1,
					SquadStreamID: squad.Id,
					IsLive:        false,
					IsOwner:       true,
					SquadViewable: false,
					InviteID:      i.Invitation.Id,
					Action:        models.CreatorActionTypeSendInvite,
					Method:        "",
				})
			})

			Convey("Should fire on remove_invitation", func() {
				senderID := util.NewUserID()
				intSenderID, err := strconv.Atoi(senderID)
				So(err, ShouldBeNil)

				recipientID := util.NewUserID()
				squad := CreateTestSquad(ctx, t, client, injectables, senderID)

				createInvitationReq := &meepo.CreateInvitationRequest{
					SenderId:    senderID,
					RecipientId: recipientID,
					CallerId:    senderID,
				}
				i, err := client.CreateInvitation(ctx, createInvitationReq)
				So(err, ShouldBeNil)
				So(i, ShouldNotBeNil)
				So(i.Invitation, ShouldNotBeNil)

				removeReq := &meepo.DeleteInvitationRequest{
					InvitationId: i.Invitation.Id,
					CallerId:     senderID,
				}
				_, err = client.DeleteInvitation(ctx, removeReq)
				So(err, ShouldBeNil)

				spadeClient.AssertCalled(t, "TrackEvent", mock.Anything, models.CreatorActionTrackingEvent, models.CreatorActionTracking{
					ChannelID:     intSenderID,
					Channel:       "user" + senderID,
					NumMembers:    1,
					SquadStreamID: squad.Id,
					IsLive:        false,
					IsOwner:       true,
					SquadViewable: false,
					InviteID:      i.Invitation.Id,
					Action:        models.CreatorActionTypeCancelInvite,
					Method:        models.CreatorMethodTypeOwnerClickCancel,
				})
			})

			Convey("Should fire on reject_invitation", func() {
				senderID := util.NewUserID()
				recipientID := util.NewUserID()
				intRecipientID, err := strconv.Atoi(recipientID)
				So(err, ShouldBeNil)

				squad := CreateTestSquad(ctx, t, client, injectables, senderID)

				createInvitationReq := &meepo.CreateInvitationRequest{
					SenderId:    senderID,
					RecipientId: recipientID,
					CallerId:    senderID,
				}
				i, err := client.CreateInvitation(ctx, createInvitationReq)
				So(err, ShouldBeNil)
				So(i, ShouldNotBeNil)
				So(i.Invitation, ShouldNotBeNil)

				rejectReq := &meepo.RejectInvitationRequest{
					InvitationId: i.Invitation.Id,
					CallerId:     recipientID,
				}
				_, err = client.RejectInvitation(ctx, rejectReq)
				So(err, ShouldBeNil)

				spadeClient.AssertCalled(t, "TrackEvent", mock.Anything, models.CreatorActionTrackingEvent, models.CreatorActionTracking{
					ChannelID:     intRecipientID,
					Channel:       "user" + recipientID,
					NumMembers:    1,
					SquadStreamID: squad.Id,
					IsLive:        false,
					IsOwner:       false,
					SquadViewable: false,
					InviteID:      i.Invitation.Id,
					Action:        models.CreatorActionTypeRejectInvite,
					Method:        "",
				})
			})

			Convey("Should fire on start_squad, sending a join_squad event for every member", func() {
				ownerID := util.NewUserID()
				intOwnerID, err := strconv.Atoi(ownerID)
				So(err, ShouldBeNil)

				memberID := util.NewUserID()
				intMemberID, err := strconv.Atoi(memberID)
				So(err, ShouldBeNil)

				livelineClient.SetLiveChannels([]string{ownerID, memberID})

				squad := CreateTestSquadWithMemberAndInvitation(ctx, t, client, injectables, ownerID, memberID, util.NewUserID())
				squadID := squad.Id

				// Set squad as live
				_, err = client.UpdateSquad(ctx, &meepo.UpdateSquadRequest{
					Id:       squadID,
					Status:   meepo.Squad_LIVE,
					CallerId: ownerID,
				})
				So(err, ShouldBeNil)

				// The squad was started, so the owner should have fired a start_squad event
				spadeClient.AssertCalled(t, "TrackEvent", mock.Anything, models.CreatorActionTrackingEvent, models.CreatorActionTracking{
					ChannelID:     intOwnerID,
					Channel:       "user" + ownerID,
					NumMembers:    2,
					SquadStreamID: squadID,
					IsLive:        true,
					IsOwner:       true,
					SquadViewable: true,
					InviteID:      "",
					Action:        models.CreatorActionTypeStartSquad,
					Method:        models.CreatorMethodTypeOwnerClickStart,
				})

				// A member "joins" the squad as it goes live, and they are live
				spadeClient.AssertCalled(t, "TrackEvent", mock.Anything, models.CreatorActionTrackingEvent, models.CreatorActionTracking{
					ChannelID:     intMemberID,
					Channel:       "user" + memberID,
					NumMembers:    2,
					SquadStreamID: squadID,
					IsLive:        true,
					IsOwner:       false,
					SquadViewable: true,
					InviteID:      "",
					Action:        models.CreatorActionTypeJoinSquad,
					Method:        models.CreatorMethodTypeOwnerClickStart,
				})

				memberID2 := util.NewUserID()
				intMemberID2, err := strconv.Atoi(memberID2)
				So(err, ShouldBeNil)

				createInvitationReq := &meepo.CreateInvitationRequest{
					SenderId:    ownerID,
					RecipientId: memberID2,
					CallerId:    ownerID,
				}
				i, err := client.CreateInvitation(ctx, createInvitationReq)
				So(err, ShouldBeNil)
				So(i, ShouldNotBeNil)
				So(i.Invitation, ShouldNotBeNil)

				acceptReq := &meepo.AcceptInvitationRequest{
					InvitationId: i.Invitation.Id,
					CallerId:     memberID2,
				}
				_, err = client.AcceptInvitation(ctx, acceptReq)
				So(err, ShouldBeNil)

				// Another member "joins" the squad after it goes live
				spadeClient.AssertCalled(t, "TrackEvent", mock.Anything, models.CreatorActionTrackingEvent, models.CreatorActionTracking{
					ChannelID:     intMemberID2,
					Channel:       "user" + memberID2,
					NumMembers:    3,
					SquadStreamID: squadID,
					IsLive:        false,
					IsOwner:       false,
					SquadViewable: true,
					InviteID:      i.Invitation.Id,
					Action:        models.CreatorActionTypeJoinSquad,
					Method:        models.CreatorMethodTypeClickAccept,
				})
			})

			Convey("Should fire on leave_squad", func() {
				ownerID := util.NewUserID()
				memberID := util.NewUserID()
				intMemberID, err := strconv.Atoi(memberID)
				So(err, ShouldBeNil)

				livelineClient.SetLiveChannels([]string{ownerID, memberID})

				squad := CreateTestSquadWithMemberAndInvitation(ctx, t, client, injectables, ownerID, memberID, util.NewUserID())
				squadID := squad.Id

				_, e := client.LeaveSquad(ctx, &meepo.LeaveSquadRequest{
					MemberId: memberID,
					CallerId: memberID,
					SquadId:  squadID,
				})
				So(e, ShouldBeNil)

				spadeClient.AssertCalled(t, "TrackEvent", mock.Anything, models.CreatorActionTrackingEvent, models.CreatorActionTracking{
					ChannelID:     intMemberID,
					Channel:       "user" + memberID,
					NumMembers:    1,
					SquadStreamID: squadID,
					IsLive:        true,
					IsOwner:       false,
					SquadViewable: false,
					InviteID:      "",
					Action:        models.CreatorActionTypeLeaveSquad,
					Method:        models.CreatorMethodTypeClickLeaveSquad,
				})

				livelineClient.RemoveLiveChannels([]string{memberID})

				memberID2 := util.NewUserID()
				intMemberID2, err := strconv.Atoi(memberID2)
				So(err, ShouldBeNil)

				createInvitationReq := &meepo.CreateInvitationRequest{
					SenderId:    ownerID,
					RecipientId: memberID2,
					CallerId:    ownerID,
				}
				i, err := client.CreateInvitation(ctx, createInvitationReq)
				So(err, ShouldBeNil)
				So(i, ShouldNotBeNil)
				So(i.Invitation, ShouldNotBeNil)

				acceptReq := &meepo.AcceptInvitationRequest{
					InvitationId: i.Invitation.Id,
					CallerId:     memberID2,
				}
				_, err = client.AcceptInvitation(ctx, acceptReq)
				So(err, ShouldBeNil)

				// Set squad as live
				_, err = client.UpdateSquad(ctx, &meepo.UpdateSquadRequest{
					Id:       squadID,
					Status:   meepo.Squad_LIVE,
					CallerId: ownerID,
				})
				So(err, ShouldBeNil)

				// Remove the offline member2
				req := &meepo.LivecheckChannelsRequest{}
				_, err = internalClient.LivecheckChannels(ctx, req)
				So(err, ShouldBeNil)
				_, err = internalClient.LivecheckChannels(ctx, req)
				So(err, ShouldBeNil)

				spadeClient.AssertCalled(t, "TrackEvent", mock.Anything, models.CreatorActionTrackingEvent, models.CreatorActionTracking{
					ChannelID:     intMemberID2,
					Channel:       "user" + memberID2,
					NumMembers:    1,
					SquadStreamID: squadID,
					IsLive:        false,
					IsOwner:       false,
					SquadViewable: true,
					InviteID:      "",
					Action:        models.CreatorActionTypeLeaveSquad,
					Method:        models.CreatorMethodTypeGoOffline,
				})
			})

			Convey("Should fire on remove_from_squad", func() {
				ownerID := util.NewUserID()
				memberID := util.NewUserID()
				intMemberID, err := strconv.Atoi(memberID)
				So(err, ShouldBeNil)

				squad := CreateTestSquadWithMemberAndInvitation(ctx, t, client, injectables, ownerID, memberID, util.NewUserID())
				squadID := squad.Id

				_, e := client.RemoveMember(ctx, &meepo.RemoveMemberRequest{
					MemberId: memberID,
					CallerId: ownerID,
					SquadId:  squadID,
				})
				So(e, ShouldBeNil)

				spadeClient.AssertCalled(t, "TrackEvent", mock.Anything, models.CreatorActionTrackingEvent, models.CreatorActionTracking{
					ChannelID:     intMemberID,
					Channel:       "user" + memberID,
					NumMembers:    1,
					SquadStreamID: squadID,
					IsLive:        false,
					IsOwner:       false,
					SquadViewable: false,
					InviteID:      "",
					Action:        models.CreatorActionTypeRemoveFromSquad,
					Method:        models.CreatorMethodTypeOwnerClickRemove,
				})

				memberID2 := util.NewUserID()
				intMemberID2, err := strconv.Atoi(memberID2)
				So(err, ShouldBeNil)

				createInvitationReq := &meepo.CreateInvitationRequest{
					SenderId:    ownerID,
					RecipientId: memberID2,
					CallerId:    ownerID,
				}
				i, err := client.CreateInvitation(ctx, createInvitationReq)
				So(err, ShouldBeNil)
				So(i, ShouldNotBeNil)
				So(i.Invitation, ShouldNotBeNil)

				acceptReq := &meepo.AcceptInvitationRequest{
					InvitationId: i.Invitation.Id,
					CallerId:     memberID2,
				}
				_, err = client.AcceptInvitation(ctx, acceptReq)
				So(err, ShouldBeNil)

				// Set squad as live
				_, err = client.UpdateSquad(ctx, &meepo.UpdateSquadRequest{
					Id:       squadID,
					Status:   meepo.Squad_LIVE,
					CallerId: ownerID,
				})
				So(err, ShouldBeNil)

				_, e = client.RemoveMember(ctx, &meepo.RemoveMemberRequest{
					MemberId: memberID2,
					CallerId: ownerID,
					SquadId:  squadID,
				})
				So(e, ShouldBeNil)

				spadeClient.AssertCalled(t, "TrackEvent", mock.Anything, models.CreatorActionTrackingEvent, models.CreatorActionTracking{
					ChannelID:     intMemberID2,
					Channel:       "user" + memberID2,
					NumMembers:    1,
					SquadStreamID: squadID,
					IsLive:        false,
					IsOwner:       false,
					SquadViewable: true,
					InviteID:      "",
					Action:        models.CreatorActionTypeRemoveFromSquad,
					Method:        models.CreatorMethodTypeOwnerClickRemove,
				})
			})

			Convey("Should fire on inherit_ownership", func() {
				ownerID := util.NewUserID()
				memberID := util.NewUserID()
				intMemberID, err := strconv.Atoi(memberID)
				So(err, ShouldBeNil)

				livelineClient.SetLiveChannels([]string{ownerID, memberID})

				squad := CreateTestSquadWithMemberAndInvitation(ctx, t, client, injectables, ownerID, memberID, util.NewUserID())
				squadID := squad.Id

				_, e := client.LeaveSquad(ctx, &meepo.LeaveSquadRequest{
					MemberId: ownerID,
					CallerId: ownerID,
					SquadId:  squadID,
				})
				So(e, ShouldBeNil)

				spadeClient.AssertCalled(t, "TrackEvent", mock.Anything, models.CreatorActionTrackingEvent, models.CreatorActionTracking{
					ChannelID:     intMemberID,
					Channel:       "user" + memberID,
					NumMembers:    1,
					SquadStreamID: squadID,
					IsLive:        true,
					IsOwner:       true,
					SquadViewable: false,
					InviteID:      "",
					Action:        models.CreatorActionTypeInheritOwnership,
					Method:        models.CreatorMethodTypeOriginalOwnerLeave,
				})
			})

			Convey("Should fire when a pending squad is created and started", func() {
				ownerID := util.NewUserID()
				memberID := util.NewUserID()
				intOwnerID, err := strconv.Atoi(ownerID)
				So(err, ShouldBeNil)

				livelineClient.SetLiveChannels([]string{ownerID})

				createInvitationReq := &meepo.CreateInvitationRequest{
					SenderId:    ownerID,
					RecipientId: memberID,
					CallerId:    ownerID,
				}
				i, err := client.CreateInvitation(ctx, createInvitationReq)
				So(err, ShouldBeNil)
				So(i, ShouldNotBeNil)
				So(i.Invitation, ShouldNotBeNil)

				req := &meepo.GetSquadByChannelIDRequest{ChannelId: ownerID}
				squad, err := client.GetSquadByChannelID(ctx, req)
				So(err, ShouldBeNil)
				squadID := squad.Squad.Id

				spadeClient.AssertCalled(t, "TrackEvent", mock.Anything, models.SquadStateChangeTrackingEvent, models.SquadStateChangeTracking{
					SquadStreamID:  squadID,
					State:          models.SquadStatusPending,
					PreviousState:  "",
					OwnerChannelID: intOwnerID,
					OwnerChannel:   "user" + ownerID,
					NumMembers:     1,
					NumMembersLive: 1,
					Method:         models.StateChangeMethodTypeFirstInviteSent,
					LastUpdated:    nil,
				})

				acceptReq := &meepo.AcceptInvitationRequest{
					InvitationId: i.Invitation.Id,
					CallerId:     memberID,
				}
				_, err = client.AcceptInvitation(ctx, acceptReq)
				So(err, ShouldBeNil)

				_, err = client.UpdateSquad(ctx, &meepo.UpdateSquadRequest{
					Id:       squadID,
					Status:   meepo.Squad_LIVE,
					CallerId: ownerID,
				})
				So(err, ShouldBeNil)

				spadeClient.AssertCalled(t, "TrackEvent", mock.Anything, models.SquadStateChangeTrackingEvent, models.SquadStateChangeTracking{
					SquadStreamID:  squadID,
					State:          models.SquadStatusLive,
					PreviousState:  models.SquadStatusPending,
					OwnerChannelID: intOwnerID,
					OwnerChannel:   "user" + ownerID,
					NumMembers:     2,
					NumMembersLive: 1,
					Method:         models.StateChangeMethodTypeOwnerStartSquad,
					LastUpdated:    &controlledTime,
				})
			})

			Convey("Should fire when a pending squad is ended by the owner leaving", func() {
				ownerID := util.NewUserID()
				memberID := util.NewUserID()

				livelineClient.SetLiveChannels([]string{})

				squad := CreateTestSquad(ctx, t, client, injectables, ownerID)
				squadID := squad.Id

				createInvitationReq := &meepo.CreateInvitationRequest{
					SenderId:    ownerID,
					RecipientId: memberID,
					CallerId:    ownerID,
				}
				i, err := client.CreateInvitation(ctx, createInvitationReq)
				So(err, ShouldBeNil)
				So(i, ShouldNotBeNil)
				So(i.Invitation, ShouldNotBeNil)

				acceptReq := &meepo.AcceptInvitationRequest{
					InvitationId: i.Invitation.Id,
					CallerId:     memberID,
				}
				_, err = client.AcceptInvitation(ctx, acceptReq)
				So(err, ShouldBeNil)

				_, e := client.LeaveSquad(ctx, &meepo.LeaveSquadRequest{
					MemberId: memberID,
					CallerId: memberID,
					SquadId:  squadID,
				})
				So(e, ShouldBeNil)

				_, e = client.LeaveSquad(ctx, &meepo.LeaveSquadRequest{
					MemberId: ownerID,
					CallerId: ownerID,
					SquadId:  squadID,
				})
				So(e, ShouldBeNil)

				spadeClient.AssertCalled(t, "TrackEvent", mock.Anything, models.SquadStateChangeTrackingEvent, models.SquadStateChangeTracking{
					SquadStreamID:  squadID,
					State:          models.SquadStatusEnded,
					PreviousState:  models.SquadStatusPending,
					OwnerChannelID: 0,
					OwnerChannel:   "",
					NumMembers:     0,
					NumMembersLive: 0,
					Method:         models.StateChangeMethodTypeLastMemberLeave,
					LastUpdated:    &controlledTime,
				})
			})

			Convey("Should fire when a live squad is ended by the owner going offline", func() {
				ownerID := util.NewUserID()
				memberID := util.NewUserID()

				livelineClient.SetLiveChannels([]string{})

				squad := CreateTestSquad(ctx, t, client, injectables, ownerID)
				squadID := squad.Id

				createInvitationReq := &meepo.CreateInvitationRequest{
					SenderId:    ownerID,
					RecipientId: memberID,
					CallerId:    ownerID,
				}
				i, err := client.CreateInvitation(ctx, createInvitationReq)
				So(err, ShouldBeNil)
				So(i, ShouldNotBeNil)
				So(i.Invitation, ShouldNotBeNil)

				acceptReq := &meepo.AcceptInvitationRequest{
					InvitationId: i.Invitation.Id,
					CallerId:     memberID,
				}
				_, err = client.AcceptInvitation(ctx, acceptReq)
				So(err, ShouldBeNil)

				_, err = client.UpdateSquad(ctx, &meepo.UpdateSquadRequest{
					Id:       squadID,
					Status:   meepo.Squad_LIVE,
					CallerId: ownerID,
				})
				So(err, ShouldBeNil)

				_, e := client.LeaveSquad(ctx, &meepo.LeaveSquadRequest{
					MemberId: memberID,
					CallerId: memberID,
					SquadId:  squadID,
				})
				So(e, ShouldBeNil)

				req := &meepo.LivecheckChannelsRequest{}
				_, err = internalClient.LivecheckChannels(ctx, req)
				So(err, ShouldBeNil)
				_, err = internalClient.LivecheckChannels(ctx, req)
				So(err, ShouldBeNil)

				previousState := models.SquadStatusLive
				spadeClient.AssertCalled(t, "TrackEvent", mock.Anything, models.SquadStateChangeTrackingEvent, models.SquadStateChangeTracking{
					SquadStreamID:  squadID,
					State:          models.SquadStatusEnded,
					PreviousState:  previousState,
					OwnerChannelID: 0,
					OwnerChannel:   "",
					NumMembers:     0,
					NumMembersLive: 0,
					Method:         models.StateChangeMethodTypeLastMemberOffline,
					LastUpdated:    &controlledTime,
				})
			})

			Convey("Should fire when we encounter any business logic meepo error", func() {
				ownerID := util.NewUserID()
				intOwnerID, err := strconv.Atoi(ownerID)
				So(err, ShouldBeNil)

				livelineClient.SetLiveChannels([]string{ownerID})
				CreateTestSquad(ctx, t, client, injectables, ownerID)

				// Should cause a meepo error as a channel can't send a squad invitation to itself
				createInvitationReq := &meepo.CreateInvitationRequest{
					SenderId:    ownerID,
					RecipientId: ownerID,
					CallerId:    ownerID,
				}
				_, err = client.CreateInvitation(ctx, createInvitationReq)
				So(err, ShouldNotBeNil)

				spadeClient.AssertCalled(t, "TrackEvent", mock.Anything, models.SquadStreamErrorTrackingEvent, models.SquadStreamErrorTracking{
					ChannelID:       intOwnerID,
					Channel:         "user" + ownerID,
					TargetChannelID: intOwnerID,
					TargetChannel:   "user" + ownerID,
					NumMembers:      -1,
					SquadStreamID:   "",
					InviteID:        "",
					ErrorCode:       meepo_errors.ErrInvitationInvalid,
					Method:          models.ErrorMethodTypeCreateInvitation,
				})
			})
		})
	})

	ts.onFinish(serverShutdownWaitTime)
}

type testSetup struct {
	ctx                 context.Context
	meepoClient         meepo.Meepo
	internalMeepoClient meepo.InternalMeepo
	httpClient          *http.Client
	host                string
	onFinish            func(timeToWait time.Duration)
	thisInstance        *service
}

func (t *testSetup) Setup() error {
	ctx, cancelFunc := context.WithTimeout(context.Background(), testWaitTime)
	t.ctx = ctx
	Reset(cancelFunc)
	t.meepoClient = meepo.NewMeepoProtobufClient(t.host, &http.Client{})
	t.internalMeepoClient = meepo.NewInternalMeepoProtobufClient(t.host, &http.Client{})
	t.httpClient = &http.Client{}
	return nil
}

func addMapValues(m *distconf.InMemory, vals map[string][]byte) error {
	for k, v := range vals {
		if err := m.Write(k, v); err != nil {
			return err
		}
	}
	return nil
}

type panicPanic struct{}

func (p panicPanic) OnPanic(pnc interface{}) {
	panic(pnc)
}

func newDefaultInjectables() injectables {
	// We set up mocks for followsClient, friendshipClient, and rosterClient to ensure that
	// all users are in each others' networks by default. This way, createInvitation and other
	// network-dependent operations will succeed unless these injectables are changed.
	followsClient := &mocks.FollowsClient{}
	followsClient.On("IsFollowing", mock.Anything, mock.Anything, mock.Anything).Maybe().Return(true, nil)
	friendshipClient := &mocks.FriendshipClient{}
	friendshipClient.On("IsFriend", mock.Anything, mock.Anything, mock.Anything).Maybe().Return(false, nil)
	rosterClient := &mocks.RosterClient{}
	rosterClient.On("IsTeammate", mock.Anything, mock.Anything, mock.Anything).Maybe().Return(false, nil)

	// We set up mocks for livelineClient and spadeClient to provide a default return value when we
	// are not explicitly testing for whether or not certain tracking events fire.
	livelineClient := &mocks.LivelineClient{}
	livelineClient.On("GetLiveChannelsByChannelIDs", mock.Anything, mock.Anything).Maybe().Return(
		[]string{}, nil)
	spadeClient := &mocks.SpadeClient{}
	spadeClient.On("TrackEvent", mock.Anything, mock.Anything, mock.Anything).Maybe().Return(nil)

	// channelStatePublisher is used to publish updates on when channels become part of live squads, and when
	// they are no longer part of a live squad.
	// For our tests, we configure the default channelStatePublisher to accept all calls.
	// (Tests that are verifying publishing behavior can make assertions against the calls that the mock records.)
	channelStatePublisher := &mocks.ChannelStatePublisher{}
	channelStatePublisher.On("PublishChannelIsInLiveSquad", mock.Anything, mock.Anything, mock.Anything)
	channelStatePublisher.On("PublishChannelsInLiveSquad", mock.Anything, mock.Anything, mock.Anything)
	channelStatePublisher.On("PublishChannelOutOfLiveSquad", mock.Anything, mock.Anything)

	pubsubClient := testutil.NewPubsubMockClient()

	return injectables{
		authorizer:                  &mocks.Authorizer{},
		followsClient:               followsClient,
		friendshipClient:            friendshipClient,
		livelineClient:              livelineClient,
		pubsubClient:                pubsubClient,
		ripleyClient:                &mocks.RipleyClient{},
		rosterClient:                rosterClient,
		spadeClient:                 spadeClient,
		usersClient:                 &mocks.UsersClient{},
		channelStatePublisherClient: channelStatePublisher,
		dartClient:                  &mocks.NotifierClient{},
		clock:                       &clock.RealClock{},
		hallpassClient:              &mocks.HallpassClient{},
		pdmsClient:                  &mocks.PDMSClient{},
	}
}

func startServer(t *testing.T, i injectables, configs ...map[string][]byte) *testSetup {
	localConf := &distconf.InMemory{}

	err := addMapValues(localConf, map[string][]byte{
		// Do not hit AWS Parameter Store
		// This is because tests are run concurrently making it likely
		// that the Parameter store API limit is reached, thus making the test fail
		"ssm.region":           []byte(""),
		"meepo.listen_addr":    []byte(":0"),
		"meepo.admin_users":    []byte(adminUserIDs),
		"rollbar.access_token": []byte(""),
		"statsd.hostport":      []byte(""),
		"debug.addr":           []byte(":0"),
		"logging.to_stdout":    []byte("false"),
		"logging.to_stderr":    []byte("true"),
		// Setting squad_enabled_for_partners to true so we can pass our dummy users through the CanAccessSquads method.
		"meepo.squad_enabled_for_partners": []byte("true"),
	})
	if err != nil {
		t.Error(err)
		return nil
	}
	for _, config := range configs {
		err := addMapValues(localConf, config)
		if err != nil {
			t.Error(err)
			return nil
		}
	}

	started := make(chan string)
	finished := make(chan struct{})
	signalToClose := make(chan os.Signal)
	exitCalled := make(chan struct{})
	elevateKey := "hi"
	thisInstance := service{
		injectables: i,
		osExit: func(i int) {
			if i != 0 {
				t.Error("Invalid osExit status code", i)
			}
			close(exitCalled)
		},
		serviceCommon: service_common.ServiceCommon{
			ConfigCommon: service_common.ConfigCommon{
				Team:          teamName,
				Service:       serviceName,
				CustomReaders: []distconf.Reader{localConf},
				BaseDirectory: "../../",
				OsGetenv:      os.Getenv,
				OsHostname:    os.Hostname,
			},
			CodeVersion: CodeVersion,
			Log: &log.ElevatedLog{
				ElevateKey: elevateKey,
				NormalLog: log.ContextLogger{
					Logger: t,
				},
				DebugLog: log.ContextLogger{
					Logger: log.Discard,
				},
				LogToDebug: func(_ ...interface{}) bool {
					return false
				},
			},
			PanicLogger: panicPanic{},
		},
		sigChan: signalToClose,
		onListen: func(listeningAddr net.Addr) {
			started <- fmt.Sprintf("http://localhost:%d", listeningAddr.(*net.TCPAddr).Port)
		},
	}
	thisInstance.serviceCommon.Log.NormalLog.Dims = &thisInstance.serviceCommon.CtxDimensions
	thisInstance.serviceCommon.Log.DebugLog.Dims = &thisInstance.serviceCommon.CtxDimensions
	go func() {
		thisInstance.main()
		close(finished)
	}()

	var addressForIntegrationTests string
	select {
	case <-exitCalled:
		return nil
	case addressForIntegrationTests = <-started:
	case <-time.After(time.Second * 35):
		t.Error("Took to long to start service")
		return nil
	}

	onFinish := func(timeToWait time.Duration) {
		signalToClose <- syscall.SIGTERM
		select {
		case <-finished:
			return
		case <-time.After(timeToWait):
			t.Error("Timed out waiting for server to end")
		}
	}
	return &testSetup{
		host:         addressForIntegrationTests,
		onFinish:     onFinish,
		thisInstance: &thisInstance,
	}
}
