// +build integration

package main

import (
	"context"
	"testing"
	"time"

	"code.justin.tv/twitch-events/meepo/internal/mocks"
	"code.justin.tv/twitch-events/meepo/internal/stubs"
	"code.justin.tv/twitch-events/meepo/internal/util"
	"code.justin.tv/twitch-events/meepo/rpc/meepo"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

func TestDeleteExpiredData(t *testing.T) {
	injectables := newDefaultInjectables()

	followsClient := &mocks.FollowsClient{}
	friendshipClient := &mocks.FriendshipClient{}
	rosterClient := &mocks.RosterClient{}
	authorizer := (injectables.authorizer).(*mocks.Authorizer)
	clock := &stubs.ClockStub{}

	controlledTime := time.Date(2015, 01, 01, 01, 02, 03, 00, time.UTC)
	clock.SetNow(controlledTime)

	followsClient.On("IsFollowing", mock.Anything, mock.Anything, mock.Anything).Maybe().Return(true, nil)
	friendshipClient.On("IsFriend", mock.Anything, mock.Anything, mock.Anything).Maybe().Return(false, nil)
	rosterClient.On("IsTeammate", mock.Anything, mock.Anything, mock.Anything).Maybe().Return(false, nil)
	authorizer.On("CanUpdateSquad", mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(true)
	authorizer.On("CanGetInvitationsBySquadID", mock.Anything, mock.Anything, mock.Anything).Return(true)
	authorizer.On("CanRejectInvitation", mock.Anything, mock.Anything, mock.Anything).Return(true)
	authorizer.On("CanLeaveSquad", mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(true)
	authorizer.On("CanDeleteInvitation", mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(true)
	authorizer.On("CanAcceptInvitation", mock.Anything, mock.Anything, mock.Anything).Return(true)

	injectables.followsClient = followsClient
	injectables.friendshipClient = friendshipClient
	injectables.rosterClient = rosterClient
	injectables.clock = clock

	ts := startServer(t, injectables, map[string][]byte{
		// Do not hit AWS Parameter Store
		// This is because tests are run concurrently making it likely
		// that the Parameter store API limit is reached, thus making the test fail
		"ssm.region":        []byte(""),
		"meepo.ttl_in_days": []byte("28"),
	})
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		internalClient := ts.internalMeepoClient
		meepoClient := ts.meepoClient
		ctx := context.Background()

		ownerID := util.NewUserID()
		memberID := util.NewUserID()
		invitedID := util.NewUserID()
		squad := CreateTestSquadWithMemberAndInvitation(ctx, t, meepoClient, injectables, ownerID, memberID, invitedID)
		squadID := squad.Id

		// Ended squad
		_, err := meepoClient.LeaveSquad(ctx, &meepo.LeaveSquadRequest{
			SquadId:  squadID,
			CallerId: ownerID,
			MemberId: ownerID,
		})
		So(err, ShouldBeNil)

		_, err = meepoClient.LeaveSquad(ctx, &meepo.LeaveSquadRequest{
			SquadId:  squadID,
			CallerId: memberID,
			MemberId: memberID,
		})
		So(err, ShouldBeNil)

		endedSquadRes, err := meepoClient.GetSquadByID(ctx, &meepo.GetSquadByIDRequest{
			Id: squadID,
		})
		So(err, ShouldBeNil)
		So(endedSquadRes.GetSquad().GetStatus(), ShouldEqual, meepo.Squad_ENDED)

		// Pending squad
		ownerID2 := util.NewUserID()
		memberID2 := util.NewUserID()
		invitedID2 := util.NewUserID()
		squad2 := CreateTestSquadWithMemberAndInvitation(ctx, t, meepoClient, injectables, ownerID2, memberID2, invitedID2)
		squadID2 := squad2.Id

		// Deleted invitation
		invitedID3 := util.NewUserID()
		createInvitationRes, err := meepoClient.CreateInvitation(ctx, &meepo.CreateInvitationRequest{
			SenderId:    ownerID2,
			RecipientId: invitedID3,
			CallerId:    ownerID2,
		})
		So(err, ShouldBeNil)
		So(createInvitationRes.GetInvitation().GetStatus(), ShouldEqual, meepo.Invitation_PENDING)

		invitationID := createInvitationRes.GetInvitation().GetId()
		_, err = meepoClient.RejectInvitation(ctx, &meepo.RejectInvitationRequest{
			InvitationId: invitationID,
			CallerId:     invitedID3,
		})
		So(err, ShouldBeNil)

		_, err = meepoClient.DeleteInvitation(ctx, &meepo.DeleteInvitationRequest{
			InvitationId: invitationID,
			CallerId:     ownerID2,
		})
		So(err, ShouldBeNil)

		// Accepted invitation
		invitedID4 := util.NewUserID()
		createInvitationRes2, err := meepoClient.CreateInvitation(ctx, &meepo.CreateInvitationRequest{
			SenderId:    ownerID2,
			RecipientId: invitedID4,
			CallerId:    ownerID2,
		})
		So(err, ShouldBeNil)
		So(createInvitationRes.GetInvitation().GetStatus(), ShouldEqual, meepo.Invitation_PENDING)

		invitationID2 := createInvitationRes2.GetInvitation().GetId()
		_, err = meepoClient.AcceptInvitation(ctx, &meepo.AcceptInvitationRequest{
			InvitationId: invitationID2,
			CallerId:     invitedID4,
		})
		So(err, ShouldBeNil)

		invitationsRes, err := meepoClient.GetInvitationsBySquadID(ctx, &meepo.GetInvitationsBySquadIDRequest{
			SquadId:  squadID2,
			Status:   meepo.Invitation_DELETED,
			CallerId: ownerID2,
		})
		So(err, ShouldBeNil)
		So(invitationsRes.GetInvitations(), ShouldHaveLength, 1)
		So(invitationsRes.GetInvitations()[0].GetId(), ShouldEqual, invitationID)

		invitationsRes2, err := meepoClient.GetInvitationsBySquadID(ctx, &meepo.GetInvitationsBySquadIDRequest{
			SquadId:  squadID2,
			Status:   meepo.Invitation_PENDING,
			CallerId: ownerID2,
		})
		So(err, ShouldBeNil)
		So(invitationsRes2.GetInvitations(), ShouldHaveLength, 1)

		invitationsRes3, err := meepoClient.GetInvitationsBySquadID(ctx, &meepo.GetInvitationsBySquadIDRequest{
			SquadId:  squadID2,
			Status:   meepo.Invitation_ACCEPTED,
			CallerId: ownerID2,
		})
		So(err, ShouldBeNil)
		So(invitationsRes3.GetInvitations(), ShouldHaveLength, 1)

		squadRes, err := meepoClient.GetSquadByID(ctx, &meepo.GetSquadByIDRequest{
			Id: squadID2,
		})
		So(err, ShouldBeNil)
		So(squadRes.GetSquad(), ShouldNotBeNil)
		squad2 = squadRes.GetSquad()

		Convey("Should hard delete ended data and not delete pending and live squads", func() {
			clock.SetNow(controlledTime.AddDate(0, 0, 29))
			req := &meepo.DeleteExpiredDataRequest{}
			_, err = internalClient.DeleteExpiredData(ctx, req)
			So(err, ShouldBeNil)

			res, err := meepoClient.GetSquadByID(ctx, &meepo.GetSquadByIDRequest{
				Id: squadID,
			})
			So(err, ShouldBeNil)
			So(res.GetSquad(), ShouldBeNil)

			res, err = meepoClient.GetSquadByID(ctx, &meepo.GetSquadByIDRequest{
				Id: squadID2,
			})
			So(err, ShouldBeNil)
			So(res, ShouldNotBeNil)
			So(res.Squad, ShouldNotBeNil)
			So(res.Squad.Id, ShouldEqual, squad2.Id)
			So(res.Squad.OwnerId, ShouldEqual, squad2.OwnerId)
			So(res.Squad.Status, ShouldEqual, squad2.Status)
			So(res.Squad.MemberIds, ShouldResemble, squad2.MemberIds)

			invitationsRes, err := meepoClient.GetInvitationsBySquadID(ctx, &meepo.GetInvitationsBySquadIDRequest{
				SquadId:  squadID2,
				Status:   meepo.Invitation_DELETED,
				CallerId: ownerID2,
			})
			So(err, ShouldBeNil)
			So(invitationsRes.GetInvitations(), ShouldHaveLength, 0)

			invitationsRes2, err := meepoClient.GetInvitationsBySquadID(ctx, &meepo.GetInvitationsBySquadIDRequest{
				SquadId:  squadID2,
				Status:   meepo.Invitation_PENDING,
				CallerId: ownerID2,
			})
			So(err, ShouldBeNil)
			So(invitationsRes2.GetInvitations(), ShouldHaveLength, 1)

			invitationsRes3, err := meepoClient.GetInvitationsBySquadID(ctx, &meepo.GetInvitationsBySquadIDRequest{
				SquadId:  squadID2,
				Status:   meepo.Invitation_ACCEPTED,
				CallerId: ownerID2,
			})
			So(err, ShouldBeNil)
			So(invitationsRes3.GetInvitations(), ShouldHaveLength, 0)
		})
	})

	ts.onFinish(serverShutdownWaitTime)
}
