package errors

import "github.com/twitchtv/twirp"

// List of errors returned by the meepo service.
const (
	// ErrMetaKey is the meta key for custom error codes in the twirp error.
	ErrMetaKey string = "code"

	ErrArgumentMissing    string = "ARGUMENT_MISSING"
	ErrArgumentInvalid    string = "ARGUMENT_INVALID"
	ErrUnauthorized       string = "UNAUTHORIZED"
	ErrInvitationNotFound string = "INVITATION_NOT_FOUND"
	ErrSquadNotFound      string = "SQUAD_NOT_FOUND"

	// ErrInvitationCannotBeRejected is thrown when invitation is not in pending state.
	ErrInvitationCannotBeRejected string = "INVITATION_CANNOT_BE_REJECTED"
	// ErrInvitationCannotBeDeleted is thrown when invitation is not in pending nor rejected state.
	ErrInvitationCannotBeDeleted string = "INVITATION_CANNOT_BE_DELETED"
	// ErrInvitationAlreadyExists is thrown when an invitation from the squad to the recipient already exists
	ErrInvitationAlreadyExists string = "INVITATION_ALREADY_EXISTS"
	// ErrInvitationInvalid is thrown when the recipient of the invitation is same as the sender
	ErrInvitationInvalid string = "INVITATION_INVALID"
	// ErrInvitationCannotBeAccepted is thrown when the invitation is not in the pending state
	ErrInvitationCannotBeAccepted string = "INVITATION_CANNOT_BE_ACCEPTED"
	// ErrInvitationBlocked is thrown when a recipient's invite policy automatically rejects an invitation
	ErrInvitationBlocked string = "INVITATION_BLOCKED"
	// ErrSquadCannotBeUpdated is thrown when the squad is not in the pending state
	ErrSquadCannotBeUpdated string = "SQUAD_CANNOT_BE_UPDATED"
	// ErrSquadFull is thrown when the a squad owner attempts to create an invitation/a membership while the squad is full
	ErrSquadFull string = "SQUAD_FULL"
	// ErrSquadTooFewMembers is thrown when a squad attempts to udpate to live status while only having one member
	ErrSquadTooFewMembers string = "SQUAD_TOO_FEW_MEMBERS"
	// ErrUserInAnotherSquad is thrown when a user attempts to create a squad while in another squad
	ErrUserInAnotherSquad string = "USER_IN_ANOTHER_SQUAD"
	// ErrUserAlreadyInSquad is thrown when a squad owner attempts to create an invitation/a membership to a recipient already in the squad
	ErrUserAlreadyInSquad string = "USER_ALREADY_IN_SQUAD"
	// ErrUserNotInSquad is thrown when a user gets removed from/attempts to leave a squad that they are not in
	ErrUserNotInSquad string = "USER_NOT_IN_SQUAD"
	// ErrUserCannotAccessSquads is thrown when a user is the source of Squad Management operation they do not have permission to use
	ErrUserCannotAccessSquads string = "USER_CANNOT_ACCESS_SQUADS"
)

// ConvertTwirpErrorToErrorCode take an twirp error, and if it is a client error, set
// the value pointed to by errCode to the error code in the error's meta. Otherwise
// return the error.
func ConvertTwirpErrorToErrorCode(err twirp.Error, errCode *string) error {
	switch err.Code() {
	case twirp.NotFound, twirp.InvalidArgument, twirp.PermissionDenied, twirp.AlreadyExists:
		if code := err.Meta(ErrMetaKey); code != "" {
			*errCode = code
			return nil
		}
		return err
	default:
		return err
	}
}
