package api

import (
	"log"
	"net/http"
	"strings"
	"time"

	"code.justin.tv/feeds/distconf"
	service_common "code.justin.tv/feeds/service-common"
	"code.justin.tv/twitch-events/meepo/internal/auth"
	"code.justin.tv/twitch-events/meepo/internal/backend"
	"code.justin.tv/twitch-events/meepo/internal/playerstore"
	"code.justin.tv/twitch-events/meepo/rpc/meepo"
	"code.justin.tv/vod/eventbus"

	"github.com/twitchtv/twirp"
	statsd_hook "github.com/twitchtv/twirp/hooks/statsd"
	"goji.io"
	"goji.io/pat"
)

const (
	// requestTimeout is the amount of time we wait for a request to process before
	// cancelling the request context.
	requestTimeout = 5 * time.Second
)

// HTTPServer exposes meepo HTTP endpoints.
type HTTPServer struct {
	service_common.BaseHTTPServer
	Backend     backend.Backender
	Config      *HTTPConfig
	PlayerStore *playerstore.PlayerStore
	Auth        auth.Auth
	eventbus.EventBusProcessor
}

// HTTPConfig configures meepo HTTP parameters.
type HTTPConfig struct {
	service_common.BaseHTTPServerConfig
	shouldLog       *distconf.Bool
	staticSquads    *distconf.Str
	staticSquadsMap map[string][]string
}

// Load verifies config from distconf.
func (c *HTTPConfig) Load(d *distconf.Distconf) error {
	c.shouldLog = d.Bool("meepo.ads_debug_logging_enabled", false)
	c.staticSquads = d.Str("meepo.static_squads", "")
	c.staticSquads.Watch(c.loadStaticSquads)
	c.loadStaticSquads()
	return c.BaseHTTPServerConfig.Verify(d, "meepo")
}

func (c *HTTPConfig) loadStaticSquads() {
	lines := strings.Split(c.staticSquads.Get(), "\n")
	squads := make(map[string][]string)
	for _, line := range lines {
		line = strings.Trim(line, " ")
		if line != "" {
			squad := strings.Split(line, ",")
			if len(squad) > 2 {
				squads[squad[0]] = squad[1:]
			}
		}
	}
	c.staticSquadsMap = squads
}

// Setup sets up the base server.
func (s *HTTPServer) Setup() error {
	return s.BaseHTTPServer.Setup()
}

// Routes registers all the HTTPHandlers on the server's mux.
func (s *HTTPServer) Routes(mux *goji.Mux) {
	// Statsd hooks
	var statsdHook *twirp.ServerHooks
	if s.BaseHTTPServer.Stats != nil {
		statsdHook = statsd_hook.NewStatsdServerHooks(s.BaseHTTPServer.Stats)
	}
	hooks := twirp.ChainHooks(statsdHook)

	// Register Twirp RPC endpoints.
	twirpHandler := meepo.NewMeepoServer(s, hooks)

	// Add request timeout.
	timeoutHandler := http.TimeoutHandler(twirpHandler, requestTimeout, "request timed out")

	// Register Twirp RPC internal endpoints.
	internalHandler := meepo.NewInternalMeepoServer(s, hooks)

	mux.Handle(pat.Post(meepo.MeepoPathPrefix+"*"), timeoutHandler)
	mux.Handle(pat.Post(meepo.InternalMeepoPathPrefix+"*"), internalHandler)

	// add eventbus consumer
	ebConsumer, err := eventbus.NewEventBusConsumer(s)
	if err != nil {
		log.Fatal(err)
	}

	ebConsumerServer := eventbus.NewEventBusConsumerServer(ebConsumer, nil)
	mux.Handle(pat.Post(eventbus.EventBusConsumerPathPrefix+"*"), ebConsumerServer)
}
