package api

import (
	"context"

	"code.justin.tv/twitch-events/meepo/errors"
	"code.justin.tv/twitch-events/meepo/internal/models"
	"code.justin.tv/twitch-events/meepo/rpc/meepo"
	"github.com/twitchtv/twirp"
)

// GetInvitationsBySquadID returns the invitations to the given squad, if it exists.
func (s *HTTPServer) GetInvitationsBySquadID(ctx context.Context, r *meepo.GetInvitationsBySquadIDRequest) (*meepo.GetInvitationsBySquadIDResponse, error) {
	if r.GetSquadId() == "" {
		return nil, twirp.RequiredArgumentError("squad_id").WithMeta(errors.ErrMetaKey, errors.ErrArgumentMissing)
	}
	squadID := r.GetSquadId()

	if r.GetStatus() == meepo.Invitation_UNKNOWN {
		return nil, twirp.InvalidArgumentError("status", "must be one of 'PENDING', 'ACCEPTED', 'REJECTED' and 'DELETED'").WithMeta(errors.ErrMetaKey, errors.ErrArgumentInvalid)
	}
	status := models.NewInvitationStatusFromProto(r.GetStatus())

	if r.GetCallerId() == "" {
		return nil, twirp.RequiredArgumentError("caller_id").WithMeta(errors.ErrMetaKey, errors.ErrArgumentMissing)
	}
	callerID := r.GetCallerId()

	invitations, err := s.Auth.GetInvitationsBySquadID(ctx, squadID, status, callerID)
	if err != nil {
		return nil, err
	}

	invitationsProto := make([]*meepo.Invitation, len(invitations))

	for idx, invitation := range invitations {
		invitationProto, err := models.NewProtoFromInvitation(invitation)
		if err != nil {
			return nil, err
		}

		invitationsProto[idx] = invitationProto
	}

	return &meepo.GetInvitationsBySquadIDResponse{
		Invitations: invitationsProto,
	}, nil
}
