package api

import (
	"context"

	"code.justin.tv/twitch-events/meepo/errors"
	"code.justin.tv/twitch-events/meepo/internal/util"
	"code.justin.tv/twitch-events/meepo/rpc/meepo"

	"github.com/gofrs/uuid"
	"github.com/twitchtv/twirp"
)

// SetPrimaryPlayer sets the primary player channel ID per deviceID & squadID pair in cache.
func (s *HTTPServer) SetPrimaryPlayer(ctx context.Context, r *meepo.SetPrimaryPlayerRequest) (*meepo.SetPrimaryPlayerResponse, error) {
	requestID := ""
	uuid, err := uuid.NewV4()
	if err == nil {
		requestID = uuid.String()
	}

	ctx = s.Log.NormalLog.Dims.Append(ctx,
		"endpoint", "set_primary_player",
		"request_id", requestID,
		"primary_player_channel_id", r.GetPrimaryPlayerChannelId(),
		"device_id", r.GetDeviceId(),
		"squad_id", r.GetSquadId())
	util.LogCtx(ctx, s.Log, s.Config.shouldLog, "event", "request_received")

	if r.GetDeviceId() == "" {
		util.LogCtx(ctx, s.Log, s.Config.shouldLog,
			"event", "error_response",
			"reason", "no device id")
		return nil, twirp.RequiredArgumentError("device_id").WithMeta(errors.ErrMetaKey, errors.ErrArgumentMissing)
	}
	if r.GetSquadId() == "" {
		util.LogCtx(ctx, s.Log, s.Config.shouldLog,
			"event", "error_response",
			"reason", "no squad id")
		return nil, twirp.RequiredArgumentError("squad_id").WithMeta(errors.ErrMetaKey, errors.ErrArgumentMissing)
	}
	if r.GetPrimaryPlayerChannelId() == "" {
		util.LogCtx(ctx, s.Log, s.Config.shouldLog,
			"event", "error_response",
			"reason", "no primary player channel id")
		return nil, twirp.RequiredArgumentError("primary_player_channel_id").WithMeta(errors.ErrMetaKey, errors.ErrArgumentMissing)
	}

	deviceID := r.GetDeviceId()
	squadID := r.GetSquadId()
	primaryPlayerChannelID := r.GetPrimaryPlayerChannelId()
	err = s.PlayerStore.SetPrimaryPlayerChannelID(ctx, deviceID, squadID, primaryPlayerChannelID)
	if err != nil {
		util.LogCtx(ctx, s.Log, s.Config.shouldLog,
			"event", "error_response",
			"reason", "SetPrimaryPlayerChannelID error")
		return nil, err
	}

	util.LogCtx(ctx, s.Log, s.Config.shouldLog,
		"event", "show_ad_response",
		"reason", "fallthrough")
	return &meepo.SetPrimaryPlayerResponse{
		PrimaryPlayerChannelId: primaryPlayerChannelID,
	}, nil
}
