package api

import (
	"context"

	"code.justin.tv/twitch-events/meepo/errors"
	"code.justin.tv/twitch-events/meepo/internal/models"
	"code.justin.tv/twitch-events/meepo/internal/util"
	"code.justin.tv/twitch-events/meepo/rpc/meepo"

	"github.com/gofrs/uuid"
	"github.com/twitchtv/twirp"
)

// ShouldShowAd determines whether to show ads.
func (s *HTTPServer) ShouldShowAd(ctx context.Context, r *meepo.ShouldShowAdRequest) (*meepo.ShouldShowAdResponse, error) {
	requestID := ""
	uuid, err := uuid.NewV4()
	if err == nil {
		requestID = uuid.String()
	}

	ctx = s.Log.NormalLog.Dims.Append(ctx,
		"endpoint", "should_show_ad",
		"request_id", requestID,
		"device_id", r.GetDeviceId(),
		"ad_type", r.GetAdType().String(),
		"channel_id", r.GetChannelId(),
		"device_id", r.GetDeviceId(),
		"user_id", r.GetUserId(),
		"player_type", r.GetPlayerType().String())

	util.LogCtx(ctx, s.Log, s.Config.shouldLog, "event", "request_received")

	if r.GetChannelId() == "" {
		util.LogCtx(ctx, s.Log, s.Config.shouldLog,
			"event", "error_response",
			"reason", "no channel id")
		return nil, twirp.RequiredArgumentError("channel_id").WithMeta(errors.ErrMetaKey, errors.ErrArgumentMissing)
	}

	squad, err := s.Backend.GetSquadByChannelID(ctx, r.GetChannelId())
	if err != nil {
		util.LogCtx(ctx, s.Log, s.Config.shouldLog,
			"event", "error_response",
			"reason", "getSquadByChannelID error")
		return nil, err
	}

	if squad != nil {
		util.LogCtx(ctx, s.Log, s.Config.shouldLog,
			"event", "get_squad",
			"squad_id", squad.ID,
			"squad_status", squad.Status)
	}

	if squad == nil || squad.Status != models.SquadStatusLive {
		util.LogCtx(ctx, s.Log, s.Config.shouldLog,
			"event", "show_ad_response",
			"reason", "squad not live")
		return &meepo.ShouldShowAdResponse{ShowAd: true}, nil
	}

	if r.GetAdType() == meepo.ShouldShowAdRequest_PRE_ROLL {
		if r.GetPlayerType() == meepo.ShouldShowAdRequest_SECONDARY {
			util.LogCtx(ctx, s.Log, s.Config.shouldLog,
				"event", "decline_ad_response",
				"reason", "pre roll - secondary player")
			return &meepo.ShouldShowAdResponse{ShowAd: false}, nil
		}
	} else { // MID and POST rolls
		deviceID := r.GetDeviceId()
		if deviceID == "" {
			s.Stats.IncC("should_show_ad.no_device_id", 1, 1.0)
			util.LogCtx(ctx, s.Log, s.Config.shouldLog,
				"event", "show_ad_response",
				"reason", "mid roll and post roll - no device id")
			return &meepo.ShouldShowAdResponse{ShowAd: true}, nil
		}

		// Cannot trust PlayerType in mid-rolls because this will not update when user changes primary player channel
		primaryPlayerChannelID, err := s.PlayerStore.GetPrimaryPlayerChannelID(ctx, deviceID, squad.ID)
		if err != nil {
			util.LogCtx(ctx, s.Log, s.Config.shouldLog,
				"event", "error_response",
				"reason", "GetPrimaryPlayerChannelID error")
			return nil, err
		}

		if primaryPlayerChannelID != nil && r.GetChannelId() != *primaryPlayerChannelID {
			util.LogCtx(ctx, s.Log, s.Config.shouldLog,
				"event", "decline_ad_response",
				"reason", "viewer not in cache")
			return &meepo.ShouldShowAdResponse{ShowAd: false}, nil
		}
	}

	util.LogCtx(ctx, s.Log, s.Config.shouldLog,
		"event", "show_ad_response",
		"reason", "fallthrough")
	return &meepo.ShouldShowAdResponse{ShowAd: true}, nil
}
