package api

import (
	"context"
	"fmt"

	"code.justin.tv/twitch-events/meepo/errors"
	"code.justin.tv/twitch-events/meepo/internal/models"
	"code.justin.tv/twitch-events/meepo/rpc/meepo"
	"github.com/twitchtv/twirp"
)

// UpdateSquad returns the squad information for the given squad ID, if it exists.
func (s *HTTPServer) UpdateSquad(ctx context.Context, r *meepo.UpdateSquadRequest) (*meepo.UpdateSquadResponse, error) {
	if r.GetId() == "" {
		return nil, twirp.RequiredArgumentError("id").WithMeta(errors.ErrMetaKey, errors.ErrArgumentMissing)
	}
	id := r.GetId()

	if r.GetStatus() == meepo.Squad_UNKNOWN {
		return nil, twirp.InvalidArgumentError("status", "must be one of 'PENDING', 'LIVE', and 'ENDED'").WithMeta(errors.ErrMetaKey, errors.ErrArgumentInvalid)
	}
	status := models.NewSquadStatusFromProto(r.GetStatus())

	if r.GetCallerId() == "" {
		return nil, twirp.RequiredArgumentError("caller_id").WithMeta(errors.ErrMetaKey, errors.ErrArgumentMissing)
	}
	callerID := r.GetCallerId()

	squad, err := s.Auth.UpdateSquad(ctx, id, status, callerID)
	if err != nil {
		return nil, err
	}

	if squad == nil {
		return nil, twirp.NotFoundError(fmt.Sprintf("squad with id %v does not exist", id)).WithMeta(errors.ErrMetaKey, errors.ErrSquadNotFound)
	}

	squadProto, err := models.NewProtoFromSquad(squad)
	if err != nil {
		return nil, err
	}

	return &meepo.UpdateSquadResponse{
		Squad: squadProto,
	}, nil
}
