package auth

import (
	"context"

	meepoerrors "code.justin.tv/twitch-events/meepo/errors"
	"code.justin.tv/twitch-events/meepo/internal/models"
	"github.com/twitchtv/twirp"
)

// AcceptInvitation wraps the corresponding backend function with an authorizer
func (a *auth) AcceptInvitation(ctx context.Context, invitationID, callerID string) (*models.Invitation, error) {
	if !a.authorizer.CanAcceptInvitation(ctx, invitationID, callerID) {
		a.backend.FireSquadStreamErrorTrackingEvent(ctx, models.SquadStreamErrorTrackingEventInfo{
			ChannelID: callerID,
			InviteID:  invitationID,
			Method:    models.ErrorMethodTypeAcceptInvitation,
			ErrorCode: meepoerrors.ErrUnauthorized,
		})
		return nil, twirp.NewError(twirp.PermissionDenied, "The caller cannot accept the invitation").WithMeta(meepoerrors.ErrMetaKey, meepoerrors.ErrUnauthorized)
	}

	return a.backend.AcceptInvitation(ctx, invitationID, callerID)
}

// CanAcceptInvitation determines whether a userID is allowed to accept an invitation with ID, invitationID
func (a *auth) CanAcceptInvitation(ctx context.Context, invitationID, callerID string) bool {
	if callerID == "" || invitationID == "" {
		return false
	}

	if a.utils.IsAdmin(callerID) {
		return true
	}

	dbInvitation, err := a.backend.GetInvitationByID(ctx, invitationID)
	if err != nil || dbInvitation == nil {
		return false
	}

	isAuthorized, err := a.backend.CanAccessSquads(ctx, dbInvitation.RecipientID)
	if err != nil {
		a.log.LogCtx(ctx, "err", err, "cannot determine squad access")
		return false
	} else if !isAuthorized {
		return false
	}
	isAuthorized, err = a.backend.CanAccessSquads(ctx, dbInvitation.SenderID)
	if err != nil {
		a.log.LogCtx(ctx, "err", err, "cannot determine squad access")
		return false
	} else if !isAuthorized {
		return false
	}

	if a.utils.IsInvitationRecipient(callerID, dbInvitation) {
		return true
	}

	isAuthorized, err = a.utils.IsTwitchEditor(ctx, callerID, dbInvitation.RecipientID)
	if err != nil {
		a.log.LogCtx(ctx, "err", err, "could not get twitch editor status")
	} else if isAuthorized {
		return true
	}

	isAuthorized, err = a.utils.IsTwitchAdmin(ctx, callerID)
	if err != nil {
		a.log.LogCtx(ctx, "err", err, "could not get twitch admin status")
	}

	return isAuthorized
}
