package auth_test

import (
	"context"
	"testing"

	"github.com/pkg/errors"

	"code.justin.tv/twitch-events/meepo/internal/mocks"
	"code.justin.tv/twitch-events/meepo/internal/models"
	"code.justin.tv/twitch-events/meepo/internal/testutil"
	"code.justin.tv/twitch-events/meepo/internal/util"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

func TestCanAcceptInvitation(t *testing.T) {
	Convey("CanAcceptInvitation", t, func() {
		i := setUpInjectables()
		authHelpers := (i.Utils).(*mocks.Utils)
		backend := (i.Backend).(*mocks.Backender)

		auth, err := SetUpAuthorizer(t, i)
		if err != nil {
			t.Error("Unable to setup Auth mocks")
		}
		ctx := context.Background()

		invitationID := testutil.MustNewID()
		recipientID := util.NewUserID()
		senderID := util.NewUserID()
		invitation := &models.DBInvitation{
			ID:          invitationID,
			RecipientID: recipientID,
			SenderID:    senderID,
		}
		backend.On("GetInvitationByID", mock.Anything, invitationID).Return(invitation, nil)

		Convey("No invitationID should return false", func() {
			So(auth.CanAcceptInvitation(ctx, "", util.NewUserID()), ShouldBeFalse)
		})

		Convey("No callerID should return false", func() {
			So(auth.CanAcceptInvitation(ctx, invitationID, ""), ShouldBeFalse)
		})

		Convey("Invalid invitationIDs should return false", func() {
			invitationID := testutil.MustNewID()
			callerID := util.NewUserID()
			authHelpers.On("IsAdmin", callerID).Return(false)

			Convey("Non-existent invitation", func() {
				backend.On("GetInvitationByID", mock.Anything, invitationID).Return(nil, nil)

				So(auth.CanAcceptInvitation(ctx, invitationID, callerID), ShouldBeFalse)
			})

			Convey("Error getting invitations", func() {
				backend.On("GetInvitationByID", mock.Anything, invitationID).Return(nil, errors.New("test error"))

				So(auth.CanAcceptInvitation(ctx, invitationID, callerID), ShouldBeFalse)
			})
		})

		Convey("Admins as callerID should return true", func() {
			adminUserID := util.NewUserID()

			// Should authorize callerID independent of the following checks
			backend.On("CanAccessSquads", mock.Anything, senderID).Return(false, nil)
			backend.On("CanAccessSquads", mock.Anything, recipientID).Return(false, nil)
			authHelpers.On("IsTwitchAdmin", mock.Anything, adminUserID).Return(false, nil)
			authHelpers.On("IsTwitchEditor", mock.Anything, adminUserID, recipientID).Return(false, nil)
			authHelpers.On("IsInvitationRecipient", adminUserID, invitation).Return(false)

			authHelpers.On("IsAdmin", adminUserID).Return(true)
			So(auth.CanAcceptInvitation(ctx, invitationID, adminUserID), ShouldBeTrue)
		})

		Convey("Sender and Recipient has access to squads", func() {
			backend.On("CanAccessSquads", mock.Anything, senderID).Return(true, nil)
			backend.On("CanAccessSquads", mock.Anything, recipientID).Return(true, nil)

			Convey("Twitch Admins as callerID should return true", func() {
				twitchAdminID := util.NewUserID()
				// Should authorize callerID independent of the following checks
				authHelpers.On("IsAdmin", twitchAdminID).Return(false)
				authHelpers.On("IsTwitchEditor", mock.Anything, twitchAdminID, recipientID).Return(false, nil)
				authHelpers.On("IsInvitationRecipient", twitchAdminID, invitation).Return(false)

				authHelpers.On("IsTwitchAdmin", mock.Anything, twitchAdminID).Return(true, nil)
				So(auth.CanAcceptInvitation(ctx, invitationID, twitchAdminID), ShouldBeTrue)
			})

			Convey("Editors for Invitation Recipient as callerID should return true", func() {
				editorID := util.NewUserID()
				// Should authorize callerID independent of the following checks
				authHelpers.On("IsAdmin", editorID).Return(false)
				authHelpers.On("IsTwitchAdmin", mock.Anything, editorID).Return(false, nil)
				authHelpers.On("IsInvitationRecipient", editorID, invitation).Return(false)

				authHelpers.On("IsTwitchEditor", mock.Anything, editorID, recipientID).Return(true, nil)
				So(auth.CanAcceptInvitation(ctx, invitationID, editorID), ShouldBeTrue)
			})

			Convey("Invitation Recipients as callerID should return true", func() {
				//  callerID is not a Meepo Admin, Twitch Admin nor Editor
				authHelpers.On("IsAdmin", recipientID).Return(false)
				authHelpers.On("IsTwitchAdmin", mock.Anything, recipientID).Return(false, nil)
				authHelpers.On("IsTwitchEditor", mock.Anything, recipientID, recipientID).Return(false, nil)

				authHelpers.On("IsInvitationRecipient", recipientID, invitation).Return(true)
				So(auth.CanAcceptInvitation(ctx, invitationID, recipientID), ShouldBeTrue)
			})

			Convey("Unauthorized Users should return false", func() {
				userID := util.NewUserID()
				authHelpers.On("IsAdmin", userID).Return(false)
				authHelpers.On("IsTwitchAdmin", mock.Anything, userID).Return(false, nil)
				authHelpers.On("IsTwitchEditor", mock.Anything, userID, recipientID).Return(false, nil)
				authHelpers.On("IsInvitationRecipient", userID, invitation).Return(false)

				So(auth.CanAcceptInvitation(ctx, invitationID, userID), ShouldBeFalse)
			})
		})

		Convey("Invitation Recipient needs access to Squad Stream feature", func() {
			// callerID is not a Meepo Admin, Twitch Admin nor Editor
			authHelpers.On("IsAdmin", recipientID).Return(false)
			authHelpers.On("IsTwitchAdmin", mock.Anything, recipientID).Return(false, nil)
			authHelpers.On("IsTwitchEditor", mock.Anything, recipientID, recipientID).Return(false, nil)

			backend.On("CanAccessSquads", mock.Anything, recipientID).Return(false, nil)
			backend.On("CanAccessSquads", mock.Anything, senderID).Return(true, nil)
			authHelpers.On("IsInvitationRecipient", recipientID, invitation).Return(true)
			So(auth.CanAcceptInvitation(ctx, invitationID, recipientID), ShouldBeFalse)
		})

		Convey("Invitation Sender needs access to Squad Stream feature", func() {
			// callerID is not a Meepo Admin, Twitch Admin nor Editor
			authHelpers.On("IsAdmin", recipientID).Return(false)
			authHelpers.On("IsTwitchAdmin", mock.Anything, recipientID).Return(false, nil)
			authHelpers.On("IsTwitchEditor", mock.Anything, recipientID, recipientID).Return(false, nil)

			backend.On("CanAccessSquads", mock.Anything, recipientID).Return(true, nil)
			backend.On("CanAccessSquads", mock.Anything, senderID).Return(false, nil)
			authHelpers.On("IsInvitationRecipient", recipientID, invitation).Return(true)
			So(auth.CanAcceptInvitation(ctx, invitationID, recipientID), ShouldBeFalse)
		})
	})
}
