package auth

import (
	"context"

	"code.justin.tv/twitch-events/meepo/internal/models"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/log"
	"code.justin.tv/twitch-events/meepo/clients"
	"code.justin.tv/twitch-events/meepo/internal/backend"
)

// Auth contains a wrapped a backend function with an authorizer
type Auth interface {
	AcceptInvitation(ctx context.Context, invitationID, callerID string) (*models.Invitation, error)
	AuthorizeTopicSubscription(ctx context.Context, topic string, callerID string) (bool, error)
	CreateMembership(ctx context.Context, targetUserID, squadID, callerID string) (*models.Member, error)
	CreateInvitation(ctx context.Context, senderID, recipientID, callerID string) (*models.Invitation, error)
	CreateSquad(ctx context.Context, ownerID, callerID string) (*models.Squad, error)
	DeleteInvitation(ctx context.Context, invitationID string, callerID string) (*models.Invitation, error)
	GetPendingInvitationsByRecipientID(ctx context.Context, channelID, callerID string) ([]*models.Invitation, error)
	GetPendingInvitationsCountByRecipientID(ctx context.Context, channelID string, callerID string) (int64, error)
	GetInvitationsBySquadID(ctx context.Context, squadID string, status models.InvitationStatus, callerID string) ([]*models.Invitation, error)
	LeaveSquad(ctx context.Context, memberID, squadID, callerID string) (*models.Squad, error)
	RejectInvitation(ctx context.Context, invitationID, callerID string, reasonRejected models.InvitationReasonRejected) (*models.Invitation, error)
	RejectOutOfNetworkInvitations(ctx context.Context, channelID, callerID string, reasonRejected models.InvitationReasonRejected) error
	RemoveMember(ctx context.Context, memberID, squadID, callerID string) (*models.Squad, error)
	UpdateSquad(ctx context.Context, squadID string, status models.SquadStatus, callerID string) (*models.Squad, error)
}

// Authorizer contains authorization functions for each endpoint
type Authorizer interface {
	CanAcceptInvitation(ctx context.Context, invitationID, callerID string) bool
	CanLeaveSquad(ctx context.Context, memberID, squadID, callerID string) bool
	CanCreateMembership(ctx context.Context, targetUserID, squadID, callerID string) bool
	CanCreateInvitation(ctx context.Context, senderID, recipientID, callerID string) bool
	CanCreateSquad(ctx context.Context, ownerID, callerID string) bool
	CanDeleteInvitation(ctx context.Context, invitationID, callerID string) bool
	CanGetInvitationsBySquadID(ctx context.Context, squadID, callerID string) bool
	CanRejectInvitation(ctx context.Context, invitationID, callerID string) bool
	CanRejectOutOfNetworkInvitations(ctx context.Context, channelID, callerID string) bool
	CanGetPendingInvitationsByRecipientID(ctx context.Context, channelID, callerID string) bool
	CanGetPendingInvitationsCountByRecipientID(ctx context.Context, channelID, callerID string) bool
	CanRemoveMember(ctx context.Context, memberID, squadID, callerID string) bool
	CanSubscribe(ctx context.Context, topicName, topicUserID, callerID string) bool
	CanUpdateSquad(ctx context.Context, squadID string, callerID string) bool
}

// Config contains the configuration values for auth.
type Config struct {
	adminUsers *distconf.Str
}

// Load loads the config for auth.
func (c *Config) Load(d *distconf.Distconf) error {
	c.adminUsers = d.Str("meepo.admin_users", "")
	return nil
}

// Clients contains the clients required for auth.
type Clients struct {
	Hallpass clients.HallpassClient
	Users    clients.UsersClient
}

// auth contains auth's dependencies.
type auth struct {
	backend    backend.Backender
	authorizer Authorizer

	utils Utils

	log *log.ElevatedLog
}

// NewAuth initializes an Auth using the provided backend
func NewAuth(backend backend.Backender, authorizer Authorizer) (Auth, error) {
	return &auth{
		backend:    backend,
		authorizer: authorizer,
	}, nil
}

// NewAuthorizer initializes an Authorizer to be used for tests
func NewAuthorizer(backend backend.Backender, utils Utils, logger *log.ElevatedLog) (Authorizer, error) {
	return &auth{
		backend: backend,
		utils:   utils,
		log:     logger,
	}, nil
}
