package auth_test

import (
	"testing"

	"code.justin.tv/twitch-events/meepo/clients"

	"code.justin.tv/twitch-events/meepo/internal/backend"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/log"
	a "code.justin.tv/twitch-events/meepo/internal/auth"
	"code.justin.tv/twitch-events/meepo/internal/mocks"
)

type injectables struct {
	Backend  backend.Backender
	Hallpass clients.HallpassClient
	Users    clients.UsersClient
	Utils    a.Utils
}

// Instantiate dependencies for Auth interface to be mocked in tests
func setUpInjectables() injectables {
	return injectables{
		Backend:  &mocks.Backender{},
		Hallpass: &mocks.HallpassClient{},
		Users:    &mocks.UsersClient{},
		Utils:    &mocks.Utils{},
	}
}

// Instantiate a config for Auth using in-memory reader
func SetUpAuthConfig(configs ...map[string][]byte) (*a.Config, error) {
	localConf := &distconf.InMemory{}

	// Default config values
	err := addMapValues(localConf, map[string][]byte{
		"meepo.admin_users": []byte("123456789"),
	})
	if err != nil {
		return nil, err
	}
	for _, config := range configs {
		err := addMapValues(localConf, config)
		if err != nil {
			return nil, err
		}
	}

	readers := []distconf.Reader{localConf}
	authConfig := &a.Config{}
	err = authConfig.Load(&distconf.Distconf{Readers: readers})
	if err != nil {
		return nil, err
	}
	return authConfig, nil
}

// Instantiate a fully mocked out Auth interface instance to be used in tests
func SetUpAuthorizer(t *testing.T, i injectables) (a.Authorizer, error) {
	return a.NewAuthorizer(
		i.Backend,
		i.Utils,
		&log.ElevatedLog{
			ElevateKey: "example_key",
			NormalLog: log.ContextLogger{
				Logger: t,
			},
			DebugLog: log.ContextLogger{
				Logger: log.Discard,
			},
			LogToDebug: func(_ ...interface{}) bool {
				return false
			},
		},
	)
}

// Instantiate a helper to be used in tests
func SetUpForUtils(t *testing.T, config *a.Config, i injectables) a.Utils {
	return a.NewUtils(
		config,
		&a.Clients{
			Hallpass: i.Hallpass,
			Users:    i.Users,
		},
		&log.ElevatedLog{
			ElevateKey: "example_key",
			NormalLog: log.ContextLogger{
				Logger: t,
			},
			DebugLog: log.ContextLogger{
				Logger: log.Discard,
			},
			LogToDebug: func(_ ...interface{}) bool {
				return false
			},
		})
}

func addMapValues(m distconf.Writer, vals map[string][]byte) error {
	for k, v := range vals {
		if err := m.Write(k, v); err != nil {
			return err
		}
	}
	return nil
}
