package auth

import (
	"context"
	"fmt"
	"strings"

	meepoerrors "code.justin.tv/twitch-events/meepo/errors"
	"github.com/twitchtv/twirp"
)

// AuthorizeTopicSubscription determines whether a caller is allowed to subscribe to a pubsub topic
func (a *auth) AuthorizeTopicSubscription(ctx context.Context, topic string, callerID string) (bool, error) {
	parts := strings.Split(topic, ".")
	if len(parts) != 2 {
		return false, twirp.NewError(
			twirp.InvalidArgument,
			fmt.Sprintf("The topic, %s cannot be recognized.", topic),
		).WithMeta(meepoerrors.ErrMetaKey, meepoerrors.ErrUnauthorized)
	}

	if !a.authorizer.CanSubscribe(ctx, parts[0], parts[1], callerID) {
		return false, twirp.NewError(
			twirp.PermissionDenied,
			fmt.Sprintf("The user with ID, %s is not allowed to subscribe to this topic.", callerID),
		).WithMeta(meepoerrors.ErrMetaKey, meepoerrors.ErrUnauthorized)
	}

	return true, nil
}

// CanSubscribe authorizes the request to subscribe to a pubsub topic
func (a *auth) CanSubscribe(ctx context.Context, topicName, topicUserID, callerID string) bool {
	if topicName == "" || topicUserID == "" || callerID == "" {
		return false
	}

	if topicName != "channel-squad-invites" {
		return true
	}

	if topicUserID == callerID {
		return true
	}

	if a.utils.IsAdmin(callerID) {
		return true
	}

	isAuthorized, err := a.utils.IsTwitchEditor(ctx, callerID, topicUserID)
	if err != nil {
		a.log.LogCtx(ctx, "err", err, "could not get twitch editor status")
	} else if isAuthorized {
		return true
	}

	isAuthorized, err = a.utils.IsTwitchAdmin(ctx, callerID)
	if err != nil {
		a.log.LogCtx(ctx, "err", err, "could not get twitch admin status")
	}

	return isAuthorized
}
