package auth_test

import (
	"context"
	"testing"

	"code.justin.tv/twitch-events/meepo/internal/mocks"
	"code.justin.tv/twitch-events/meepo/internal/testutil"
	"code.justin.tv/twitch-events/meepo/internal/util"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

func TestCanSubscribe(t *testing.T) {
	Convey("CanSubscribe", t, func() {
		i := setUpInjectables()
		authHelpers := (i.Utils).(*mocks.Utils)

		auth, err := SetUpAuthorizer(t, i)
		if err != nil {
			t.Error("Unable to setup Auth mocks")
		}
		ctx := context.Background()

		Convey("No topicName should return false", func() {
			So(auth.CanSubscribe(ctx, "", util.NewUserID(), util.NewUserID()), ShouldBeFalse)
		})

		Convey("No topicUserID should return false", func() {
			So(auth.CanSubscribe(ctx, "channel-squad-invites", "", util.NewUserID()), ShouldBeFalse)
		})

		Convey("No callerID should return false", func() {
			So(auth.CanSubscribe(ctx, "channel-squad-invites", util.NewUserID(), ""), ShouldBeFalse)
		})

		Convey("Topics other than `channel-squad-invites` should allow subscription", func() {
			topicUserID := util.NewUserID()
			callerID := util.NewUserID()
			squadID := testutil.MustNewID()

			authHelpers.On("IsAdmin", callerID).Return(false)
			authHelpers.On("IsTwitchAdmin", mock.Anything, callerID).Return(false, nil)
			authHelpers.On("IsTwitchEditor", mock.Anything, callerID, topicUserID).Return(false, nil)

			So(auth.CanSubscribe(ctx, squadID, topicUserID, callerID), ShouldBeTrue)
		})

		Convey("Admins as callerID should return true", func() {
			topicUserID := util.NewUserID()
			adminID := util.NewUserID()

			authHelpers.On("IsTwitchAdmin", mock.Anything, adminID).Return(false, nil)
			authHelpers.On("IsTwitchEditor", mock.Anything, adminID, topicUserID).Return(false, nil)

			authHelpers.On("IsAdmin", adminID).Return(true)
			So(auth.CanSubscribe(ctx, "channel-squad-invites", topicUserID, adminID), ShouldBeTrue)
		})

		Convey("Twitch Admins as callerID should return true", func() {
			topicUserID := util.NewUserID()
			twitchAdminID := util.NewUserID()

			authHelpers.On("IsAdmin", twitchAdminID).Return(false)
			authHelpers.On("IsTwitchEditor", mock.Anything, twitchAdminID, topicUserID).Return(false, nil)

			authHelpers.On("IsTwitchAdmin", mock.Anything, twitchAdminID).Return(true, nil)
			So(auth.CanSubscribe(ctx, "channel-squad-invites", topicUserID, twitchAdminID), ShouldBeTrue)
		})

		Convey("Twitch Editors as callerID should return true", func() {
			topicUserID := util.NewUserID()
			editorID := util.NewUserID()

			authHelpers.On("IsAdmin", editorID).Return(false)
			authHelpers.On("IsTwitchAdmin", mock.Anything, editorID).Return(false, nil)

			authHelpers.On("IsTwitchEditor", mock.Anything, editorID, topicUserID).Return(true, nil)
			So(auth.CanSubscribe(ctx, "channel-squad-invites", topicUserID, editorID), ShouldBeTrue)
		})

		Convey("Authorized users as callerID should return true", func() {
			callerID := util.NewUserID()

			authHelpers.On("IsAdmin", callerID).Return(false)
			authHelpers.On("IsTwitchAdmin", mock.Anything, callerID).Return(false, nil)
			authHelpers.On("IsTwitchEditor", mock.Anything, callerID, callerID).Return(false, nil)

			So(auth.CanSubscribe(ctx, "channel-squad-invites", callerID, callerID), ShouldBeTrue)
		})

		Convey("Unauthorized users as callerID should return false", func() {
			topicUserID := util.NewUserID()
			userID := util.NewUserID()
			authHelpers.On("IsAdmin", userID).Return(false)
			authHelpers.On("IsTwitchAdmin", mock.Anything, userID).Return(false, nil)
			authHelpers.On("IsTwitchEditor", mock.Anything, userID, topicUserID).Return(false, nil)

			So(auth.CanSubscribe(ctx, "channel-squad-invites", topicUserID, userID), ShouldBeFalse)
		})
	})
}
