package auth

import (
	"context"

	meepoerrors "code.justin.tv/twitch-events/meepo/errors"
	"code.justin.tv/twitch-events/meepo/internal/models"
	"github.com/twitchtv/twirp"
)

// CreateInvitation wraps the corresponding backend function with an authorizer
func (a *auth) CreateInvitation(ctx context.Context, senderID, recipientID, callerID string) (*models.Invitation, error) {
	if !a.authorizer.CanCreateInvitation(ctx, senderID, recipientID, callerID) {
		a.backend.FireSquadStreamErrorTrackingEvent(ctx, models.SquadStreamErrorTrackingEventInfo{
			ChannelID:       callerID,
			TargetChannelID: &recipientID,
			Method:          models.ErrorMethodTypeCreateInvitation,
			ErrorCode:       meepoerrors.ErrUnauthorized,
		})
		return nil, twirp.NewError(twirp.PermissionDenied, "The caller cannot create the invitation").WithMeta(meepoerrors.ErrMetaKey, meepoerrors.ErrUnauthorized)
	}

	return a.backend.CreateInvitation(ctx, senderID, recipientID, callerID)
}

// CanCreateInvitation determines whether a callerID is allowed to create an invitation to recipientID
func (a *auth) CanCreateInvitation(ctx context.Context, senderID, recipientID, callerID string) bool {
	if callerID == "" || senderID == "" || recipientID == "" {
		return false
	}

	// Meepo admins can create invitations regardless of senderID and recipientID squad access status
	if a.utils.IsAdmin(callerID) {
		return true
	}

	isAuthorized, err := a.backend.CanAccessSquads(ctx, senderID)
	if err != nil {
		a.log.LogCtx(ctx, "err", err, "cannot determine squad access for senderID")
		return false
	} else if !isAuthorized {
		a.backend.FireSquadStreamErrorTrackingEvent(ctx, models.SquadStreamErrorTrackingEventInfo{
			ChannelID:       senderID,
			TargetChannelID: &recipientID,
			Method:          models.ErrorMethodTypeCreateInvitation,
			ErrorCode:       meepoerrors.ErrUserCannotAccessSquads,
		})
		return false
	}

	isAuthorized, err = a.backend.CanAccessSquads(ctx, recipientID)
	if err != nil {
		a.log.LogCtx(ctx, "err", err, "cannot determine squad access for recipientID")
		return false
	} else if !isAuthorized {
		a.backend.FireSquadStreamErrorTrackingEvent(ctx, models.SquadStreamErrorTrackingEventInfo{
			ChannelID:       senderID,
			TargetChannelID: &recipientID,
			Method:          models.ErrorMethodTypeCreateInvitation,
			ErrorCode:       meepoerrors.ErrUserCannotAccessSquads,
		})
		return false
	}

	if callerID == senderID {
		return true
	}

	// Editor of senderID can create invitations on behalf of senderID
	isAuthorized, err = a.utils.IsTwitchEditor(ctx, callerID, senderID)
	if err != nil {
		a.log.LogCtx(ctx, "err", err, "could not get twitch editor status")
	} else if isAuthorized {
		return true
	}

	// Twitch admins can only create invitations for senderID and recipientID with squad access
	isAuthorized, err = a.utils.IsTwitchAdmin(ctx, callerID)
	if err != nil {
		a.log.LogCtx(ctx, "err", err, "could not get twitch admin status")
	}

	return isAuthorized
}
