package auth_test

import (
	"context"
	"testing"

	"code.justin.tv/twitch-events/meepo/internal/util"

	"github.com/stretchr/testify/mock"

	"code.justin.tv/twitch-events/meepo/internal/mocks"

	. "github.com/smartystreets/goconvey/convey"
)

func TestCanCreateInvitation(t *testing.T) {
	Convey("CanCreateInvitation", t, func() {
		i := setUpInjectables()
		authHelpers := (i.Utils).(*mocks.Utils)
		backend := (i.Backend).(*mocks.Backender)

		auth, err := SetUpAuthorizer(t, i)
		if err != nil {
			t.Error("Unable to setup Auth mocks")
		}
		ctx := context.Background()

		backend.On("FireSquadStreamErrorTrackingEvent", mock.Anything, mock.Anything)

		Convey("No senderID should return false", func() {
			So(auth.CanCreateInvitation(ctx, "", util.NewUserID(), util.NewUserID()), ShouldBeFalse)
		})

		Convey("No recipientID should return false", func() {
			So(auth.CanCreateInvitation(ctx, util.NewUserID(), "", util.NewUserID()), ShouldBeFalse)
		})

		Convey("No callerID should return false", func() {
			So(auth.CanCreateInvitation(ctx, util.NewUserID(), util.NewUserID(), ""), ShouldBeFalse)
		})

		Convey("Admins as callerID should return true", func() {
			senderID := util.NewUserID()
			recipientID := util.NewUserID()
			adminUserID := util.NewUserID()

			backend.On("CanAccessSquads", mock.Anything, senderID).Return(false, nil)
			backend.On("CanAccessSquads", mock.Anything, recipientID).Return(false, nil)
			authHelpers.On("IsTwitchAdmin", mock.Anything, adminUserID).Return(false, nil)
			authHelpers.On("IsTwitchEditor", mock.Anything, adminUserID, senderID).Return(false, nil)

			authHelpers.On("IsAdmin", adminUserID).Return(true)
			So(auth.CanCreateInvitation(ctx, senderID, recipientID, adminUserID), ShouldBeTrue)
		})

		Convey("Twitch Admins as callerID", func() {
			twitchAdminID := util.NewUserID()
			senderID := util.NewUserID()
			recipientID := util.NewUserID()

			authHelpers.On("IsAdmin", twitchAdminID).Return(false)
			authHelpers.On("IsTwitchEditor", mock.Anything, twitchAdminID, senderID).Return(false, nil)
			authHelpers.On("IsTwitchAdmin", mock.Anything, twitchAdminID).Return(true, nil)

			Convey("Not authorized when senderID or recipientID cannot access squads", func() {
				backend.On("CanAccessSquads", mock.Anything, senderID).Return(false, nil)
				backend.On("CanAccessSquads", mock.Anything, recipientID).Return(true, nil)

				So(auth.CanCreateInvitation(ctx, senderID, recipientID, twitchAdminID), ShouldBeFalse)
			})

			Convey("Authorized when both sender and recipient has access to squads", func() {
				backend.On("CanAccessSquads", mock.Anything, senderID).Return(true, nil)
				backend.On("CanAccessSquads", mock.Anything, recipientID).Return(true, nil)

				So(auth.CanCreateInvitation(ctx, senderID, recipientID, twitchAdminID), ShouldBeTrue)
			})
		})

		Convey("Twitch Editors as callerID", func() {
			editorID := util.NewUserID()
			senderID := util.NewUserID()
			recipientID := util.NewUserID()

			authHelpers.On("IsAdmin", editorID).Return(false)
			authHelpers.On("IsTwitchAdmin", mock.Anything, editorID).Return(false, nil)

			Convey("Not authorized when senderID or recipientID cannot access squads", func() {
				backend.On("CanAccessSquads", mock.Anything, senderID).Return(false, nil)
				backend.On("CanAccessSquads", mock.Anything, recipientID).Return(true, nil)
				authHelpers.On("IsTwitchEditor", mock.Anything, editorID, senderID).Return(true, nil)

				So(auth.CanCreateInvitation(ctx, senderID, recipientID, editorID), ShouldBeFalse)
			})

			Convey("Authorized when both senderID and recipientID has access to squads", func() {
				backend.On("CanAccessSquads", mock.Anything, senderID).Return(true, nil)
				backend.On("CanAccessSquads", mock.Anything, recipientID).Return(true, nil)
				authHelpers.On("IsTwitchEditor", mock.Anything, editorID, senderID).Return(true, nil)

				So(auth.CanCreateInvitation(ctx, senderID, recipientID, editorID), ShouldBeTrue)
			})

			Convey("Authorized when editorID cannot access squads", func() {
				backend.On("CanAccessSquads", mock.Anything, senderID).Return(true, nil)
				backend.On("CanAccessSquads", mock.Anything, recipientID).Return(true, nil)
				backend.On("CanAccessSquads", mock.Anything, editorID).Return(false, nil)
				authHelpers.On("IsTwitchEditor", mock.Anything, editorID, senderID).Return(true, nil)

				So(auth.CanCreateInvitation(ctx, senderID, recipientID, editorID), ShouldBeTrue)
			})
		})

		Convey("Invitation Sender as callerID should return true", func() {
			senderID := util.NewUserID()
			recipientID := util.NewUserID()

			backend.On("CanAccessSquads", mock.Anything, senderID).Return(true, nil)
			backend.On("CanAccessSquads", mock.Anything, recipientID).Return(true, nil)
			authHelpers.On("IsTwitchEditor", mock.Anything, senderID, senderID).Return(false, nil)
			authHelpers.On("IsAdmin", senderID).Return(false)
			authHelpers.On("IsTwitchAdmin", mock.Anything, senderID).Return(false, nil)

			So(auth.CanCreateInvitation(ctx, senderID, recipientID, senderID), ShouldBeTrue)
		})

		Convey("Unauthorized user as callerID should return false", func() {
			senderID := util.NewUserID()
			recipientID := util.NewUserID()
			userID := util.NewUserID()

			backend.On("CanAccessSquads", mock.Anything, senderID).Return(false, nil)
			backend.On("CanAccessSquads", mock.Anything, recipientID).Return(true, nil)
			authHelpers.On("IsTwitchEditor", mock.Anything, userID, senderID).Return(false, nil)
			authHelpers.On("IsAdmin", userID).Return(false)
			authHelpers.On("IsTwitchAdmin", mock.Anything, userID).Return(false, nil)

			So(auth.CanCreateInvitation(ctx, senderID, recipientID, userID), ShouldBeFalse)
		})
	})
}
