package auth

import (
	"context"

	meepoerrors "code.justin.tv/twitch-events/meepo/errors"
	"code.justin.tv/twitch-events/meepo/internal/models"
	"github.com/twitchtv/twirp"
)

// CreateMembership wraps the corresponding backend function with an authorizer
func (a *auth) CreateMembership(ctx context.Context, targetUserID, squadID, callerID string) (*models.Member, error) {
	if !a.authorizer.CanCreateMembership(ctx, targetUserID, squadID, callerID) {
		a.backend.FireSquadStreamErrorTrackingEvent(ctx, models.SquadStreamErrorTrackingEventInfo{
			ChannelID:       callerID,
			TargetChannelID: &targetUserID,
			SquadID:         squadID,
			Method:          models.ErrorMethodTypeCreateMembership,
			ErrorCode:       meepoerrors.ErrUnauthorized,
		})
		return nil, twirp.NewError(twirp.PermissionDenied, "You cannot modify the squad").WithMeta(meepoerrors.ErrMetaKey, meepoerrors.ErrUnauthorized)
	}

	return a.backend.CreateMembership(ctx, targetUserID, squadID, callerID)
}

// CanCreateMembership determines whether a callerID is allowed to add a member to a squad
func (a *auth) CanCreateMembership(ctx context.Context, targetUserID, squadID, callerID string) bool {
	if targetUserID == "" || squadID == "" || callerID == "" {
		return false
	}

	if a.utils.IsAdmin(callerID) {
		return true
	}

	// targetUserID must have access to squads
	isAuthorized, err := a.backend.CanAccessSquads(ctx, targetUserID)
	if err != nil {
		a.log.LogCtx(ctx, "err", err, "cannot determine squad access for target member")
		return false
	} else if !isAuthorized {
		return false
	}

	squad, err := a.backend.GetSquadByID(ctx, squadID)
	if err != nil {
		a.log.LogCtx(ctx, "err", err, "could not get squad")
		return false
	}
	if squad == nil {
		a.log.LogCtx(ctx, "err", err, "squad does not exist")
		a.backend.FireSquadStreamErrorTrackingEvent(ctx, models.SquadStreamErrorTrackingEventInfo{
			ChannelID:       callerID,
			TargetChannelID: &targetUserID,
			SquadID:         squadID,
			Method:          models.ErrorMethodTypeCreateMembership,
			ErrorCode:       meepoerrors.ErrSquadNotFound,
		})
		return false
	}

	if squad.OwnerID == nil {
		return false
	}

	if *squad.OwnerID == callerID {
		return true
	}

	isAuthorized, err = a.utils.IsTwitchAdmin(ctx, callerID)
	if err != nil {
		a.log.LogCtx(ctx, "err", err, "could not get twitch admin status")
	}

	return isAuthorized
}
