package auth_test

import (
	"context"
	"testing"

	"github.com/pkg/errors"

	"code.justin.tv/twitch-events/meepo/internal/mocks"
	"code.justin.tv/twitch-events/meepo/internal/models"
	"code.justin.tv/twitch-events/meepo/internal/testutil"
	"code.justin.tv/twitch-events/meepo/internal/util"

	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

func TestCanCreateMembership(t *testing.T) {
	Convey("CanCreateMembership", t, func() {
		i := setUpInjectables()
		authHelpers := (i.Utils).(*mocks.Utils)
		backend := (i.Backend).(*mocks.Backender)

		auth, err := SetUpAuthorizer(t, i)
		if err != nil {
			t.Error("Unable to setup Auth mocks")
		}
		ctx := context.Background()

		squadID := testutil.MustNewID()
		ownerID := util.NewUserID()
		backend.On("FireSquadStreamErrorTrackingEvent", mock.Anything, mock.Anything)
		backend.On("GetSquadByID", mock.Anything, squadID).Return(&models.Squad{
			ID:      squadID,
			OwnerID: &ownerID,
		}, nil)

		Convey("No targetUserID should return false", func() {
			So(auth.CanCreateMembership(ctx, "", testutil.MustNewID(), util.NewUserID()), ShouldBeFalse)
		})

		Convey("No squadID should return false", func() {
			So(auth.CanCreateMembership(ctx, util.NewUserID(), "", util.NewUserID()), ShouldBeFalse)
		})

		Convey("No callerID should return false", func() {
			So(auth.CanCreateMembership(ctx, util.NewUserID(), util.NewUserID(), ""), ShouldBeFalse)
		})

		Convey("Invalid squadID should return false", func() {
			squadID := testutil.MustNewID()
			targetUserID := util.NewUserID()

			Convey("Non-existent squads should return false", func() {
				backend.On("GetSquadByID", mock.Anything, squadID).Return(nil, nil)
				backend.On("CanAccessSquads", mock.Anything, targetUserID).Return(true, nil)
				authHelpers.On("IsTwitchAdmin", mock.Anything, ownerID).Return(false, nil)
				authHelpers.On("IsAdmin", ownerID).Return(false)

				So(auth.CanCreateMembership(ctx, targetUserID, squadID, ownerID), ShouldBeFalse)
			})

			Convey("No ownerID should return false", func() {
				backend.On("GetSquadByID", mock.Anything, squadID).Return(&models.Squad{
					ID:      squadID,
					OwnerID: nil,
				}, nil)
				backend.On("CanAccessSquads", mock.Anything, targetUserID).Return(true, nil)
				authHelpers.On("IsTwitchAdmin", mock.Anything, ownerID).Return(false, nil)
				authHelpers.On("IsAdmin", ownerID).Return(false)

				So(auth.CanCreateMembership(ctx, targetUserID, squadID, ownerID), ShouldBeFalse)
			})

			Convey("Error getting squads should return false", func() {
				backend.On("GetSquadByID", mock.Anything, squadID).Return(nil, errors.New("test error"))
				backend.On("CanAccessSquads", mock.Anything, targetUserID).Return(true, nil)
				authHelpers.On("IsTwitchAdmin", mock.Anything, ownerID).Return(false, nil)
				authHelpers.On("IsAdmin", ownerID).Return(false)

				So(auth.CanCreateMembership(ctx, targetUserID, squadID, ownerID), ShouldBeFalse)
			})
		})

		Convey("Admins as callerID should return true", func() {
			adminUserID := util.NewUserID()
			targetUserID := util.NewUserID()
			backend.On("CanAccessSquads", mock.Anything, targetUserID).Return(false, nil)
			authHelpers.On("IsTwitchAdmin", mock.Anything, adminUserID).Return(false, nil)

			authHelpers.On("IsAdmin", adminUserID).Return(true)
			So(auth.CanCreateMembership(ctx, util.NewUserID(), squadID, adminUserID), ShouldBeTrue)
		})

		Convey("Twitch Admins as callerID", func() {
			twitchAdminID := util.NewUserID()
			targetUserID := util.NewUserID()

			Convey("Not authorized to create membership for users without squads access", func() {
				backend.On("CanAccessSquads", mock.Anything, targetUserID).Return(false, nil)
				authHelpers.On("IsTwitchAdmin", mock.Anything, twitchAdminID).Return(true, nil)
				authHelpers.On("IsAdmin", twitchAdminID).Return(false)

				So(auth.CanCreateMembership(ctx, targetUserID, squadID, twitchAdminID), ShouldBeFalse)
			})

			Convey("Authorized to create membership for squad without being owner", func() {
				backend.On("CanAccessSquads", mock.Anything, targetUserID).Return(true, nil)
				authHelpers.On("IsTwitchAdmin", mock.Anything, twitchAdminID).Return(true, nil)
				authHelpers.On("IsAdmin", twitchAdminID).Return(false)

				So(auth.CanCreateMembership(ctx, targetUserID, squadID, twitchAdminID), ShouldBeTrue)
			})
		})

		Convey("Squad Owners as callerID should return true", func() {
			targetUserID := util.NewUserID()
			backend.On("CanAccessSquads", mock.Anything, targetUserID).Return(true, nil)
			authHelpers.On("IsTwitchAdmin", mock.Anything, ownerID).Return(false, nil)
			authHelpers.On("IsAdmin", ownerID).Return(false)

			So(auth.CanCreateMembership(ctx, targetUserID, squadID, ownerID), ShouldBeTrue)
		})

		Convey("Unauthorized user as callerID should return false", func() {
			callerID := util.NewUserID()
			targetUserID := util.NewUserID()
			backend.On("CanAccessSquads", mock.Anything, targetUserID).Return(true, nil)
			authHelpers.On("IsTwitchAdmin", mock.Anything, callerID).Return(false, nil)
			authHelpers.On("IsAdmin", callerID).Return(false)

			So(auth.CanCreateMembership(ctx, targetUserID, squadID, callerID), ShouldBeFalse)
		})
	})
}
