package auth

import (
	"context"

	meepoerrors "code.justin.tv/twitch-events/meepo/errors"
	"code.justin.tv/twitch-events/meepo/internal/models"
	"github.com/twitchtv/twirp"
)

// CreateSquad wraps the corresponding backend function with an authorizer
func (a *auth) CreateSquad(ctx context.Context, ownerID, callerID string) (*models.Squad, error) {
	if !a.authorizer.CanCreateSquad(ctx, ownerID, callerID) {
		a.backend.FireSquadStreamErrorTrackingEvent(ctx, models.SquadStreamErrorTrackingEventInfo{
			ChannelID:       callerID,
			TargetChannelID: &ownerID,
			Method:          models.ErrorMethodTypeCreateSquad,
			ErrorCode:       meepoerrors.ErrUnauthorized,
		})
		return nil, twirp.NewError(twirp.PermissionDenied, "The caller cannot create a squad").WithMeta(meepoerrors.ErrMetaKey, meepoerrors.ErrUnauthorized)
	}

	return a.backend.CreateSquad(ctx, ownerID, callerID)
}

// CanCreateSquad determines whether the caller is authorized to create a squad.
func (a *auth) CanCreateSquad(ctx context.Context, ownerID, callerID string) bool {
	if callerID == "" || ownerID == "" {
		return false
	}

	if a.utils.IsAdmin(callerID) {
		return true
	}

	if ownerID != callerID {
		return false
	}

	isAuthorized, err := a.backend.CanAccessSquads(ctx, ownerID)
	if err != nil {
		a.log.LogCtx(ctx, "err", err, "cannot determine squad access")
		return false
	}

	return isAuthorized
}
