package auth_test

import (
	"context"
	"testing"

	"code.justin.tv/twitch-events/meepo/internal/mocks"
	"code.justin.tv/twitch-events/meepo/internal/util"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

func TestCanCreateSquad(t *testing.T) {
	Convey("CanCreateSquad", t, func() {
		i := setUpInjectables()
		authHelpers := (i.Utils).(*mocks.Utils)
		backend := (i.Backend).(*mocks.Backender)

		auth, err := SetUpAuthorizer(t, i)
		if err != nil {
			t.Error("Unable to setup Auth mocks")
		}
		ctx := context.Background()

		Convey("No ownerID should return false", func() {
			So(auth.CanCreateSquad(ctx, "", util.NewUserID()), ShouldBeFalse)
		})

		Convey("No callerID should return false", func() {
			So(auth.CanCreateSquad(ctx, util.NewUserID(), ""), ShouldBeFalse)
		})

		Convey("Admins as callerID should return true", func() {
			ownerID := util.NewUserID()
			adminUserID := util.NewUserID()
			// Can create squad for owners without squad access
			backend.On("CanAccessSquads", mock.Anything, ownerID).Return(false, nil)

			authHelpers.On("IsAdmin", adminUserID).Return(true)
			So(auth.CanCreateSquad(ctx, ownerID, adminUserID), ShouldBeTrue)
		})

		Convey("Owner as callerID should return true", func() {
			ownerID := util.NewUserID()
			authHelpers.On("IsAdmin", ownerID).Return(false)

			Convey("Authorized to create squad", func() {
				backend.On("CanAccessSquads", mock.Anything, ownerID).Return(true, nil)
				So(auth.CanCreateSquad(ctx, ownerID, ownerID), ShouldBeTrue)
			})

			Convey("Not Authorized to create squad without squad stream access", func() {
				backend.On("CanAccessSquads", mock.Anything, ownerID).Return(false, nil)
				So(auth.CanCreateSquad(ctx, ownerID, ownerID), ShouldBeFalse)
			})
		})

		Convey("Unauthorized Users should return false", func() {
			ownerID := util.NewUserID()
			callerID := util.NewUserID()
			authHelpers.On("IsAdmin", callerID).Return(false)
			backend.On("CanAccessSquads", mock.Anything, ownerID).Return(true, nil)

			So(auth.CanCreateSquad(ctx, ownerID, callerID), ShouldBeFalse)
		})
	})
}
