package auth

import (
	"context"

	meepoerrors "code.justin.tv/twitch-events/meepo/errors"
	"code.justin.tv/twitch-events/meepo/internal/models"
	"github.com/twitchtv/twirp"
)

// DeleteInvitation wraps the corresponding backend function with an authorizer
func (a *auth) DeleteInvitation(ctx context.Context, invitationID string, callerID string) (*models.Invitation, error) {
	if !a.authorizer.CanDeleteInvitation(ctx, invitationID, callerID) {
		a.backend.FireSquadStreamErrorTrackingEvent(ctx, models.SquadStreamErrorTrackingEventInfo{
			ChannelID: callerID,
			InviteID:  invitationID,
			Method:    models.ErrorMethodTypeDeleteInvitation,
			ErrorCode: meepoerrors.ErrUnauthorized,
		})
		return nil, twirp.NewError(twirp.PermissionDenied, "The caller cannot delete the invitation").WithMeta(meepoerrors.ErrMetaKey, meepoerrors.ErrUnauthorized)
	}

	return a.backend.DeleteInvitation(ctx, invitationID, callerID)
}

// CanDeleteInvitation determines whether the caller is authorized to delete the invitation.
func (a *auth) CanDeleteInvitation(ctx context.Context, invitationID, callerID string) bool {
	if callerID == "" || invitationID == "" {
		return false
	}

	if a.utils.IsAdmin(callerID) {
		return true
	}

	dbInvitation, err := a.backend.GetInvitationByID(ctx, invitationID)
	if err != nil || dbInvitation == nil {
		return false
	}

	if a.utils.IsInvitationSender(callerID, dbInvitation) {
		return true
	}

	isAuthorized, err := a.utils.IsTwitchEditor(ctx, callerID, dbInvitation.SenderID)
	if err != nil {
		a.log.LogCtx(ctx, "err", err, "could not get twitch editor status")
	} else if isAuthorized {
		return true
	}

	isAuthorized, err = a.utils.IsTwitchAdmin(ctx, callerID)
	if err != nil {
		a.log.LogCtx(ctx, "err", err, "could not get twitch admin status")
	}

	return isAuthorized
}
