package auth

import (
	"context"

	meepoerrors "code.justin.tv/twitch-events/meepo/errors"
	"code.justin.tv/twitch-events/meepo/internal/models"
	"github.com/twitchtv/twirp"
)

// GetInvitationsBySquadID wraps the corresponding backend function with an authorizer
func (a *auth) GetInvitationsBySquadID(ctx context.Context, squadID string, status models.InvitationStatus, callerID string) ([]*models.Invitation, error) {
	if !a.authorizer.CanGetInvitationsBySquadID(ctx, squadID, callerID) {
		a.backend.FireSquadStreamErrorTrackingEvent(ctx, models.SquadStreamErrorTrackingEventInfo{
			ChannelID: callerID,
			SquadID:   squadID,
			Method:    models.ErrorMethodTypeGetInvitationsBySquadID,
			ErrorCode: meepoerrors.ErrUnauthorized,
		})
		return nil, twirp.NewError(twirp.PermissionDenied, "The caller cannot get invitations for the squad.").WithMeta(meepoerrors.ErrMetaKey, meepoerrors.ErrUnauthorized)
	}

	return a.backend.GetInvitationsBySquadID(ctx, squadID, status, callerID)
}

// GetInvitationsBySquadID determines whether the caller is authorized to get the invitations for a squad.
func (a *auth) CanGetInvitationsBySquadID(ctx context.Context, squadID, callerID string) bool {
	if squadID == "" || callerID == "" {
		return false
	}

	if a.utils.IsAdmin(callerID) {
		return true
	}

	squad, err := a.backend.GetSquadByID(ctx, squadID)
	if err != nil || squad == nil {
		return false
	}

	if a.utils.IsSquadMember(squad, callerID) {
		return true
	}

	if a.utils.IsTwitchEditorForSquadMember(ctx, squad, callerID) {
		return true
	}

	isAuthorized, err := a.utils.IsTwitchAdmin(ctx, callerID)
	if err != nil {
		a.log.LogCtx(ctx, "err", err, "could not get twitch admin status")
	}

	return isAuthorized
}
