package auth_test

import (
	"context"
	"testing"

	"github.com/pkg/errors"

	"code.justin.tv/twitch-events/meepo/internal/mocks"
	"code.justin.tv/twitch-events/meepo/internal/models"
	"code.justin.tv/twitch-events/meepo/internal/testutil"
	"code.justin.tv/twitch-events/meepo/internal/util"

	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

func TestCanGetInvitationsBySquadID(t *testing.T) {
	Convey("CanGetInvitationsBySquadID", t, func() {
		i := setUpInjectables()
		authHelpers := (i.Utils).(*mocks.Utils)
		backend := (i.Backend).(*mocks.Backender)

		auth, err := SetUpAuthorizer(t, i)
		if err != nil {
			t.Error("Unable to setup Auth mocks")
		}
		ctx := context.Background()

		squadID := testutil.MustNewID()
		memberID := util.NewUserID()
		squad := &models.Squad{
			ID:        squadID,
			MemberIDs: []string{memberID, util.NewUserID()},
		}
		backend.On("GetSquadByID", mock.Anything, squadID).Return(squad, nil)

		Convey("No squadID should return false", func() {
			So(auth.CanGetInvitationsBySquadID(ctx, "", util.NewUserID()), ShouldBeFalse)
		})

		Convey("No callerID should return false", func() {
			So(auth.CanGetInvitationsBySquadID(ctx, testutil.MustNewID(), ""), ShouldBeFalse)
		})

		Convey("Invalid squadID should return false", func() {
			squadID := testutil.MustNewID()
			callerID := util.NewUserID()

			Convey("Non-existent squads should return false", func() {
				backend.On("GetSquadByID", mock.Anything, squadID).Return(nil, nil)
				authHelpers.On("IsAdmin", callerID).Return(false)

				So(auth.CanGetInvitationsBySquadID(ctx, squadID, callerID), ShouldBeFalse)
			})

			Convey("Error getting squads should return false", func() {
				backend.On("GetSquadByID", mock.Anything, squadID).Return(nil, errors.New("test error"))
				authHelpers.On("IsAdmin", callerID).Return(false)

				So(auth.CanGetInvitationsBySquadID(ctx, squadID, callerID), ShouldBeFalse)
			})
		})

		Convey("Admins as callerID should return true", func() {
			adminUserID := util.NewUserID()
			authHelpers.On("IsSquadMember", &squad, adminUserID).Return(false, nil)
			authHelpers.On("IsTwitchEditorForSquadMember", mock.Anything, squad, adminUserID).Return(false, nil)
			authHelpers.On("IsTwitchAdmin", mock.Anything, adminUserID).Return(false, nil)

			authHelpers.On("IsAdmin", adminUserID).Return(true)
			So(auth.CanGetInvitationsBySquadID(ctx, squadID, adminUserID), ShouldBeTrue)
		})

		Convey("Twitch Admins as callerID should return true", func() {
			twitchAdminID := util.NewUserID()
			authHelpers.On("IsSquadMember", squad, twitchAdminID).Return(false, nil)
			authHelpers.On("IsTwitchEditorForSquadMember", mock.Anything, squad, twitchAdminID).Return(false, nil)
			authHelpers.On("IsAdmin", twitchAdminID).Return(false)

			authHelpers.On("IsTwitchAdmin", mock.Anything, twitchAdminID).Return(true, nil)
			So(auth.CanGetInvitationsBySquadID(ctx, squadID, twitchAdminID), ShouldBeTrue)
		})

		Convey("Squad Members as callerID should return true", func() {
			authHelpers.On("IsTwitchAdmin", mock.Anything, memberID).Return(false, nil)
			authHelpers.On("IsAdmin", memberID).Return(false)
			authHelpers.On("IsTwitchEditorForSquadMember", mock.Anything, squad, memberID).Return(false, nil)

			authHelpers.On("IsSquadMember", squad, memberID).Return(true, nil)
			So(auth.CanGetInvitationsBySquadID(ctx, squadID, memberID), ShouldBeTrue)
		})

		Convey("Twitch Editors of Squad Members as callerID should return true", func() {
			editorID := util.NewUserID()
			authHelpers.On("IsTwitchAdmin", mock.Anything, editorID).Return(false, nil)
			authHelpers.On("IsAdmin", editorID).Return(false)
			authHelpers.On("IsSquadMember", squad, editorID).Return(false, nil)

			authHelpers.On("IsTwitchEditorForSquadMember", mock.Anything, squad, editorID).Return(true, nil)
			So(auth.CanGetInvitationsBySquadID(ctx, squadID, editorID), ShouldBeTrue)
		})

		Convey("Unauthorized user as callerID should return false", func() {
			callerID := util.NewUserID()
			authHelpers.On("IsTwitchAdmin", mock.Anything, callerID).Return(false, nil)
			authHelpers.On("IsAdmin", callerID).Return(false)
			authHelpers.On("IsSquadMember", squad, callerID).Return(false, nil)
			authHelpers.On("IsTwitchEditorForSquadMember", mock.Anything, squad, callerID).Return(false, nil)

			So(auth.CanGetInvitationsBySquadID(ctx, squadID, callerID), ShouldBeFalse)
		})
	})
}
