package auth

import (
	"context"

	meepoerrors "code.justin.tv/twitch-events/meepo/errors"
	"code.justin.tv/twitch-events/meepo/internal/models"
	"github.com/twitchtv/twirp"
)

// GetPendingInvitationsByRecipientID wraps the corresponding backend function with an authorizer
func (a *auth) GetPendingInvitationsByRecipientID(ctx context.Context, channelID, callerID string) ([]*models.Invitation, error) {
	if !a.authorizer.CanGetPendingInvitationsByRecipientID(ctx, channelID, callerID) {
		a.backend.FireSquadStreamErrorTrackingEvent(ctx, models.SquadStreamErrorTrackingEventInfo{
			ChannelID:       callerID,
			TargetChannelID: &channelID,
			Method:          models.ErrorMethodTypeGetPendingInvitationsByRecipientID,
			ErrorCode:       meepoerrors.ErrUnauthorized,
		})
		return nil, twirp.NewError(twirp.PermissionDenied, "The caller cannot get invitation for the channel").WithMeta(meepoerrors.ErrMetaKey, meepoerrors.ErrUnauthorized)
	}

	return a.backend.GetPendingInvitationsByRecipientID(ctx, channelID)
}

// CanGetPendingInvitationsByRecipientID determines whether the caller is authorized to get pending invitation.
func (a *auth) CanGetPendingInvitationsByRecipientID(ctx context.Context, channelID, callerID string) bool {
	if callerID == "" || channelID == "" {
		return false
	}

	if a.utils.IsAdmin(callerID) {
		return true
	}

	if channelID == callerID {
		return true
	}

	isAuthorized, err := a.utils.IsTwitchEditor(ctx, callerID, channelID)
	if err != nil {
		a.log.LogCtx(ctx, "err", err, "could not get twitch editor status")
	} else if isAuthorized {
		return true
	}

	isAuthorized, err = a.utils.IsTwitchAdmin(ctx, callerID)
	if err != nil {
		a.log.LogCtx(ctx, "err", err, "could not get twitch admin status")
	}

	return isAuthorized
}
