package auth_test

import (
	"context"
	"testing"

	"code.justin.tv/twitch-events/meepo/internal/mocks"
	"code.justin.tv/twitch-events/meepo/internal/util"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

func TestCanGetPendingInvitationsByRecipientID(t *testing.T) {
	Convey("CanGetPendingInvitationsByRecipientID", t, func() {
		i := setUpInjectables()
		authHelpers := (i.Utils).(*mocks.Utils)

		auth, err := SetUpAuthorizer(t, i)
		if err != nil {
			t.Error("Unable to setup Auth mocks")
		}
		ctx := context.Background()

		Convey("No channelID should return false", func() {
			So(auth.CanGetPendingInvitationsByRecipientID(ctx, "", util.NewUserID()), ShouldBeFalse)
		})

		Convey("No callerID should return false", func() {
			So(auth.CanGetPendingInvitationsByRecipientID(ctx, util.NewUserID(), ""), ShouldBeFalse)
		})

		Convey("Admins as callerID should return true", func() {
			channelID := util.NewUserID()
			adminUserID := util.NewUserID()

			authHelpers.On("IsTwitchAdmin", mock.Anything, adminUserID).Return(false, nil)
			authHelpers.On("IsTwitchEditor", mock.Anything, adminUserID, channelID).Return(false, nil)

			authHelpers.On("IsAdmin", adminUserID).Return(true)
			So(auth.CanGetPendingInvitationsByRecipientID(ctx, channelID, adminUserID), ShouldBeTrue)
		})

		Convey("Twitch Admins as callerID should return true", func() {
			channelID := util.NewUserID()
			twitchAdminID := util.NewUserID()

			authHelpers.On("IsAdmin", twitchAdminID).Return(false)
			authHelpers.On("IsTwitchEditor", mock.Anything, twitchAdminID, channelID).Return(false, nil)

			authHelpers.On("IsTwitchAdmin", mock.Anything, twitchAdminID).Return(true, nil)
			So(auth.CanGetPendingInvitationsByRecipientID(ctx, channelID, twitchAdminID), ShouldBeTrue)
		})

		Convey("Editors for invitation recipient as callerID should return true", func() {
			channelID := util.NewUserID()
			editorID := util.NewUserID()

			authHelpers.On("IsAdmin", editorID).Return(false)
			authHelpers.On("IsTwitchAdmin", mock.Anything, editorID).Return(false, nil)

			authHelpers.On("IsTwitchEditor", mock.Anything, editorID, channelID).Return(true, nil)
			So(auth.CanGetPendingInvitationsByRecipientID(ctx, channelID, editorID), ShouldBeTrue)
		})

		Convey("Invitation Recipient as callerID should return true", func() {
			channelID := util.NewUserID()

			authHelpers.On("IsAdmin", channelID).Return(false)
			authHelpers.On("IsTwitchAdmin", mock.Anything, channelID).Return(false, nil)
			authHelpers.On("IsTwitchEditor", mock.Anything, channelID, channelID).Return(false, nil)

			So(auth.CanGetPendingInvitationsByRecipientID(ctx, channelID, channelID), ShouldBeTrue)
		})

		Convey("Unauthorized Users should return false", func() {
			channelID := util.NewUserID()
			userID := util.NewUserID()

			authHelpers.On("IsAdmin", userID).Return(false)
			authHelpers.On("IsTwitchAdmin", mock.Anything, userID).Return(false, nil)
			authHelpers.On("IsTwitchEditor", mock.Anything, userID, channelID).Return(false, nil)

			So(auth.CanGetPendingInvitationsByRecipientID(ctx, channelID, userID), ShouldBeFalse)
		})
	})
}
