package auth

import (
	"context"

	meepoerrors "code.justin.tv/twitch-events/meepo/errors"
	"code.justin.tv/twitch-events/meepo/internal/models"
	"github.com/twitchtv/twirp"
)

// GetPendingInvitationsCountByRecipientID wraps the corresponding backend function with an authorizer
func (a *auth) GetPendingInvitationsCountByRecipientID(ctx context.Context, channelID string, callerID string) (int64, error) {
	if !a.authorizer.CanGetPendingInvitationsCountByRecipientID(ctx, channelID, callerID) {
		a.backend.FireSquadStreamErrorTrackingEvent(ctx, models.SquadStreamErrorTrackingEventInfo{
			ChannelID:       callerID,
			TargetChannelID: &channelID,
			Method:          models.ErrorMethodTypeGetPendingInvitationsCountByRecipientID,
			ErrorCode:       meepoerrors.ErrUnauthorized,
		})
		return 0, twirp.NewError(twirp.PermissionDenied, "The caller cannot get the invitation count.").WithMeta(meepoerrors.ErrMetaKey, meepoerrors.ErrUnauthorized)
	}

	return a.backend.GetPendingInvitationsCountByRecipientID(ctx, channelID, callerID)
}

// CanGetPendingInvitationsCountByRecipientID determines whether the caller is authorized to get pending invitation count.
func (a *auth) CanGetPendingInvitationsCountByRecipientID(ctx context.Context, channelID, callerID string) bool {
	if callerID == "" || channelID == "" {
		return false
	}

	if a.utils.IsAdmin(callerID) {
		return true
	}

	if channelID == callerID {
		return true
	}

	isAuthorized, err := a.utils.IsTwitchEditor(ctx, callerID, channelID)
	if err != nil {
		a.log.LogCtx(ctx, "err", err, "could not get twitch editor status")
	} else if isAuthorized {
		return true
	}

	isAuthorized, err = a.utils.IsTwitchAdmin(ctx, callerID)
	if err != nil {
		a.log.LogCtx(ctx, "err", err, "could not get twitch admin status")
	}

	return isAuthorized
}
