package auth

import (
	"context"

	meepoerrors "code.justin.tv/twitch-events/meepo/errors"
	"code.justin.tv/twitch-events/meepo/internal/models"
	"github.com/twitchtv/twirp"
)

// LeaveSquad determines whether a userID is allowed to leave a squad with ID, squadID
func (a *auth) LeaveSquad(ctx context.Context, memberID, squadID, callerID string) (*models.Squad, error) {
	if !a.authorizer.CanLeaveSquad(ctx, memberID, squadID, callerID) {
		a.backend.FireSquadStreamErrorTrackingEvent(ctx, models.SquadStreamErrorTrackingEventInfo{
			ChannelID:       callerID,
			TargetChannelID: &memberID,
			SquadID:         squadID,
			Method:          models.ErrorMethodTypeLeaveSquad,
			ErrorCode:       meepoerrors.ErrUnauthorized,
		})
		return nil, twirp.NewError(twirp.PermissionDenied, "The caller cannot leave the squad").WithMeta(meepoerrors.ErrMetaKey, meepoerrors.ErrUnauthorized)
	}

	return a.backend.LeaveSquad(ctx, memberID, squadID, callerID, false)
}

func (a *auth) CanLeaveSquad(ctx context.Context, memberID, squadID, callerID string) bool {
	if callerID == "" || squadID == "" || memberID == "" {
		return false
	}

	if memberID == callerID {
		return true
	}

	isAuthorized, err := a.utils.IsTwitchEditor(ctx, callerID, memberID)
	if err != nil {
		a.log.LogCtx(ctx, "err", err, "could not get twitch editor status")
	} else if isAuthorized {
		return true
	}

	if a.utils.IsAdmin(callerID) {
		return true
	}

	isAuthorized, err = a.utils.IsTwitchAdmin(ctx, callerID)
	if err != nil {
		a.log.LogCtx(ctx, "err", err, "could not get twitch admin status")
	} else if isAuthorized {
		return true
	}

	return false
}
