package auth_test

import (
	"context"
	"testing"

	"code.justin.tv/twitch-events/meepo/internal/mocks"
	"code.justin.tv/twitch-events/meepo/internal/testutil"
	"code.justin.tv/twitch-events/meepo/internal/util"

	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

func TestCanLeaveSquad(t *testing.T) {

	adminUserID := "123456789"

	Convey("CanLeaveSquad", t, func() {
		i := setUpInjectables()
		authHelpers := (i.Utils).(*mocks.Utils)

		auth, err := SetUpAuthorizer(t, i)
		if err != nil {
			t.Error("Unable to setup Auth mocks")
		}
		ctx := context.Background()

		squadID := testutil.MustNewID()
		memberID := util.NewUserID()

		Convey("No callerID should return false", func() {
			So(auth.CanLeaveSquad(ctx, memberID, squadID, ""), ShouldBeFalse)
		})

		Convey("No memberID should return false", func() {
			So(auth.CanLeaveSquad(ctx, "", squadID, util.NewUserID()), ShouldBeFalse)
		})

		Convey("No squadID should return false", func() {
			So(auth.CanLeaveSquad(ctx, memberID, "", util.NewUserID()), ShouldBeFalse)
		})

		Convey("Own user as callerID should return true", func() {
			// Should authorize callerID independent of the following checks
			authHelpers.On("IsTwitchAdmin", mock.Anything, mock.Anything).Return(false, nil)
			authHelpers.On("IsTwitchEditor", mock.Anything, mock.Anything, mock.Anything).Return(false, nil)
			authHelpers.On("IsAdmin", adminUserID).Return(false)

			So(auth.CanLeaveSquad(ctx, memberID, squadID, memberID), ShouldBeTrue)
		})

		Convey("Admins as callerID should return true", func() {
			// Should authorize callerID independent of the following checks
			authHelpers.On("IsTwitchAdmin", mock.Anything, mock.Anything).Return(false, nil)
			authHelpers.On("IsTwitchEditor", mock.Anything, mock.Anything, mock.Anything).Return(false, nil)

			authHelpers.On("IsAdmin", adminUserID).Return(true)
			So(auth.CanLeaveSquad(ctx, memberID, squadID, adminUserID), ShouldBeTrue)
		})

		Convey("Twitch Admins as callerID should return true", func() {
			twitchAdminID := util.NewUserID()
			// Should authorize callerID independent of the following checks
			authHelpers.On("IsAdmin", mock.Anything).Return(false)
			authHelpers.On("IsTwitchEditor", mock.Anything, mock.Anything, mock.Anything).Return(false, nil)

			authHelpers.On("IsTwitchAdmin", mock.Anything, twitchAdminID).Return(true, nil)
			So(auth.CanLeaveSquad(ctx, memberID, squadID, twitchAdminID), ShouldBeTrue)
		})

		Convey("Editors for Leaving Member as callerID should return true", func() {
			editorID := util.NewUserID()
			// Should authorize callerID independent of the following checks
			authHelpers.On("IsAdmin", mock.Anything).Return(false)
			authHelpers.On("IsTwitchAdmin", mock.Anything, mock.Anything).Return(false, nil)

			authHelpers.On("IsTwitchEditor", mock.Anything, editorID, mock.Anything).Return(true, nil)
			So(auth.CanLeaveSquad(ctx, memberID, squadID, editorID), ShouldBeTrue)
		})

		Convey("Unauthorized Users should return false", func() {
			authHelpers.On("IsAdmin", mock.Anything).Return(false)
			authHelpers.On("IsTwitchAdmin", mock.Anything, mock.Anything).Return(false, nil)
			authHelpers.On("IsTwitchEditor", mock.Anything, mock.Anything, mock.Anything).Return(false, nil)

			So(auth.CanLeaveSquad(ctx, memberID, squadID, util.NewUserID()), ShouldBeFalse)
		})
	})
}
